/*
 * mod_auth_shadow.c
 *
 * An apache module to authenticate using the /etc/shadow file.
 * This module interacts with another program "validate", which
 * is setuid root.  Thus the /etc/shadow file can remain 
 * root:root 0400.
 *
 * Author: Brian Duggan <bduggan@matatu.org>
 * Some code was taken from the sample code supplied with
 * _Apache Modules_ by Stein and MacEachern.  Parts of this
 * were also influenced by mod_auth.c.
 *
 * Adapted for Apache2: Bernard du Breuil 
 *	<bernard.l.dubreuil@erdc.usace.army.mil>
 * I went back to mod_auth.c to see how it was converted.
 */

#include "apr_strings.h"
#include "ap_config.h"

#include "httpd.h"  
#include "http_config.h"  
#include "http_core.h"  
#include "http_log.h"  
#include "http_protocol.h"  
#include "http_request.h"  

#include <shadow.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>
#include <pwd.h>
#include <grp.h>
#include "validate.h"

#ifndef INSTBINDIR
#error INSTBINDIR should be defined as the location of the validate executable
<><><><><><>Crash and burn<><><><><>
#endif

/* do not accept empty "" strings */
#define strtrue(s) (s && *s)

/* 
 * configure like so:
 * 
 * LoadModule auth_shadow_module modules/mod_auth_shadow.so
 * <Location /test>
 * AuthType Basic 
 * AuthName WhateverAuthnameYouWant
 * AuthShadow on
 * require valid-user 
 * </Location>
 */

typedef struct {
    int auth_shadow_flag; /* 1 for yes, 0 for no */
} auth_shadow_config_rec;

static void *create_auth_shadow_dir_config(apr_pool_t *p, char *d)
{
    auth_shadow_config_rec *sec =
    (auth_shadow_config_rec *) apr_palloc(p, sizeof(*sec));
    sec->auth_shadow_flag = 0;	
    return sec;
}

static const command_rec auth_shadow_cmds[] =
{
    AP_INIT_FLAG("AuthShadow", ap_set_flag_slot,
	(void *)APR_OFFSETOF(auth_shadow_config_rec, auth_shadow_flag),
     OR_AUTHCFG, "On or Off depending on whether to use /etc/shadow"),
    {NULL}
};

static const char module_name[] = "auth_shadow_module";

static void my_register_hooks();

module AP_MODULE_DECLARE_DATA auth_shadow_module = 
{
    STANDARD20_MODULE_STUFF, 
    create_auth_shadow_dir_config , /* dir config creator */
    NULL,                  /* merge  per-dir    config structures */
    NULL,                  /* create per-server config structures */
    NULL,                  /* merge  per-server config structures */
    auth_shadow_cmds,      /* [config file] command table */
    my_register_hooks	   /* register hooks */
};

// Authentication function
static int auth_shadow_handler(request_rec *r);

// Authorization function
static int auth_shadow_valid_user(request_rec *r);

/* Apache 2 hooks for functions other than the handler itself. */
static void my_register_hooks()
{
	/* We want to run before mod_auth runs. */
	static const char *aszpost[] = {"mod_auth.c", NULL};
	/* Authentication function */
	ap_hook_check_user_id(auth_shadow_handler,NULL,aszpost,APR_HOOK_MIDDLE);
	/* Authorization function */
	ap_hook_auth_checker(auth_shadow_valid_user, NULL, aszpost, APR_HOOK_MIDDLE);
}

/* Internal functions */
static int user_in_group (char *user, const char *groupname);
static int auth_shadow_valid_user(request_rec *r);
static int auth_shadow_authorize(const char *user, const char* pw, 
		request_rec* r);

/*
 * auth_shadow_authorize
 *
 * See if a username/pw combination is valid.
 *
 * Returns 1 if the pw is correct, 0 if it's incorrect, -1 if there's an error.
 */

static int auth_shadow_authorize(const char *user, const char* pw, 
		request_rec* r)
{
    int filedes[2];  /* fd's for pipe.  Read from 0, write to 1*/
    char validate_prog[255];
    int ret, status;
    FILE* fp;

    if (strlen(INSTBINDIR) > 240) {
            ap_log_error(APLOG_MARK, APLOG_EMERG, APR_EBADPATH,
	    r->server,
            "%s: Error -- directory name too long.",module_name);
            return(-1);
    }

    if (pipe(filedes)!=0) {
        ap_log_error(APLOG_MARK, APLOG_EMERG, errno, r->server,
        "%s: Unable to open pipe.  Error: %d",module_name, errno);
        return(-1);
    }

    ret = fork();
    if (ret==-1) {
        ap_log_error(APLOG_MARK, APLOG_EMERG, errno, r->server,
        "%s: Unable to fork. Error: %d",module_name, errno);
        return(-1);
    }

    if (ret==0) { /* Child */

        /* Set stdin to filedes[0] */
        dup2(filedes[0],STDIN_FILENO);

        /* ...and close the other file descriptor. */
        if (close(filedes[0])!=0) {
            ap_log_error(APLOG_MARK, APLOG_EMERG, errno, r->server,
               "%s: Unable to close file descriptor. Error: %d",
		module_name, errno);
            exit(1);
        }

        sprintf(validate_prog,"%s/validate",INSTBINDIR);
        execl(validate_prog,
            validate_prog,
            NULL);

        /* We shouldn't reach this point */
        ap_log_error(APLOG_MARK, APLOG_EMERG, errno, r->server,
        	"%s: Unable to exec. Error: %d",module_name, errno);
        exit(1);
    }

    /* Parent */

    /* We write to the pipe, then wait for the child to finish. */
    fp = fdopen(filedes[1],"w");
    if (!fp) {
        ap_log_error(APLOG_MARK, APLOG_EMERG, errno, r->server,
        "%s: Unable to open pipe for writing: %d",module_name, errno);
        return(-1);
    }

    fprintf(fp, "%s\n",user);
    fprintf(fp, "%s\n",pw);
    fclose(fp);
    if (close(filedes[0])!=0) {
        ap_log_error(APLOG_MARK, APLOG_EMERG, errno, r->server,
        "%s: Unable to close file descriptor. Error: %d",module_name, errno);
        return(-1);
    }

    ret = wait(&status);
    if (ret==0 || ret==-1) {
        ap_log_error(APLOG_MARK, APLOG_EMERG, errno, r->server,
        "%s: Error while waiting for child: %d.",module_name, errno);
        return(-1);
    }

    if (status==0)
        return 1;  /* Correct pw */
    return 0; /* Nope */
}

/*
 * auth_shadow_handler
 *
 * See if the username/pw combination is valid.
 */

static int auth_shadow_handler(request_rec *r)
{
     int ret;
     const char *sent_pw; 
     char str[200];
     int rc = ap_get_basic_auth_pw(r, &sent_pw); 
     char user[MAX_USERNAME_LENGTH+1];
     char passwd[MAX_PW_LENGTH+1];
     int n;
     auth_shadow_config_rec *s =
        (auth_shadow_config_rec *) ap_get_module_config(r->per_dir_config, &auth_shadow_module);           

     if(rc != OK) return rc;
     if (s->auth_shadow_flag != 1)
        { return DECLINED; }            

     if(!(strtrue(r->user) && strtrue(sent_pw))) {
         ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, 
		"Both a username and password must be provided for %s", 
                   r->uri);  
         ap_note_basic_auth_failure(r);  
         return HTTP_UNAUTHORIZED;
     }

     /* Truncate to max length. */
     n = strlen(r->user);
     if (n > MAX_USERNAME_LENGTH) n=MAX_USERNAME_LENGTH;
     strncpy(user,r->user,n); /* Copy to user[0..n-1] */
     user[n] = '\0';

     n = strlen(sent_pw);
     if (n > MAX_PW_LENGTH) n=MAX_PW_LENGTH;
     strncpy(passwd,sent_pw,n); /* Copy to passwd[0..n-1] */
     passwd[n] = '\0';

     ret = auth_shadow_authorize(user,passwd,r);
     if (ret==-1)
         return HTTP_INTERNAL_SERVER_ERROR;
     if (ret!=1) {
         ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, 
		"Invalid password entered for user %s", user);  
         ap_note_basic_auth_failure(r);  
         return HTTP_UNAUTHORIZED;
     }

     return OK;
}

/*
 * user_in_group
 *
 * See whether a given user is a member of a given group
 * (True if either the user is in the group in /etc/group
 * or if the user's primary group is this group.)
 *
 * returns: 1 for yes, 0 for no
 */
static int user_in_group (char *user, const char *groupname) {
    int ret; // Return value from posix func's 0 is good; non-zero is errno 
	     // However, errno will be zero if the entry is not in the file.
    struct group grp;  /* Group structure */
    struct group *grpp = NULL;  /* Group structure */
    // This buffer should probably be done with pool storage.
#   define GBUFSIZ (9*500)
    char gbuffer[GBUFSIZ]; /* Lists all members of the group */
    // This buffer should probably be done with pool storage.
    struct passwd pw; /* Passwd structure */
    struct passwd *pwp = NULL; /* Passwd entry structure */
#   define PBUFSIZ (512)
    char pbuffer[PBUFSIZ];
    char **m;         /* List of members */
    int user_gid;     /* The user's primary group id. */

    if (!strtrue(groupname)) {
        ap_log_error(APLOG_MARK, APLOG_ERR, 0, NULL,
        	"%s: Blank group specification.", module_name);
        return 0;
    }

    ap_log_error(APLOG_MARK, APLOG_DEBUG, 0, NULL,
       "%s: Looking in group file for %d: got user: %s", module_name, 
		groupname, user);

    // Get group information from group file using re-entrant
    // version of getgrnam().
    ret = getgrnam_r(groupname, &grp, gbuffer, GBUFSIZ, &grpp);

    if (ret != 0 || !grpp)
    {
        ap_log_error(APLOG_MARK, APLOG_ERR, ret, NULL,
    	    "%s: Specified group %s not found.", module_name, groupname);
        return 0; /* No such group */
     }

    /* Check the list of users on the group entry. */
    m = grpp->gr_mem;
    if (!m)  
    {   /* Error */
        ap_log_error(APLOG_MARK, APLOG_ERR, ret, NULL,
        	"%s: Error reading user list for group %s",
		module_name,groupname);
        return 0;
    }

    // Scan the list of users for this group.
    while (*m) {
        if (!strcmp(*m,user))
            return 1;
        m++;
    }

    // If debugging output user and group that we are looking for.
    ap_log_error(APLOG_MARK, APLOG_DEBUG, ret, NULL,
    	"user: %s, grp: %s", user, groupname);

    // Get password entry using re-entrant version of getpwnam().
    ret = getpwnam_r(user, &pw, pbuffer, PBUFSIZ, &pwp);
    if (ret != 0 || !pwp) 
    {
	// Should be one since user was authenticated.
        ap_log_error(APLOG_MARK, APLOG_EMERG, ret, NULL,
	"%s: Couldn't fetch user passwd file entry (max size: %d): %s", 
		module_name, PBUFSIZ, user);
        return 0;
    }
    // If group id on passwd entry matches group id from the
    // the group file then we have a match.
    if (pwp->pw_gid == grpp->gr_gid) 
        return 1;

    // Group did not match in the pw file.  Send debug info to log.
    ap_log_error(APLOG_MARK, APLOG_DEBUG, ret, NULL,
	"req grp %s, grpfile grp %s, passfile grpid %d, grpfile grpid %d", 
	groupname, grpp->gr_name, pwp->pw_gid, grpp->gr_gid);

    // No match
    return 0;
}

/*
 *  auth_shadow_valid_user
 *
 *  Check the requires field to see if this user has access to
 *  this resource.
 *
 */

static int auth_shadow_valid_user(request_rec *r)
{
       /* req_arr is the array of requires lines */
       const apr_array_header_t *req_arr = ap_requires(r);
       require_line *requires;
       char *user = r->user; /* The user connected. */
       int m = r->method_number;  /* The method number 
			 	(e.g. M_GET, M_POST, etc..) */
       int i;
       int method_restricted = 0; 
       const char *line;          /* The requires line. */
       const char *w;             /* A word from the requires line. */

       auth_shadow_config_rec *s =
            (auth_shadow_config_rec *) ap_get_module_config(r->per_dir_config, &auth_shadow_module);           

       if (s->auth_shadow_flag != 1)
            return DECLINED;

       if (!req_arr) {
            /* No requires lines.  Any user will do. */
            return OK;
       }
       requires = (require_line *) req_arr->elts;
        
       for (i=0; i < req_arr->nelts; i++) {
            /* Process one requires line. */

            if (!(requires[i].method_mask & (AP_METHOD_BIT << m))) 
                continue;  /* The method coming through isn't restricted */
             method_restricted = 1;  /* Something was restricted */
             line = requires[i].requirement;
             w = ap_getword_white(r->pool, &line);
             if (!strcmp(w, "valid-user"))
                return OK;
             if (!strcmp(w, "user")) {
                     /* See if the user is one of the listed users. */
                     while (line[0]) {
                        w = ap_getword_conf(r->pool, &line);
                        if (!strcmp(user,w))
                            return OK;
                     }
             }
             else if (!strcmp(w,"group")) {
                /* See if the user is a member of one of the listed groups. */
		if (!line[0])
		    ap_log_error(APLOG_MARK, APLOG_ERR, 0, NULL,
		        "Group required but none specified for %s",
			r->uri);
		else while (line[0]) {
                    w = ap_getword_conf(r->pool, &line);
                    if (user_in_group(user,w))
                        return OK;
                }
             }
       }
    if (!method_restricted)  /* This method wasn't restricted */
        return OK;
   
    ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, 0, r, 
           "access to %s failed.  Reason: user %s not allowed access", 
           r->uri, user);

    ap_note_basic_auth_failure(r);
    return HTTP_UNAUTHORIZED;
}
