/* ====================================================================
 * Copyright (c) 1995 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * IT'S CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */


/* Uncomment if you want to use a HARDCODE'd check (default off) */
/* #define _HARDCODE_ */

#ifdef _HARDCODE_
  /* Uncomment if you want to use your own Hardcode (default off) */
  /*             MUST HAVE _HARDCODE_ defined above!                */
  /* #include "your_function_here.c" */
#endif


#include "apr_lib.h"

#include "ap_config.h"
#include "ap_provider.h"
#include "mod_auth.h"

#define APR_WANT_STRFUNC
#include "apr_want.h"
#include "apr_strings.h"

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "http_request.h"   /* for ap_hook_(check_user_id | auth_checker)*/
#include <sys/wait.h>
#include <signal.h>
#if APR_HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifndef STANDARD20_MODULE_STUFF
#error This module requires Apache 2.2.0 or later.
#endif

/* Names of environment variables used to pass data to authenticator */
#define ENV_USER	"USER"
#define ENV_PASS	"PASS"
#define ENV_GROUP	"GROUP"
#define ENV_URI		"URI"
#define ENV_IP		"IP"
#define ENV_HOST	"HOST"		/* Remote Host */
#define ENV_HTTP_HOST   "HTTP_HOST"	/* Local Host */
/* Define this if you want cookies passed to the script */
#define ENV_COOKIE	"COOKIE"

/* Maximum number of arguments passed to an authenticator */
#define MAX_ARG 32

/* Default authentication method - "pipe", "environment" or "checkpass" */
#define DEFAULT_METHOD "pipe"

/*
 * Structure for the module itself.  The actual definition of this structure
 * is at the end of the file.
 */
module AP_MODULE_DECLARE_DATA authnz_external_module;

/*
 *  Data types for per-directory and per-server configuration
 */

typedef struct
{
    char *auth_name;		 /* Auth keyword for current dir */
    char *group_name;		 /* Group keyword for current dir */
    int  authoritative;		 /* Are we authoritative in current dir? */
    int  groupsatonce;		 /* Check all groups in one call in this dir? */

} authnz_external_dir_config_rec;


typedef struct
{
    apr_table_t *auth_path;	 /* Hash mapping auth keywords to paths */
    apr_table_t *auth_method;	 /* Hash mapping auth keywords to methods */

    apr_table_t *group_path;	 /* Hash mapping group keywords to paths */
    apr_table_t *group_method;	 /* Hash mapping group keywords to methods */

} authnz_external_svr_config_rec;


/*
 * Creators for per-dir and server configurations.  These are called
 * via the hooks in the module declaration to allocate and initialize
 * the per-directory and per-server configuration data structures declared
 * above.
 */

static void *create_authnz_external_dir_config(apr_pool_t *p, char *d)
{
    authnz_external_dir_config_rec *dir= (authnz_external_dir_config_rec *)
	apr_palloc(p, sizeof(authnz_external_dir_config_rec));

    dir->auth_name= NULL;	/* no default */
    dir->group_name= NULL;	/* no default */
    dir->authoritative= 1;	/* strong by default */
    dir->groupsatonce= 1;	/* default to on */
    return dir;
}


static void *create_authnz_external_svr_config( apr_pool_t *p, server_rec *s)
{
    authnz_external_svr_config_rec *svr= (authnz_external_svr_config_rec *)
	apr_palloc(p, sizeof(authnz_external_svr_config_rec));

    svr->auth_method=  apr_table_make(p, 4);
    svr->auth_path=    apr_table_make(p, 4);
    svr->group_method= apr_table_make(p, 4);
    svr->group_path=   apr_table_make(p, 4);
    /* Note: 4 is only initial hash size - they can grow bigger) */

    return (void *)svr;
}


/*
 * Handler for a AddExternalAuth server config line - add a external auth
 * type to the server configuration
 */

static const char *add_extauth(cmd_parms *cmd, void *dummy, const char *keyword,
				const char *path)
{
    authnz_external_svr_config_rec *svr= (authnz_external_svr_config_rec *)
	ap_get_module_config( cmd->server->module_config,
	    &authnz_external_module);

    apr_table_set( svr->auth_path,   keyword, path );
    apr_table_set( svr->auth_method, keyword, DEFAULT_METHOD );

    return NULL;
}


/*
 * Handler for a AddExternalGroup server config line - add a external group
 * type to the server configuration
 */

static const char *add_extgroup(cmd_parms *cmd, void *dummy,
			 	const char *keyword, const char *path)
{
    authnz_external_svr_config_rec *svr= (authnz_external_svr_config_rec *)
	ap_get_module_config( cmd->server->module_config,
	    &authnz_external_module);

    apr_table_set( svr->group_path,   keyword, path );
    apr_table_set( svr->group_method, keyword, DEFAULT_METHOD );

    return NULL;
}

/*
 * Handler for a SetExternalAuthMethod server config line - change an external
 * auth method in the server configuration
 */

static const char *set_authnz_external_method(cmd_parms *cmd, void *dummy,
					const char *keyword, const char *method)
{
    authnz_external_svr_config_rec *svr= (authnz_external_svr_config_rec *)
	ap_get_module_config( cmd->server->module_config,
	    &authnz_external_module);

    apr_table_set( svr->auth_method, keyword, method );
	
    return NULL;
}


/*
 * Handler for a SetExternalGroupMethod server config line - change an external
 * group method in the server configuration
 */

static const char *set_extgroup_method(cmd_parms *cmd, void *dummy,
					const char *keyword, const char *method)
{
    authnz_external_svr_config_rec *svr= (authnz_external_svr_config_rec *)
	ap_get_module_config( cmd->server->module_config,
	    &authnz_external_module);

    apr_table_set( svr->group_method, keyword, method );
	
    return NULL;
}


/*
 * Config file commands that this module can handle
 */

static const command_rec authnz_external_cmds[] =
{
    AP_INIT_TAKE1("AuthExternal",
	ap_set_string_slot,
	(void *)APR_OFFSETOF(authnz_external_dir_config_rec,auth_name),
	OR_AUTHCFG,
	"a keyword indicating which authenticator to use"),

    AP_INIT_TAKE2("AddExternalAuth",
	add_extauth,
	NULL,
	RSRC_CONF,
	"a keyword followed by a path to the authenticator program"),
	    
    AP_INIT_TAKE2("SetExternalAuthMethod",
	set_authnz_external_method,
	NULL,
	RSRC_CONF,
	"a keyword followed by the method by which the data is passed"),

    AP_INIT_TAKE1("GroupExternal",
	ap_set_string_slot,
	(void *)APR_OFFSETOF(authnz_external_dir_config_rec,group_name),
	OR_AUTHCFG,
	"a keyword indicating which group checker to use"),
		
    AP_INIT_TAKE2("AddExternalGroup",
	add_extgroup,
	NULL,
	RSRC_CONF,
	"a keyword followed by a path to the group check program"),

    AP_INIT_TAKE2("SetExternalGroupMethod",
	set_extgroup_method,
	NULL,
	RSRC_CONF, 
	"a keyword followed by the method by which the data is passed"),

    AP_INIT_FLAG("AuthzExternalAuthoritative",
	ap_set_flag_slot,
	(void *)APR_OFFSETOF(authnz_external_dir_config_rec, authoritative),
	OR_AUTHCFG,
	"Set to 'off' to allow access control to be passed along to lower "
	    "modules if this module can't confirm access rights" ),

    AP_INIT_FLAG("AuthExternalGroupsAtOnce",
	ap_set_flag_slot,
	(void *)APR_OFFSETOF(authnz_external_dir_config_rec, groupsatonce),
	OR_AUTHCFG,
	"Set to 'off' if group authenticator cannot handle multiple group "
	    "names in one invocation" ),

    { NULL }
};


/*
 * Run an external authentication program using either the "pipe",
 * "checkpasswd" or "environment" method to pass the arguments.
 */

static int exec_external(const char *extpath, const char *extmethod,
		const request_rec *r, const char *dataname, const char *data)
{
    int pipe_to_auth[2];
    int pid, status;
    conn_rec *c= r->connection;
    const char *remote_host;
    int usecheck= extmethod && !strcasecmp(extmethod, "checkpassword");
    int usepipe= usecheck || (extmethod && !strcasecmp(extmethod, "pipe"));

    if (usepipe && pipe(pipe_to_auth))

	/* pipe() failed - weird */
	return -3;

    if ( (pid= fork()) < 0 )
    {
	/* fork() failed - weird */
	if (usepipe)
	{
	    close(pipe_to_auth[0]);
	    close(pipe_to_auth[1]);
	}
	return -4;
    }
    else if (pid == 0)
    {
	/* We are the child process */

	char *child_env[11];
	char *child_arg[MAX_ARG+2];
	const char *t;
	const char *cookie, *host;
	int i= 0;

	/* Put authentication type, PATH, hostname, ip address, uri, etc
	 * into environment */
	child_env[i++]= apr_pstrcat(r->pool, "AUTHTYPE=", dataname, NULL);

	child_env[i++]= apr_pstrcat(r->pool, "PATH=", getenv("PATH"), NULL);

	remote_host= ap_get_remote_host(c, r->per_dir_config, REMOTE_HOST,NULL);
	if (remote_host != NULL)
	    child_env[i++]= apr_pstrcat(r->pool, ENV_HOST"=", remote_host,NULL);

	if (c->remote_ip)
	    child_env[i++]= apr_pstrcat(r->pool, ENV_IP"=", c->remote_ip, NULL);

	if (r->uri)
	    child_env[i++]= apr_pstrcat(r->pool, ENV_URI"=", r->uri, NULL);

	if ((host= apr_table_get(r->headers_in, "Host")) != NULL)
	    child_env[i++]= apr_pstrcat(r->pool, ENV_HTTP_HOST"=", host, NULL);

#ifdef ENV_COOKIE
	if ((cookie= apr_table_get(r->headers_in, "Cookie")) != NULL)
	    child_env[i++]= apr_pstrcat(r->pool, ENV_COOKIE"=", cookie, NULL);
#endif

	/* Direct stdout and stderr to log file */
	ap_error_log2stderr(r->server);
	dup2(2,1);

	/* Close any open file descriptors and such in the pool. This won't
	 * close the pipe because it isn't in the pool.  It will close
	 * anything that might be on fd 3, which is important to do before
	 * we try to attach the pipe to that in the checkpassword case.  */
	apr_pool_cleanup_for_exec();

	if (usepipe)
	{
	    /* Connect stdin to pipe */
	    dup2(pipe_to_auth[0], usecheck ? 3 : 0);
	    close(pipe_to_auth[0]);
	    close(pipe_to_auth[1]);
	}
	else
	{
	    /* Put user name and password/group into environment */
	    child_env[i++]= apr_pstrcat(r->pool, ENV_USER"=", r->user, NULL);
	    child_env[i++]= apr_pstrcat(r->pool, dataname, "=", data, NULL);
	}

	/* End of environment */
	child_env[i]= NULL;

	/* Construct argument array */
	for (t= extpath, i=0; t[0] && (i <= MAX_ARG + 1);
	     child_arg[i++]= ap_getword_white(r->pool, &t)) {}
	child_arg[i]= NULL;

	/* Overwrite ourselves with the authenticator program */
	execve(child_arg[0], child_arg, child_env);

	/* If execve failed: */ exit(-1);
    }
    else
    {
	/* We are the parent process */

	if (usepipe)
	{
	    close(pipe_to_auth[0]);
	  
	    /* Send the user */
	    write(pipe_to_auth[1], r->user, strlen(r->user));
	    write(pipe_to_auth[1], usecheck ? "\0" : "\n", 1);
	  
	    /* Send the password */
	    write(pipe_to_auth[1], data, strlen(data));
	    write(pipe_to_auth[1], usecheck ? "\0" : "\n", 1);

	    /* Send dummy timestamp for checkpassword */
	    if (usecheck) write(pipe_to_auth[1], "0", 2);

	    /* Close output */
	    close(pipe_to_auth[1]);
	}
      
	/* Await a response */
	waitpid(pid, &status, 0);
	return WIFEXITED(status) ? WEXITSTATUS(status) : -2;
    }
}


/* Call the hardcoded function specified by the external path.  Of course,
 * you'll have to write the hardcoded functions yourself and insert them
 * into this source file, as well as inserting a call to them into this
 * routine.
 */

static int exec_hardcode(const request_rec *r, const char *extpath,
	const char *password)
{
#ifdef _HARDCODE_
    char *check_type;		/* Pointer to HARDCODE type check  */
    char *config_file;		/* Pointer to HARDCODE config file */
    int standard_auth= 0;

    /* Parse a copy of extpath into type and filename */
    check_type= apr_pstrdup(r->pool, extpath);
    config_file= strchr(check_type, ':');
    if (config_file != NULL)
    {
	*config_file= '\0';		   /* Mark end of type */
	config_file++;       	           /* Start of filename */
    }

    /* This is where you make your function call.  Here is an example of
     * what one looks like:
     *
     *   if (strcmp(check_type,"RADIUS")==0)
     *      code= radcheck(r->user,password,config_file);
     *
     * Replace 'radcheck' with whatever the name of your function is.
     * Replace 'RADIUS' with whatever you are using as the <type> in:
     *     AddExternalAuth <keyword> <type>:<config file>
     */

    if (strcmp(check_type,"EXAMPLE")==0)		/* change this! */
        code= example(r->user,password,config_file);	/* change this! */
    else
	code= -5;
    return code;
#else
    return -4;		/* If _HARDCODE_ is not defined, always fail */
#endif /* _HARDCODE_ */
}


static int authz_external_check_user_access(request_rec *r) 
{
    authnz_external_dir_config_rec *dir= (authnz_external_dir_config_rec *)
	ap_get_module_config(r->per_dir_config, &authnz_external_module);

    authnz_external_svr_config_rec *svr= (authnz_external_svr_config_rec *)
	ap_get_module_config(r->server->module_config, &authnz_external_module);

    int code;
    int m= r->method_number;
    const char *extpath, *extmethod;
    char *extname= dir->group_name;
    int required_group= 0;
    register int x;
    const char *t, *w;
    const apr_array_header_t *reqs_arr= ap_requires(r);
    const char *filegroup= NULL;
    require_line *reqs;

    /* If no external authenticator has been configured, pass */
    if ( !extname ) return DECLINED;

    /* If there are no Require arguments, pass */
    if (!reqs_arr) return DECLINED;
    reqs=  (require_line *)reqs_arr->elts;


    /* Loop through the "Require" argument list */
    for(x= 0; x < reqs_arr->nelts; x++)
    {
	if (!(reqs[x].method_mask & (AP_METHOD_BIT << m))) continue;

	t= reqs[x].requirement;
	w= ap_getword_white(r->pool, &t);

	/* The 'file-group' directive causes mod_authz_owner to store the
	 * group name of the file we are trying to access in a note attached
	 * to the request.  It's our job to decide if the user actually is
	 * in that group.  If the note is missing, we just decline.
	 */
	if ( !strcasecmp(w, "file-group"))
	{
	    filegroup= apr_table_get(r->notes, AUTHZ_GROUP_NOTE);
	    if (filegroup == NULL) continue;
	}

	if( !strcmp(w,"group") || filegroup != NULL)
	{
	    required_group= 1;

	    if (t[0] || filegroup != NULL)
	    {
		/* Get the path and method associated with that external */
		if (!(extpath= apr_table_get(svr->group_path, extname)) ||
		    !(extmethod= apr_table_get(svr->group_method,
			    extname)))
		{
		    errno= 0;
		    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
		    	"invalid GroupExternal keyword (%s)", extname);
		    ap_note_basic_auth_failure(r);
		    return HTTP_INTERNAL_SERVER_ERROR;
		}

		if (filegroup != NULL)
		{
		    /* Check if user is in the group that owns the file */
		    code= exec_external(extpath, extmethod, r, ENV_GROUP,
			    filegroup);
		    if (code == 0) return OK;
		}
		else if (dir->groupsatonce)
		{
		    /* Pass rest of require line to authenticator */
		    code= exec_external(extpath, extmethod, r, ENV_GROUP, t);
		    if (code == 0) return OK;
		}
		else
		{
		    /* Call authenticator once for each group name on line */
		    do {
		        w= ap_getword_white(r->pool, &t);
			code= exec_external(extpath,
				extmethod, r, ENV_GROUP, w);
			if (code == 0) return OK;
		    } while(t[0]);
		}
	    }
	}
    }
    
    /* If we didn't see a 'require group' or aren't authoritive, decline */
    if (!required_group || !dir->authoritative)
	return DECLINED;

    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
    	"access to %s failed, reason: user %s not allowed access",
    	r->uri, r->user);

    ap_note_basic_auth_failure(r);
    return HTTP_UNAUTHORIZED;
}



/* Password checker for basic authentication - given a login/password,
 * check if it is valid.  Returns one of AUTH_DENIED, AUTH_GRANTED,
 * or AUTH_GENERAL_ERROR.
 */

static authn_status authn_external_check_password(request_rec *r,
	const char *user, const char *password)
{
    const char *extpath, *extmethod;
    authnz_external_dir_config_rec *dir= (authnz_external_dir_config_rec *)
	    ap_get_module_config(r->per_dir_config, &authnz_external_module);

    authnz_external_svr_config_rec *svr= (authnz_external_svr_config_rec *)
	    ap_get_module_config(r->server->module_config,
		&authnz_external_module);
    const char *extname= dir->auth_name;
    int code= 1;

    /* Check if we are supposed to handle this authentication */
    if ( extname == NULL )
    {
	ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
	    "No AuthExternal name has been set");
	return AUTH_GENERAL_ERROR;
    }

    /* Get the path associated with that external */
    if (!(extpath= apr_table_get(svr->auth_path, extname)))
    {
	ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
	    "Invalid AuthExternal keyword (%s)", extname);
	return AUTH_GENERAL_ERROR;
    }

    /* Do the authentication, by the requested method */
    extmethod= apr_table_get(svr->auth_method, extname);
    if ( extmethod && !strcasecmp(extmethod, "function") )
	code= exec_hardcode(r, extpath, password);
    else
    	code= exec_external(extpath, extmethod, r, ENV_PASS, password);

    /* If return code was zero, authentication succeeded */
    if (code == 0) return AUTH_GRANTED;
    
    /* Otherwise it failed */
    errno= 0;
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
    	"AuthExtern %s [%s]: Failed (%d) for user %s",
	extname, extpath, code, r->user);
    return AUTH_DENIED;
}


#if 0
/* Password checker for digest authentication - given a login/password,
 * check if it is valid.  Returns one of AUTH_USER_FOUND, AUTH_USER_NOT_FOUND,
 * or AUTH_GENERAL_ERROR.   Not implemented at this time.
 */

auth_status *authn_external_get_realm_hash(request_rec *r, const char *user,
    const char *realm, char **rethash);
{
}
#endif


static const authn_provider authn_external_provider =
{
    &authn_external_check_password,
#if 0
    &authn_external_get_realm_hash
#else
    NULL		/* No support for digest authentication at this time */
#endif
};


static void register_hooks(apr_pool_t *p)
{
    ap_register_provider(p, AUTHN_PROVIDER_GROUP, "external", "0",
	    &authn_external_provider);

    ap_hook_auth_checker(authz_external_check_user_access, NULL, NULL,
	    APR_HOOK_MIDDLE);
}
    

module AP_MODULE_DECLARE_DATA authnz_external_module = {
    STANDARD20_MODULE_STUFF,
    create_authnz_external_dir_config,	  /* create per-dir config */
    NULL,			  /* merge per-dir config - dflt is override */
    create_authnz_external_svr_config, /* create per-server config */
    NULL,			  /* merge per-server config */
    authnz_external_cmds,	  /* command apr_table_t */
    register_hooks		  /* register hooks */
};
