/*
* libcafix - communicates with CASIO graphing calculators.
* Copyright (C) 2001  Gran Weinholt <weinholt@linux.nu>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* $Id: comm.c,v 1.3 2001/09/07 17:57:27 weinholt Exp $
*/
#include <stdlib.h>
#include <unistd.h>
#include <cafix.h>

static char rcsid[] = "$Id: comm.c,v 1.3 2001/09/07 17:57:27 weinholt Exp $\n";

/* These are in header.c */
void cafix_hdr_generate(cafixdata *data);
void cafix_hdr_translate(cafixdata *data);
void cafix_hdr_get_format(cafixdata *data);

/* From io.c */
void _cafix_sync(int fd);

/* Empty callback */
int cafix_empty_status_callback(cafixdata *data)
{
    return 0;
}

/* Send header. */
int cafix_send_header(int fd, cafixdata *data, int (*status_callback) (cafixdata *data))
{
    int snafu = 0;
    char bar;

    cafix_hdr_generate(data);

    /* The first of everything is ':' */
    bar = ':';
    write(fd, &bar, 1);

    data->status = STAT_SENDING_HDR;
    for (data->transmitted = 0; data->transmitted < data->hlength; data->transmitted++) {
	snafu += write(fd, &data->rawheader[data->transmitted], 1);
	//printf("%i ", (signed char) data->rawheader[data->transmitted]);
	if (status_callback(data)) {
	    data->status = STAT_USER_ABORT;
	    return 1;
	}
    }
    if (snafu != data->transmitted) {
	data->status = STAT_ERROR_COMM;
	return 1;
    }
    _cafix_sync(fd);
    status_callback(data);

    data->transmitted = 0;
    return 0;
}

/* Tell the calc. we want to send data */
int cafix_handshake_send(int fd, cafixdata *data)
{
    char bar;

    bar = 0x16;
    if (data->interactive == 1)
	bar = 0x06;
    write(fd, &bar, 1);
    if ((read(fd, &bar, 1) < 1)) {
	data->status = STAT_ERROR_COMM;
	return 1;
    }
    if (data->interactive == 1 && bar == 0x06)
	return 0;
    if (bar != 0x13) {
	data->status = STAT_ERROR_NOT_READY;
	return 1;
    }
    return 0;
}

/* Send header and data. If this returns non-zero then check data->status. */
int cafix_send(int fd, cafixdata *data, int (*status_callback) (cafixdata *data))
{
    /* 
     * Calculate the CRC of the data.
     * Send header and then send data.
     */
    int i, snafu = 0;
    char foo, bar;
    
    if (!data)
	return 1;

    /* Calculate the CRC of the data. */
    data->crc = 0;
    for (i = 0; i < data->length; i++)
    	data->crc += (char)(data->data[i]);
    data->crc = abs(255 - (data->crc % 256)) + 1;
    
    /* Send header. */
    if (cafix_send_header(fd, data, status_callback))
	return 1;	/* data->status already set */

    data->status = STAT_SENDING;

    /* Check if the calc. accepted the header. */
    read(fd, &foo, 1);
    switch (foo) {
    case 0x06:	/* Everything is fine. */
	break;
    case 0x24:	/* Out of memory */
	data->status = STAT_ERROR_MEMORY;
	return 1;
    case 0x2b:	/* CRC error */
	data->status = STAT_ERROR_CRC;
	return 1;
    case 0x21:	/* File already exists */
	data->status = STAT_OVERWRITE;
	/* Ask the user if we should overwrite. */
	if (!status_callback(data)) {
	    /* User told us to abort. */
	    bar = 0x15;
	    write(fd, &bar, 1);
	    read(fd, &bar, 1);
	    if (bar != 0x06) {
		/* FIXME: somewhere around here we have to send some abortcode i think */
		data->status = STAT_ERROR_UNKNOWN;
		return 1;
	    }
	    data->status = STAT_USER_ABORT;
	    return 1;
	}
	bar = 0x06;
	write(fd, &bar, 1);
	read(fd, &bar, 1);
	if (bar != 0x06) {   	/* If it's 0x06 then everything is OK. */
	    data->status = STAT_ERROR_UNKNOWN;
	    return 1;
	}
	break;
    case 0x15:
	/* Something wrong with the header. XXX: ? */
	data->status = STAT_ERROR_HEADER;
	return 1;
	break;
    case 0x00:
	/* Does this mean Zero length data, send no more, it's OK? */
	data->status = STAT_OK;
	return 0;
    case 0x22:
    default:
	data->status = STAT_ERROR_UNKNOWN;
	return 1;
    }

    if (data->length == 0) {
	/* FIXME: isn't this taken care of when it's needed by the
	   case 0x00 in the above switch? */
	data->status = STAT_OK;
	return 0;
    }

    /* Transmit all the data */
    data->status = STAT_SENDING_DATA;
    foo = ':';
    write(fd, &foo, 1);
    for (data->transmitted = 0; data->transmitted < data->length; data->transmitted++) {
	snafu += write(fd, data->data + data->transmitted, 1);
	/* XXX: the higher 11 gets, the faster the transfer is,
	   but the less accurate a statusbar gets. */
	if (data->transmitted % 13 == 0)
	    _cafix_sync(fd);
	/*printf("%c ", data->data[data->transmitted]);*/
	if (status_callback(data)) {
	    data->status = STAT_USER_ABORT;
	    return 1;
	}
    }
    if (snafu != data->transmitted) {
	data->status = STAT_ERROR_COMM;
	return 1;
    }
    status_callback(data);
    
    write(fd, &data->crc, 1);
    read(fd, &bar, 1);
    if (bar != 0x06) {
	data->status = STAT_ERROR_CRC;
	return 1;
    }
    data->status = STAT_OK;
    return 0;
}

int cafix_receive_header(int fd, cafixdata *data, int (*status_callback) (cafixdata *data))
{
    int snafu;
    char bar;

    /* All good things begin with ':' */
    read(fd, &bar, 1);
    if (bar != ':') {
	data->status = STAT_ERROR_COMM;
	return 1;
    }

    data->status = STAT_RECEIVING_HDR;
    data->hlength = 100;	/* this gets adjusted by hdr_get_format */
    for (snafu = data->transmitted = 0; data->transmitted < data->hlength; data->transmitted++) {
	snafu += read(fd, &data->rawheader[data->transmitted], 1);
	if (data->transmitted == 7)
	    cafix_hdr_get_format(data);	/* Now we see how long the header really is */
	if (status_callback(data)) {
	    data->status = STAT_USER_ABORT;
	    return 1;
	}
    }
    if (snafu != data->transmitted) {
	data->status = STAT_ERROR_COMM;
	return 1;
    }
    status_callback(data);

    data->transmitted = 0;
    return 0;
}

/* Handshake that we want to receive data. */
int cafix_handshake_receive(int fd, cafixdata *data)
{
    char bar;

    read(fd, &bar, 1);
    switch (bar) {
    case 0x15:
	bar = 0x13;
	write(fd, &bar, 1);
	data->interactive = 1;
	return 0;
    case 0x16:
	bar = 0x13;
	write(fd, &bar, 1);
	data->interactive = 0;
	return 0;
    default:
	data->status = STAT_ERROR_NOT_READY;
	return 1;
    }
    return 0;
}

int cafix_receive(int fd, cafixdata *data, int (*status_callback) (cafixdata *data))
{
    /*
     * Receive header. Translate it.
     * Receive data.
     */
    int bar, snafu;
    char crc = 0;

    if (cafix_receive_header(fd, data, status_callback))
	return 1;

    data->status = STAT_RECEIVING;

    cafix_hdr_translate(data);

    if (data->format == FMT_REQUEST) {
	/* Don't answer at all, the handshake after this will do the stuff. */
	data->status = STAT_OK;
	return 0;
    }

    /* No data to get, right? */
    if (data->length == 0) {
	/* XXX: is this always the right thing to do? */
	bar = 0x00;
	write(fd, &bar, 1);
	data->status = STAT_OK;
	return 0;
    }

    /* This type of transfer requires some magical sort of CRC-send after
	every 13 bytes but then something unknown happens. */
    if (data->datatype == DATA_MT) {
	data->status = STAT_NOT_IMPLEMENTED;
	return 1;
    }

    /* Tell the calculator that we accepted the header */
    bar = 0x06;
    write(fd, &bar, 1);
    /* FIXME: implement crc-error? */

    /* Allocate memory for data and receive */
    data->status = STAT_RECEIVING_DATA;
    if ((data->data = calloc(1, (unsigned int)data->length)) == NULL) {
	data->status = STAT_ERROR_MEMORY;
	return 1;
    }
    read(fd, &bar, 1);
    if (bar != ':') {
	/* XXX: Is this always ':'? Try it with all data types. */
	data->status = STAT_ERROR_COMM;
	return 1;
    }
    for (snafu = data->transmitted = 0; data->transmitted < (unsigned int)data->length; data->transmitted++) {
	snafu += read(fd, data->data + data->transmitted, 1);
	if (snafu == data->transmitted)	// if we can't receive CRC isn't correct
	    crc += data->data[data->transmitted];
	if (data->format == FMT_CDUMP && (data->transmitted == 1024 || data->transmitted == 2048)) {
	    char fnord;

	    read(fd, &fnord, 1);
	    // FIXME: compare this to our crc?
	    fnord = 0x06;
	    write(fd, &fnord, 1);
	    read(fd, &fnord, 1);
	    // FIXME: fnord should now be ':'
	    crc = 0;
	}
	if (data->format == FMT_PROGRAM && data->transmitted == 189 && data->datatype == DATA_9800_PZ && data->rawheader[1] == 'Z') {
	    /* This is eveil.., but the better part of the protocol _is_ eveil */
	    /* Actually, I don't think this works. Maybe there's a table of the programs in the first part of the data. */
	    char fnord;

	    read(fd, &fnord, 1);
	    // FIXME: compare this to our crc?
	    fnord = 0x06;
	    write(fd, &fnord, 1);
	    read(fd, &fnord, 1);
	    // FIXME: fnord should now be ':'
	    crc = 0;
	}
	if (data->datatype == DATA_MT && data->transmitted > 0 && (data->transmitted % 13) == 0) {
	    char fnord;

	    read(fd, &fnord, 1);
	    // FIXME: compare this to our crc?
	    fnord = 0x06;
	    write(fd, &fnord, 1);
	    read(fd, &fnord, 1);
	    // FIXME: fnord should now be ':'
	    crc = 0;
	}
	if (status_callback(data)) {
	    data->status = STAT_USER_ABORT;
	    return 1;
	}
    }
    if (snafu != data->transmitted) {
	data->status = STAT_ERROR_COMM;
	return 1;
    }
    status_callback(data);

    /* Receive CRC */
    read(fd, &data->crc, 1);
    crc = abs(255 - (crc % 256)) + 1;
    // FIXME: check if crc is correct?

    /* Tell the calculator the CRC was good */
    bar = 0x06;
    write(fd, &bar, 1);
    
    data->status = STAT_OK;
    return 0;
}
