/* This file is part of libccc
 *
 * AUTHORS
 *     Sven Herzberg  <herzi@gnome-de.org>
 *
 * Copyright (C) 2007  Sven Herzberg
 *
 * This work is provided "as is"; redistribution and modification
 * in whole or in part, in any medium, physical or electronic is
 * permitted without restriction.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * In no event shall the authors or contributors be liable for any
 * direct, indirect, incidental, special, exemplary, or consequential
 * damages (including, but not limited to, procurement of substitute
 * goods or services; loss of use, data, or profits; or business
 * interruption) however caused and on any theory of liability, whether
 * in contract, strict liability, or tort (including negligence or
 * otherwise) arising in any way out of the use of this software, even
 * if advised of the possibility of such damage.
 */

#include "s-item.h"

#include <ccc/cc-brush-color.h>
#include <ccc/cc-line.h>
#include <ccc/cc-rectangle.h>

#include "callgrind-helpers.h"
#include "cc-test-view.h"

static CcView* view  = NULL;
static CcItem* item  = NULL;
static CcItem* child = NULL;

static void
setup (void)
{
	view = cc_test_view_new ();
	item = cc_item_new ();
	g_object_ref_sink (item);
	child = cc_item_new ();
	g_object_ref_sink (child);
}

static void
teardown (void)
{
	g_object_unref (view);
	view = NULL;
	g_object_unref (item);
	item = NULL;
	g_object_unref (child);
	child = NULL;
}

static void
item_view_register (CcItem  * item,
		    CcView  * view,
		    gboolean* registered)
{
	if (G_OBJECT_TYPE (view) == CC_TYPE_TEST_VIEW) {
		*registered = TRUE;
	}
}

START_TEST(test_register_plain)
{
	/* prepare */
	gboolean registered;
	g_signal_connect (item, "view-register",
			  G_CALLBACK (item_view_register), &registered);

	/* check */
	registered = FALSE;

	CALLGRIND_START_INSTRUMENTATION;
	CALLGRIND_ZERO_STATS;
	cc_view_set_root (view, item);
	CALLGRIND_DUMP_STATS_AT("test_plain_register");
	CALLGRIND_STOP_INSTRUMENTATION;

	fail_unless (registered);

	/* cleanup */
}
END_TEST

START_TEST(test_register_recurse)
{
	gboolean registered;

	cc_item_append (item, child);
	g_signal_connect (child, "view-register",
			  G_CALLBACK (item_view_register), &registered);
	registered = FALSE;
	cc_view_set_root (view, item);
	fail_unless (registered);
}
END_TEST

#ifdef ENABLE_IMPOSSIBLE
static void
item_view_register_order (CcItem  * item,
			  CcView  * view,
			  gboolean* passed)
{
	enum {
		ORDER_START, /* initial state */
		ORDER_VIEW,  /* registered with the test view but not with the item */
		ORDER_ITEM,  /* registered with the item and the test view */
		ORDER_DONE = ORDER_ITEM
	};
	static gint state = ORDER_START;

	if (state == ORDER_START && g_type_is_a (G_OBJECT_TYPE (view), CC_TYPE_TEST_VIEW)) {
		state = ORDER_VIEW;
	} else if (state == ORDER_VIEW && g_type_is_a (G_OBJECT_TYPE (view), CC_TYPE_ITEM)) {
		state = ORDER_ITEM;
	} else {
		gchar const* states[] = {
			"start", "view", "item"
		};
		g_warning ("Error while checking view register order\nstate: %s\nview:  %s",
			   states[state], G_OBJECT_TYPE_NAME (view));
	}

	if (state == ORDER_DONE) {
		*passed = TRUE;
	}
}

START_TEST(test_register_order)
{
	gboolean passed = FALSE;

	cc_view_set_root (view, item);
	g_signal_connect (child, "view-register",
			  G_CALLBACK (item_view_register_order), &passed);
	cc_item_append (item, child);

	fail_unless (passed);
}
END_TEST
#endif

START_TEST(test_register_lazy)
{
	gboolean registered;

	cc_view_set_root (view, item);
	g_signal_connect (child, "view-register",
			  G_CALLBACK (item_view_register), &registered);
	registered = FALSE;
	cc_item_append (item, child);
	fail_unless (registered);
}
END_TEST

// TODO: item_append(item, child) => set_root(view, child) => no signal on item

static void
item_view_unregister (CcItem  * item,
		      CcView  * view,
		      gboolean* unregistered)
{
	if (G_OBJECT_TYPE (view) == CC_TYPE_TEST_VIEW) {
		*unregistered = TRUE;
	}
}

START_TEST(test_unregister_plain)
{
	gboolean unregistered;
	cc_view_set_root (view, item);
	g_signal_connect (item, "view-unregister",
			  G_CALLBACK (item_view_unregister), &unregistered);
	unregistered = FALSE;
	cc_view_set_root (view, NULL);
	fail_unless (unregistered);
}
END_TEST

START_TEST(test_unregister_recurse)
{
	gboolean unregistered;

	cc_item_append (item, child);
	cc_view_set_root (view, item);
	g_signal_connect (child, "view-unregister",
			  G_CALLBACK (item_view_unregister), &unregistered);
	unregistered = FALSE;
	cc_view_set_root (view, NULL);
	fail_unless (unregistered);
}
END_TEST

#ifdef ENABLE_IMPOSSIBLE
START_TEST(test_unregister_order)
{
	gboolean passed = FALSE;

	fail_unless (passed);
}
END_TEST
#endif

START_TEST(test_unregister_lazy)
{
	gboolean unregistered;

	cc_item_append (item, child);
	cc_view_set_root (view, item);
	g_signal_connect (child, "view-unregister",
			  G_CALLBACK (item_view_unregister), &unregistered);
	unregistered = FALSE;
	cc_item_remove (item, child);
	fail_unless (unregistered);
}
END_TEST

#include <ccc.h>

START_TEST(test_camera)
{
	CcItem* canvas = cc_item_new ();
	CcItem* demo   = cc_text_new ("Sliff Sloff Slubb");
	CcItem* camera = cc_camera_new_root (demo);
	CcItem* handle = cc_circle_new ();
	cc_view_set_zoom (CC_VIEW(camera), 0.25);
	cc_circle_set_radius (CC_CIRCLE (handle), 5.0);
	cc_circle_set_anchor (CC_CIRCLE (handle), 100.0, 100.0);
	cc_shape_set_brush_content (CC_SHAPE (handle), cc_brush_color_new (cc_color_new_rgb (0.64, 0.0, 0.0)));
	cc_item_append (canvas, demo);
	cc_item_append (canvas, camera);
	cc_item_append (canvas, handle);
	cc_view_set_root (view, canvas);
}
END_TEST

TCase*
tcase_item_view_signals (void)
{
	TCase* self = tcase_create ("View Signals");
	tcase_add_checked_fixture (self, setup, teardown);
	tcase_add_test (self, test_register_plain);
	tcase_add_test (self, test_unregister_plain);
	tcase_add_test (self, test_register_recurse);
	tcase_add_test (self, test_unregister_recurse);
#ifdef ENABLE_IMPOSSIBLE
	tcase_add_test (self, test_register_order);
	tcase_add_test (self, test_unregister_order);
#endif
	tcase_add_test (self, test_register_lazy);
	tcase_add_test (self, test_unregister_lazy);
	tcase_add_test (self, test_camera);
	return self;
}

