# Creates a fill-in form.
# Dynamic elements of form, for example, CKBrowser, CKCheckbox, CKRadioButton,
# CKPopUpButton, CKText, CKTextField, CKSubmitButton, CKResetButton and
# CKFileUpload are used within CKForm or HTML form.
#
# == Bindings
# Required attributes: none
#
# <b>method</b>::     Encode type to send form data.
#                     You can use "POST" or "GET" as the value.
# <b>enctype</b>::    Encode type for form data. When you use CKFileUpload,
#                     set this attribute to "multipart/form-data".
# <b>fileupload</b>:: If you set this attribute to true,
#                     "enctype" attribute is setted to "multipart/form-data".
#                     You can use this instead of "enctype" when using CKFileUpload.
# <b>href</b>::       URL to which the browser directs.
# <b>target</b>::     Frame in a frameset that receive the page.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKForm < CKElement
	include ElementState, ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			component.application.element_id = id
		end

		def own_attributes
			['method', 'enctype', 'fileupload', 'href', 'target']
		end
	end

	def run
		@attr_method     = fetch('method') || 'post'              # String
		@attr_enctype    = fetch 'enctype'                        # String
		@attr_href       = fetch('href')   || application.baseurl # String
		@attr_target     = fetch 'target'                         # String
		@attr_query      = fetch('query')  || {}                  # Hash
		@attr_fileupload = fetch 'fileupload'                     # true / false

		unless ( @attr_method.downcase == 'post' ) or \
		       ( @attr_method.downcase == 'get'  ) then
			@attr_method = 'post'
		end

		if @attr_fileupload == true then
			@attr_enctype = "multipart/form-data"
		end

		@attr_query.update state_for_conditional
	end

	def to_s
		parser               = CKHTMLParser.new( parent, body )
		value                = parser.parse
		element_id           = CKElementID.new
		element_id.component = parent.class.to_s
		element_id.element   = name
		
		to_s =  "<form method=\"#@attr_method\""
		to_s << " action=\"#@attr_href\""     if @attr_href
		to_s << " target=\"#@attr_target\""   if @attr_target
		to_s << " enctype=\"#@attr_enctype\"" if @attr_enctype
		to_s << other_attributes_string
		to_s << ">\n<div>\n"
		to_s << "<input type=\"hidden\" name=\"element_id\" "
		to_s << "value=\"#{element_id}\">\n"

		if application.session? then
			to_s << "<input type=\"hidden\" "
			to_s << "name=\"#{application.session_key}\" "
			to_s << "value=\"#{application.session_id}\">\n"
		end
		if @attr_query then
			to_s << _create_hidden_field( @attr_query )
		end

		to_s << "</div>\n#{value}\n</form>\n"
	end

	private

	def _create_hidden_field( query )
		fields = ''
		query.each do | key, value |
			fields << \
				"<input type=\"hidden\" name=\"#{key}\" value=\"#{value}\">\n"
		end
		fields
	end
end


# Creates a list whose multiple items can be selected.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attribute: <b>list</b>
#
# <b>escape</b>::   Escapes HTML control characters in "value"
#                   if the "escape" is true. The default value is true.
# <b>list</b>::     Array which is iterated through.
# <b>values</b>::   Array which is value for each "value" attributes of
#                   <option> elements.
# <b>selected</b>:: Items which are chosen from the list.
# <b>multiple</b>:: Multiple items of the list can be selected
#                   if the value is true.
# <b>size</b>::     Size of item in appearance.
# <b>enabled</b>::  If the value is false, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKBrowser < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			if component.parse_ckd_value definition['escape'] then
				value.each do | _value |
					_value = CKUtilities.unescape_html _value
				end
			end

			# set value as Array
			component.take_value( definition['selected'], value )
		end

		def own_attributes
			['escape', 'list', 'values', 'selected', 'multiple', 'size', 'enabled']
		end
	end

	def run
		@attr_name     = element_id
		@attr_escape   = fetch( 'escape' )
		@attr_list     = fetch( 'list' )
		@attr_values   = fetch( 'values' )
		@attr_selected = fetch( 'selected' ) || []
		@attr_multiple = fetch( 'multiple' )
		@attr_size     = fetch( 'size' )

		@attr_escape = true if @attr_escape.nil?

		check_required_attributes( ['list'] )
	end

	def to_s
		to_s = '<select'
		if @attr_name     then to_s << " name=\"#@attr_name\"" end
		if @attr_size     then to_s << " size=\"#@attr_size\"" end
		if @attr_multiple then to_s << " multiple" end
		to_s << other_attributes_string
		to_s << ">\n"

		if @attr_list then
			item = index = value = nil
			@attr_list.each_with_index do | item, index |
				to_s << "<option"
				if (@attr_selected == item) or @attr_selected.include?(item) then
					to_s << " selected"
				end
				if @attr_values then
					value = @attr_values[index]
					if (@attr_selected == value) or @attr_selected.include?(value) then
						to_s << " selected"
					end
					if @attr_escape then
						value = CKUtilities.escape_html value
					end
					to_s << " value=\"#{value}\""
				end
				to_s << ">"

				if @attr_escape then
					item = CKUtilities.escape_html item
				end
				to_s << "#{item}</option>\n"
			end
		end

		to_s << '</select>'
	end
end


# Creates a checkbox.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attributes: <b>selection</b> and <b>value</b>, or <b>checked</b>
#
# You use this element in two ways. One it the way to use "checked" attribute. 
# The other is the way to use both "selection" and "value" attributes. 
#
# <b>checked</b>::   If neither "value" nor "selection" attribute is nil and
#                    the value of "selection" is equal to that of "value",
#                    the check box is checked.
# <b>value</b>::     When the check box is checked,
#                    the value of "value" attribute is set to the component by
#                    the method specified by "selection" attribute.
# <b>selection</b>:: Object that the user chose from the check box.
# <b>enabled</b>::   If the value is false, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKCheckbox < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			if definition['checked']
				action = definition['checked']
				value  = true
			else
				action = definition['selection']
				value  = value.last
			end
			component.take_value( action, value )
		end

		def own_attributes
			['checked', 'value', 'selection', 'enabled']
		end
	end

	def run
		@attr_name      = element_id.to_s
		@attr_checked   = fetch( 'checked' )   # true or false
		@attr_value     = fetch( 'value' )     # String
		@attr_selection = fetch( 'selection' ) # String
		if ( ( @attr_value and @attr_selection ) and \
		     ( @attr_value ==  @attr_selection ) ) then
			@attr_checked = true
		end

		if application.request.form_values[element_id.to_s].empty? then
			@attr_checked = false
			if attr_name = fetch('checked', false) then
				parent.take_value(attr_name, false)
			end
		end

		check_required_attributes( ['selection','value'], ['checked'] )
		check_conflicted_attributes( 'selection', 'checked' )
		check_conflicted_attributes( 'value', 'checked' )
	end

	def to_s
		to_s =  "<input type=\"checkbox\" name=\"#@attr_name\""
		to_s << %Q' value="#{@attr_value || 1}"'
		if @attr_checked then to_s << " checked" end
		to_s << other_attributes_string
		to_s << '>'
	end
end


# Creates a pop-up menu.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attribute: <b>list</b>
#
# <b>escape</b>::   Escapes HTML control characters in the items of the list
#                   if the "escape" is true. The default value is true.
# <b>list</b>::     Array which is iterated through.
# <b>values</b>::   Array which is value for each "value" attributes of
#                   <option> elements.
# <b>default</b>::  The first item if no item is selected.
# <b>selected</b>:: Item that are chosen from the selection list.
# <b>enabled</b>::  If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKPopUpButton < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			if component.parse_ckd_value definition['escape'] then
				value = CKUtilities.unescape_html value.last
			else
				value = value.last
			end

			component.take_value( definition['selected'], value )
		end

		def own_attributes
			['escape', 'list', 'values', 'default', 'selected', 'enabled']
		end
	end

	def run
		@attr_name     = element_id.to_s
		@attr_escape   = fetch( 'escape' ) || true # true or false
		@attr_list     = fetch( 'list' )           # Enumerable
		@attr_values   = fetch( 'values' )         # Enumerable
		@attr_default  = fetch( 'default' )        # String
		@attr_selected = fetch( 'selected' )       # String

		check_required_attributes( ['list'] )
	end

	def to_s
		to_s = '<select'
		if @attr_name then to_s << " name=\"#@attr_name\"" end
		to_s << other_attributes_string
		to_s << ">\n"

		if @attr_default then
			to_s << '<option value=""'
			if @attr_selected == nil then
				to_s << " selected"
			end
			to_s << ">#@attr_default</option>\n"
		end

		if @attr_list then
			item = index = value = nil
			@attr_list.each_with_index do | item, index |
				to_s << "<option"
				if @attr_selected == item then
					to_s << " selected"
				end
				if @attr_values then
					value = @attr_values[index].to_s
					if @attr_selected == value then
						to_s << " selected"
					end
					if @attr_escape then
						value = CKUtilities.escape_html value
					end
					to_s << " value=\"#{value}\""
				end
				to_s << ">"

				# escape
				if @attr_escape then
					item = CKUtilities.escape_html item
				end
				to_s << "#{item}</option>\n"
			end
		end

		to_s << '</select>'
	end
end


# Creates a radio button.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attributes: <b>selection</b> and <b>value</b>, or <b>checked</b>
#
# <b>name</b>::      Name that identifies the radio button's group.
# <b>checked</b>::   If neither "value" nor "selection" attribute is nil and
#                    the value of "selection" is equal to that of "value",
#                    the check box is checked.
# <b>value</b>::     When the check box is checked,
#                    the value of "value" attribute is set to the component by
#                    the method specified by "selection" attribute.
# <b>selection</b>:: Object that the user chose from the check box.
# <b>enabled</b>::   If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKRadioButton < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			if definition['checked'] then
				action = definition['checked']
				value  = true
			else
				action = definition['selection']
			end
			component.take_value( action, value )
		end

		def own_attributes
			['name', 'checked', 'value', 'selection', 'enabled']
		end
	end

	def element_id
		id = super
		id.element = @attr_name
		id
	end

	def run
		@attr_name      = fetch( 'name' )      # String
		@attr_checked   = fetch( 'checked' )   # true or false
		@attr_value     = fetch( 'value' )     # String
		@attr_selection = fetch( 'selection' ) # String
		if ( ( @attr_value and @attr_selection ) and \
			( @attr_value == @attr_selection ) ) then
			@attr_checked = true
		end

		check_required_attributes( ['selection','value'], ['checked'] )
		check_conflicted_attributes( 'selection', 'checked' )
		check_conflicted_attributes( 'value', 'checked' )
	end

	def to_s
		value = "#{definition['oid']}.#@attr_value"
		to_s  = "<input type=\"radio\" name=\"#{element_id}\" value=\"#{value}\""
		to_s << " checked"  if @attr_checked
		to_s << other_attributes_string
		to_s << ">"
	end
end


# Creates a reset button.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attributes: none
#
# <b>value</b>:: Title of the button.
# <b>enabled</b>::  If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKResetButton < CKElement
	include ElementAttribute

	class << self
		def own_attributes
			['value', 'enabled']
		end
	end

	def run
		@attr_value = fetch( 'value' )    # String
		unless @attr_value then @attr_value = 'Reset' end
	end

	def to_s
		to_s  = "<input type=\"reset\" value=\"#@attr_value\""
		to_s << other_attributes_string
		to_s << ">"
		to_s
	end
end


# Creates a submit button.
# This element must be used within HTML form.
#
# == Bindings
# Required attributes: none
#
# <b>action</b>::   Method to invoke when the button is clicked.
# <b>value</b>::    Title of the button.
# <b>enabled</b>::  If the value is true, the element appears but is not active.
#                   In addition, it doesn't send the form data 
#                   although the button is clicked.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKSubmitButton < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			component.application.element_id = id
		end

		def own_attributes
			['action', 'value', 'enabled']
		end
	end

	def run
		@attr_name     = element_id
		@attr_action   = fetch( 'action', false ) # String
		@attr_value    = fetch( 'value' )         # String
		if @attr_value == nil then @attr_value = 'Submit' end
	end

	def to_s
		to_s = "<input type=\"submit\" name=\"#@attr_name\" value=\"#@attr_value\""
		to_s << other_attributes_string
		to_s << ">"
	end
end


# Creates a text area.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attribute: <b>value</b>
#
# <b>value</b>::    Value of the text area. 
# <b>columns</b>::  Column size.
# <b>rows</b>::     Row size.
# <b>validate</b>:: Format string to validate input value.
# <b>pass</b>::     If validating is passed, the value is true.
# <b>enabled</b>::  If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKText < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			key = definition['value']
			component[key] = value.last

			if (format_key = definition['validate']) and \
			   (pass_key = definition['pass']) then
				component.validate_with_format_key(key, format_key, pass_key)
			end
		end

		def own_attributes
			['value', 'columns', 'rows', 'enabled', 'validate', 'pass']
		end
	end

	def run
		@attr_name     = element_id
		@attr_value    = fetch( 'value' )    # String
		@attr_columns  = fetch( 'columns' )  # Integer
		@attr_rows     = fetch( 'rows' )     # Integer

		check_required_attributes( ['value'] )
	end

	def to_s
		to_s = "<textarea name=\"#@attr_name\""
		if @attr_columns  then to_s << " cols=\"#@attr_columns\"" end
		if @attr_rows     then to_s << " rows=\"#@attr_rows\""    end
		to_s << other_attributes_string
		to_s << '>'
		if @attr_value   then to_s << CKUtilities.escape_html(@attr_value) end
		to_s << '</textarea>'
	end
end


# Creates a text input field.
# This element must be used within CKForm or HTML form.
#
# == Bindings
# Required attribute: <b>value</b>
#
# <b>value</b>::     Value of the text field. If you set an accessor method to
#                    this element, the form data is set to a component automatically
#                    by the method.
# <b>type</b>::      Type of the text field. "text" is for a normal text input field,
#                    "password" is for a password field and 
#                    "hidden" is for a hidden field.
# <b>size</b>::      Size of the text field.
# <b>maxlength</b>:: Max length of data for the text field.
# <b>validate</b>::  Format string to validate input value.
# <b>pass</b>::      If validating is passed, the value is true.
# <b>enabled</b>::   If the value is true, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKTextField < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			key = definition['value']
			component[key] = value.last

			if (format_key = definition['validate']) and \
			   (pass_key = definition['pass']) then
				component.validate_with_format_key(key, format_key, pass_key)
			end
		end

		def own_attributes
			['value', 'type', 'size', 'maxlength', 'enabled', 'validate', 'pass']
		end
	end

	def run
		@attr_name      = element_id.to_s
		@attr_type      = fetch( 'type' )      # String
		@attr_value     = fetch( 'value' )     # String
		@attr_size      = fetch( 'size' )      # Integer
		@attr_maxlength = fetch( 'maxlength' ) # Integer
		if @attr_type == nil then @attr_type = 'text' end

		check_required_attributes( ['value'] )

		@attr_value = CKUtilities.escape_html @attr_value
	end

	def to_s
		to_s = "<input type=\"#@attr_type\""
		if @attr_name      then to_s << " name=\"#@attr_name\""           end
		if @attr_value     then to_s << " value=\"#@attr_value\""         end
		if @attr_size      then to_s << " size=\"#@attr_size\""           end
		if @attr_maxlength then to_s << " maxlength=\"#@attr_maxlength\"" end
		to_s << other_attributes_string
		to_s << '>'
	end
end


# CKFileUpload generates an input form to upload files.
# To use this, set "enctype" attribute of CKForm to "multipart/form-data" or
# "fileupload" attribute to true.
#
# == Bindings
# Required attribute: <b>data</b> and <b>file</b>
#
# <b>data</b>::    Variable of the attribute is set the uploaded file as
#                  a CKByteData object. 
# <b>file</b>::    Path of the uploaded file.
# <b>enabled</b>:: If the value is false, the element appears but is not active.
#
# == Programming Topics
# * DynamicElements[www.spice-of-life.net/download/cgikit/en/userguide/elements.html]
class CKFileUpload < CKElement
	include ElementAttribute

	class << self
		def bind_request( component, definition, value, id )
			data = value.last
			if data.is_a? CKByteData then
				component.take_value( definition['data'], data )
				component.take_value( definition['file'], data.path )
			end
		end

		def own_attributes
			['data', 'file', 'enabled']
		end
	end

	def run
		check_required_attributes( ['data', 'file'] )
	end

	def to_s
		to_s  = "<input type=\"file\" name=\"#{element_id}\""
		to_s << other_attributes_string
		to_s << ">"
		to_s
	end
end


# CKPartsMaker is a Ruby module to deal with a component as parts of web page.
# A component which includes this module isn't displayed even if CGIKit recieves
# requests to show it as the target component ( CKApplication#target ).
#
# A name of parts component is recommended to have "Parts" at
# the end of the name to distinguish it from page component.
#
# == Usage
# Subclass of CKComponent includes CKPartsMaker.
#
#  class AnyParts < CKComponent
#    include CKPartsMaker
#    ...
#  end
#
module CKPartsMaker
	# When CGIKit recieves requests to show component parts,
	# CGIKit shows the page specified by this attribute.
	# A main page of an application is displayed when the value is not defined.
	attr_accessor :substitute_page
end


