/*******************************************************************************************************************************************
 cmetaclass.h

 Class metaclass description definition. This metaclass description allows dynamic analyse of instances types at runtime.

 There are 3 defined types of metaclasses :
 - generic : the metaclass describes an abstract class,
 - dynamic : the metaclass describes a non abstract class and is able to instanciate it with its default constructor,
 - capsule : the metaclass describes a non abstract class but is not able to instanciate it because either the described class does not
	     define a default constructor either the programmer's choice was not to give the access to.

 A metaclass handles the followings :
 - the base definition the described class is derived from (NULL for the absolute one),
 - the metaclass type (dynamic, generic or capsule),
 - the described class name (its type as a CString value),
 - the described class tag identification (should not equal 0L),
 - the default described class size (sizeof class),
 - "a default constructor pointer" of the described class (only if it is a dynamic one).

 Static attributes of the CMetaClass definition are defined as follow :
 - all declared metaclasses list access for the binary and its potential modules,
 - a function that searches for a metaclass from a given class name,
 - a function that searches for a metaclass from a given class tag identification,
 - a function that checks a derivation status between two metaclasses,
 - a function that searches for derived metaclass described classes definition,
 - a function that returns the declared metaclasses list hierarchy.

 The CMetaClass constructor and destructor handle the insertion and deletion of the metaclasses instances into the static list of the 
 binary defined metaclasses. There is no initialization process need in order to use the metaclass facility. All of them are declared as
 static constants and are also allocated before the binary execution entry point is called.

 Some macro are defined in this file to facilitate the metaclasses and class association :
 "myclass.h"
 	class CMyClass : public CClass
 	{
 		public :
			CMyClass ();
			SECTION_xxx_METACLASS;
 	};
 	DECLARE_xxx_METACLASS ('mycs', CMyClass, CClass);
 "myclass.c"
 	RESOLVE_xxx_METACLASS (CMyClass)
 	CMyClass::CMyClass () : CClass () { }


 	CMyClass *MyClass = new CMyClass ();
 	printf ("%s\n", metaclass_cast(MyClass) -> ClassName.Get());
	printf ("%s\n", __metaclass(CMyClass) -> ClassName.Get());
 	printf ("%d\n", classtag_cast(MyClass));
 	printf ("%d\n", __classtag(CMyClass));
	for (size_t i=0; i<METACLASSESLENGTH; i++)
		printf ("%s\n", METACLASSES(i) -> ClassName.Get());
	...
*******************************************************************************************************************************************/

#ifndef __CMETACLASS_H__
#define __CMETACLASS_H__

#include "cstring.h"

// the defined metaclass, the described absolute abstract class
class CClass;
class CMetaClass;

//-------------------------------------------------------------------------------------------------------------------------------------------
// metaclasses types definition
//-------------------------------------------------------------------------------------------------------------------------------------------
typedef enum
{
	METACLASS_GENERIC=0,	// generic metaclass
	METACLASS_DYNAMIC,	// dynamic metaclass
	METACLASS_CAPSULE	// capsule metaclass
} TMetaClass;

//-------------------------------------------------------------------------------------------------------------------------------------------
// metaclasses list definition
//-------------------------------------------------------------------------------------------------------------------------------------------
typedef NServices::TBuffer <const CMetaClass *> CMetaClasses;
typedef NServices::TNode   <CMetaClass        > CMetaClassNode;

//-------------------------------------------------------------------------------------------------------------------------------------------
// CMetaClass class
//-------------------------------------------------------------------------------------------------------------------------------------------
class CMetaClass
{
	// metaclass attributes
	public :

		// definition of the described class is derived from
		const CMetaClass *		BaseMetaClass;

		// metaclass type
		const TMetaClass		MetaClassType;

		// class tag identification
		const UInt32			ClassTag;
		
		// class name
		const CString			ClassName;

		// class default size
		const UInt16			ClassSize;

		// default constructor pointer
		CClass *			(*ClassInstanciate)	();

	// static attributes
	public :

		// the binary declared metaclasses list access
		static CMetaClasses		MetaClasses;

		// metaclass search for specified class name or class tag
		static const CMetaClass *	GetMetaClass		(const CString &);
		static const CMetaClass *	GetMetaClass		(const UInt32);

		// derivation test
		static bool			MetaClassIs		(const CMetaClass *inBase, const CMetaClass *inCandidate);

		// is there any derived metaclassed definition ?
		static bool			IsFinal			(const CMetaClass *);

		// metaclasses hierarchy access
		static CMetaClassNode		GetMetaClassTree	();
	
	// instanciation section
	public :

		// metaclass constructor
		CMetaClass			(const CMetaClass * =NULL, const TMetaClass =METACLASS_GENERIC, const UInt32 =0L, 
						 const CString & =CString(), const UInt16 =0, CClass * (*) () =NULL);

		// metaclass destructor
		virtual ~CMetaClass		();
};

//-------------------------------------------------------------------------------------------------------------------------------------------
// class tag and metaclass access macros
//-------------------------------------------------------------------------------------------------------------------------------------------

// class tag of specified class type
#define __classtag(classname) classname##Tag

// metaclass of specified class type
#define __metaclass(classname) (&classname##MetaClass)

// metaclass of specified class type as a list of metaclasses
#define __metaclasses(classname) CMetaClasses (1, __metaclass(classname))

// metaclass of specified instance
#define metaclass_cast(instance) (instance) -> GetMetaClass()

// class tag of specified instance
#define classtag_cast(instance) metaclass_cast(instance) -> ClassTag

// metaclasses list access commodity
#define METACLASSESLENGTH    CMetaClass::MetaClasses.GetLength()
#define METACLASSES(index) (*CMetaClass::MetaClasses[index])

//-------------------------------------------------------------------------------------------------------------------------------------------
// metaclasses class section prototype macros
//-------------------------------------------------------------------------------------------------------------------------------------------

#define SECTION_GENERIC_METACLASS public : virtual const CMetaClass * GetMetaClass () const
#define SECTION_DYNAMIC_METACLASS SECTION_GENERIC_METACLASS; static CClass * Instanciate ()
#define SECTION_CAPSULE_METACLASS SECTION_GENERIC_METACLASS

//-------------------------------------------------------------------------------------------------------------------------------------------
// metaclasses and class tag declarations macros
//-------------------------------------------------------------------------------------------------------------------------------------------

#define DECLARE_GENERIC_METACLASS(classtag,classname,baseclassname) 						\
	const static UInt32 __classtag(classname) = (UInt32) classtag;						\
	const static CMetaClass classname##MetaClass (CMetaClass::GetMetaClass(CString(""#baseclassname)), 	\
						      METACLASS_GENERIC,					\
						      classtag,							\
						      CString(""#classname),					\
						      sizeof(classname),					\
						      NULL)

#define DECLARE_DYNAMIC_METACLASS(classtag,classname,baseclassname)						\
	const static UInt32 __classtag(classname) = (UInt32) classtag;						\
	const static CMetaClass classname##MetaClass (CMetaClass::GetMetaClass(CString(""#baseclassname)), 	\
						      METACLASS_DYNAMIC,					\
						      classtag,							\
						      CString(""#classname),					\
						      sizeof(classname),					\
						      classname::Instanciate)
	
#define DECLARE_CAPSULE_METACLASS(classtag,classname,baseclassname)						\
	const static UInt32 __classtag(classname) = (UInt32) classtag;						\
	const static CMetaClass classname##MetaClass (CMetaClass::GetMetaClass(CString(""#baseclassname)), 	\
						      METACLASS_CAPSULE,					\
						      classtag,							\
						      CString(""#classname),					\
						      sizeof(classname),					\
						      NULL)

//-------------------------------------------------------------------------------------------------------------------------------------------
// metaclass class code resolution macros
//-------------------------------------------------------------------------------------------------------------------------------------------

#define RESOLVE_GENERIC_METACLASS(classname) const CMetaClass * classname::GetMetaClass () const { return __metaclass(classname); }
#define RESOLVE_DYNAMIC_METACLASS(classname) RESOLVE_GENERIC_METACLASS(classname) \
					     CClass * classname::Instanciate () { return new classname(); }
#define RESOLVE_CAPSULE_METACLASS(classname) RESOLVE_GENERIC_METACLASS(classname)

#endif
