/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



-----------------------------------------------------------------------
FILENAME:                eval_speed.c
LIBRARY:                 libgpiv
EXTERNAL FUNCTIONS:      gpiv_piv_gridgen
                         gpiv_piv_gridadapt
                         gpiv_piv_speedup_cmpr
			 gpiv_piv_compress
			 gpiv_piv_decompress
 
LAST MODIFICATION DATE:  
$Id: eval_speed.c,v 1.9 2006/01/31 13:30:13 gerber Exp $
--------------------------------------------------------------------- */

/* #include <stdio.h> */
/* #include <math.h> */
/* #include <assert.h> */
/* #include <rfftw.h> */

#include <gpiv.h>

#define EXPANSION_FACT 2
#define SMOOTH_WINDOW 2
#define NMIN_TO_INTERPOLATE 2

enum Position {
    LOWER,
    NEXT_LOWER,
    HIGHER,
    NEXT_HIGHER
};

enum HorizontalPosition {
    WEST,
    WEST_WEST,
    EAST,
    EAST_EAST
};

enum VerticalPosition {
    NORTH,
    NORTH_NORTH,
    SOUTH, 
    SOUTH_SOUTH, 
};

/*
 * Local functions
 */
static void
nearest_point(gint i,
              gfloat x, 
              gfloat point_x, 
              gfloat *min, 
              gint *index, 
              gboolean *exist
              )
/*-----------------------------------------------------------------------------
 * Test if current point_x is nearest from x
 */
{
    gfloat dif = abs(x - point_x);

    if (dif < *min) {
        *exist = TRUE;
        *min = dif;
        *index = i;
    }

}




static gboolean
nearest_index(enum Position pos,
              gint vlength,
              gfloat *src_point, 
              gfloat dest_point,
              gint *index
              )
/*-----------------------------------------------------------------------------
 * Search nearest index from piv_data belonging to point (x, y)
 * in horizontal direction
 * pos_x/y index left/right, top/bottom of point
 */
{
    gint i, j;
    gfloat min = 10.0e4;
    gboolean exist = FALSE;

    *index = 0;
/* g_message("nearest_index::  dest_point=%f", dest_point); */
    for (i = 0; i < vlength; i++) {
/* g_message("nearest_index::  src_point[%d]=%f", i, src_point[i]); */

        if (pos ==  LOWER
           && src_point[i] <= dest_point) {
            nearest_point(i, dest_point, src_point[i], &min, 
                          index, &exist);
/* g_message("nearest_index::  index_l=%d", *index); */

        } else if (pos == NEXT_LOWER
                   &&  i > 0
                  && src_point[i - 1] < dest_point) {
 
            nearest_point(i - 1, dest_point, src_point[i - 1], &min, 
                          index, &exist);
/* g_message("nearest_index::  src_point[%d]=%f index_ll=%d", */
/*           i-1, src_point[i - 1], *index); */

        } else if (pos == HIGHER
                   && src_point[i] > dest_point) {
            nearest_point(i, dest_point, src_point[i], &min, index, &exist);
/* g_message("nearest_index::  index_h=%d", *index); */
            
        } else if (pos ==  NEXT_HIGHER
                   && i <  vlength - 1 
                   && src_point[i + 1] > dest_point) {
            nearest_point(i + 1, dest_point, src_point[i + 1], 
                          &min, 
                          index, &exist);
/* g_message("nearest_index::  src_point[%d]=%f index_hh=%d", */
/*           i+1, src_point[i + 1], *index); */
            
        }
        
    }

    return exist;
}



static gdouble
bilinear_interpol(gdouble alpha_hor,
                  gdouble alpha_vert,
                  gdouble src_dx_nw,
                  gdouble src_dx_ne,
                  gdouble src_dx_sw,
                  gdouble src_dx_se
                  )
/*-----------------------------------------------------------------------------
 * Bilinear interpolation of src_dx_*
 * _ne: NORTH_EAST
 * _nw: NORTH_WEST
 * _se: SOUTH_EAST
 * _SW: SOUTH_WEST
 */
{
    gdouble dx, dx_n, dx_s;
    dx_n = (1.0 - alpha_hor) * src_dx_nw + alpha_hor * src_dx_ne;
    dx_s = (1.0 - alpha_hor) * src_dx_sw + alpha_hor * src_dx_se;
    dx =  (1.0 - alpha_vert) * dx_n + alpha_vert * dx_s;

/*     g_message("bilinear_interpol:: alpha_hor=%f alpha_vert=%f, dx_n = %f dx_s = %f => dx = %f",  */
/*               alpha_hor, alpha_vert, dx_n, dx_s, dx); */
    return dx;
}



void *
intpol_facts(gfloat *src_point, 
             gfloat *dest_point, 
             gint src_vlength,
             gint dest_vlength,
             gint *index_l,
             gint *index_h,
             gint *index_ll,
             gint *index_hh,
             double *alpha
             )
/*-----------------------------------------------------------------------------
 * calculates normalized interpolation factors for piv_data_src
 * Think of:
 *          _l (LOWER) is used for _w (WEST) or _n (NORTH),
 *          _h (HIGHER) is used for _e (EAST) or _s (SOUTH)
 *          _ll (NEXT_LOWER) is used for _ww (WEST_WEST) or _nn (NORTH_NORTH),
 *          _hh (NEXT_HIGHER) is used for _ee (EAST_EAST) or _ss (SOUTH_SOUTH)
 */
{
    gboolean *exist_l, *exist_h, *exist_ll, *exist_hh;
    double *dist_l, *dist_h, *dist_ll, *dist_hh;
    enum Position pos;
    gint i;

    exist_l = gpiv_gbolvector(dest_vlength);
    exist_h = gpiv_gbolvector(dest_vlength);
    exist_ll = gpiv_gbolvector(dest_vlength);
    exist_hh = gpiv_gbolvector(dest_vlength);

    dist_l = gpiv_dvector(dest_vlength);
    dist_h = gpiv_dvector(dest_vlength);
    dist_ll = gpiv_dvector(dest_vlength);
    dist_hh = gpiv_dvector(dest_vlength);

/*     g_warning("intpol_facts:: 1, src_vlengthth=%d dest_vlength=%d",  */
/*               src_vlength, dest_vlength); */
/*
 * Searching adjacent and next to adjacent points of dest_point in src_point
 * data array
 */
    for (i = 0; i < dest_vlength; i++) {
/* g_message("intpol_facts::   (next) adjacent point for dest_point[%d]=%f",  */
/*           i, dest_point[i]); */
      pos = LOWER;
        exist_l[i] = FALSE;
        if (exist_l[i] = 
            nearest_index(pos, 
                          src_vlength, 
                          src_point,
                          dest_point[i],
                          &index_l[i])) {
/*             g_message("intpol_facts:: index_l[%d]=%d", i, index_l[i]); */
            dist_l[i] = dest_point[i] - src_point[index_l[i]];
        } else {
/*
 * To be used for extrapolation in negative direction 
 * by applying higher and next to higher points of src data
 */
            pos = NEXT_HIGHER;
            exist_hh[i] = FALSE;
            if (exist_hh[i] = 
                nearest_index(pos, 
                              src_vlength, 
                              src_point,
                              dest_point[i],
                              &index_hh[i])) {
/*                 g_message("intpol_facts:: index_hh[%d]=%d", i, index_hh[i]); */
                dist_hh[i] = dest_point[i] - src_point[index_hh[i]];
            }
        }
        


        pos = HIGHER;
        exist_h[i] = FALSE;
        if (exist_h[i] = 
            nearest_index(pos, 
                          src_vlength, 
                          src_point,
                          dest_point[i],
                          &index_h[i])) {
/*                 g_message("intpol_facts:: index_h[%d]=%d", i, index_h[i]); */
            dist_h[i] = dest_point[i] - src_point[index_h[i]];
        } else {

/*
 * To be used for extrapolation in positive direction 
 * by applying lower and next to lower points of src data
 */
            pos = NEXT_LOWER;
            exist_ll[i] = FALSE;
            index_ll[i] = 0;
            if (exist_ll[i] = 
                nearest_index(pos, 
                              src_vlength, 
                              src_point,
                              dest_point[i],
                              &index_ll[i])) {
/*                 g_message("intpol_facts:: index_ll[%d]=%d, index_l[%d]=%d",  */
/*                           i, index_ll[i], i, index_l[i]); */
                dist_ll[i] = dest_point[i] - src_point[index_ll[i]];
            }
        }

/*
 * calculating of weight factors for inter- or extrapolation
 */

/* g_message("intpol_facts:: weight factors for dest_point[%d]=%f",  */
/*           i, dest_point[i]); */
        if (exist_l[i] && exist_h[i]) {
/* g_message("intpol_facts:: index_l[%d]=%d index_h[%d]=%d src_point_l=%f src_point_h=%f",  */
/*           i, index_l[i], i, index_h[i],  */
/*           src_point[index_l[i]], src_point[index_h[i]]); */
/*
 * Inner point: bilinear interpolation
 */
            if (src_point[index_l[i]] != src_point[index_h[i]]) {
                alpha[i] = dist_l[i] / 
                    (src_point[index_h[i]] - src_point[index_l[i]]);
            } else {
                alpha[i] = 1.0;
            }
            

        } else if (exist_l[i] && exist_ll[i] && !exist_h[i]) {
/* 
 * extrapolation from two lower values
 */
/* g_message("intpol_facts:: index_l[%d]=%d index_ll[%d]=%d src_point_l=%f src_point_ll=%f",  */
/*           i, index_l[i], i, index_ll[i],  */
/*           src_point[index_l[i]], src_point[index_ll[i]]); */

            if (src_point[index_ll[i]] != src_point[index_l[i]]) {
                alpha[i] = dist_ll[i] / 
                    (src_point[index_l[i]] - src_point[index_ll[i]]);
                index_h[i] = index_l[i];
                index_l[i] = index_ll[i];
           } else {
                alpha[i] = 1.0;
           }


        } else if (!exist_l[i] && exist_h[i] && exist_hh[i]) {
/* 
 * extrapolation from two higher values
 */
/* g_message("intpol_facts:: index_h=%d index_hh=%d src_point_h=%f src_point_hh=%f",  */
/*           index_h[i], index_hh[i], src_point[index_h[i]],  */
/*           src_point[index_hh[i]]); */

            if (src_point[index_hh[i]] != src_point[index_h[i]]) {
                alpha[i] = dist_h[i] / 
                (src_point[index_hh[i]] - src_point[index_h[i]]);
                index_l[i] = index_h[i];
                index_h[i] = index_hh[i];
            } else {
                alpha[i] = 1.0;
            }
            

        } else {
            alpha[i] = 1.0;
        }
    }


    gpiv_free_gbolvector(exist_l);
    gpiv_free_gbolvector(exist_h);
    gpiv_free_gbolvector(exist_ll);
    gpiv_free_gbolvector(exist_hh);
    
    gpiv_free_dvector(dist_l);
    gpiv_free_dvector(dist_h);
    gpiv_free_dvector(dist_ll);
    gpiv_free_dvector(dist_hh);
 
}



char *
gpiv_piv_shift_grid(GpivPivData gpd_src,
                    GpivPivData *gpd_out
                    )
/*-----------------------------------------------------------------------------
 * shifts the knots of a 2-dimensional grid containing PIV data for improved 
 * (bi-linear) interpolation
 *
 * See: T. Blu, P. Thevenaz, "Linear Interpolation Revitalized",
 * IEEE Trans. in Image Processing, vol13, no 5, May 2004
 *
-----------------------------------------------------------------------------*/
{
#define SHIFT 0.2

    char *err_msg = NULL;
    GpivPivData h_gpd, v_gpd;
    gfloat fact1 = -SHIFT / (1.0 - SHIFT);
    gfloat fact2 = 1.0 / (1 - SHIFT);
    gfloat **cx, **cy;
    gfloat delta = 0.0;
    gint i, j;

    delta = gpd_src.point_x[0][1] - gpd_src.point_x[0][0];

/*
 * Shift in horizontal (column-wise) direction
 */
    h_gpd.nx = gpd_src.nx;
    h_gpd.ny = gpd_src.ny;
    gpiv_null_pivdata(&h_gpd);
    gpiv_alloc_pivdata(&h_gpd);

    for (i = 0; i < gpd_src.ny; i++) {
        for (j = 0; j < gpd_src.nx; j++) {
/*
 * Shift the knot (sample points)
 */
            h_gpd.point_x[i][j] = gpd_src.point_x[i][j] + SHIFT * delta;
            h_gpd.point_y[i][j] = gpd_src.point_y[i][j];
            if (j == 0) {
                h_gpd.dx[i][j] = gpd_src.dx[i][j];
                h_gpd.dy[i][j] = gpd_src.dy[i][j];
            } else {
/*
 * Calculate value at shifted knot
 */
                h_gpd.dx[i][j] = fact1 * h_gpd.dx[i][j-1] + fact2 * 
                    gpd_src.dx[i][j];
                h_gpd.dy[i][j] = fact1 * h_gpd.dy[i][j-1] + fact2 * 
                    gpd_src.dy[i][j];
            }
        }
    }
    

/*
 * Shift in vertical (row-wise) direction by using the horizontal shifted nodes
 */
    v_gpd.nx = gpd_src.nx;
    v_gpd.ny = gpd_src.ny;
    gpiv_null_pivdata(&v_gpd);
    gpiv_alloc_pivdata(&v_gpd);

    for (i = 0; i < gpd_src.ny; i++) {
        for (j = 0; j < gpd_src.nx; j++) {
            v_gpd.point_x[i][j] = h_gpd.point_x[i][j];
            v_gpd.point_y[i][j] = h_gpd.point_y[i][j] + SHIFT * delta;
            if (i == 0) {
                v_gpd.dx[i][j] = h_gpd.dx[i][j];
                v_gpd.dy[i][j] = h_gpd.dy[i][j];
            } else {
                v_gpd.dx[i][j] = fact1 * v_gpd.dx[i-1][j] + fact2 * 
                    h_gpd.dx[i][j];
                v_gpd.dy[i][j] = fact1 * v_gpd.dy[i-1][j] + fact2 * 
                    h_gpd.dy[i][j];
            }
        }
    }
    

    gpiv_cp_pivdata(&v_gpd, gpd_out);

    gpiv_free_pivdata(&h_gpd);
    gpiv_free_pivdata(&v_gpd);
    return err_msg;
#undef SHIFT
}



static void
dxdy_at_new_grid_block(GpivPivData piv_data_src, 
                             GpivPivData *piv_data_dest,
                             gint expansion_factor,
                             gint smooth_window
                            )
/*-----------------------------------------------------------------------------
 * Calculates displacements from old to new grid, that has been expanded by
 * factor 2 and avarages with smoothing window. Works only correct if all neighbours 
 * at equal distances
 */
{
    int i, j, k, l, ef = expansion_factor, sw = smooth_window;
    int count = 0;
    GpivPivData pd;

    pd.nx = piv_data_dest->nx;
    pd.ny = piv_data_dest->ny;
    gpiv_alloc_pivdata(&pd);

/*
 * Copy blocks of 2x2 input data to pd 
 */
    for (i = 0; i < piv_data_src.ny; i++) {
        for (j = 0; j < piv_data_src.nx; j++) {
            for (k = 0; k < 2; k++) {
                if (ef * i+k < pd.ny) {
                    for (l = 0; l < 2; l++) {
                        if (ef * j+l < pd.nx) {
                            pd.dx[ef * i+k][ef * j+l] = piv_data_src.dx[i][j];
                            pd.dy[ef * i+k][ef * j+l] = piv_data_src.dy[i][j];
                        }
                    }
                }
            }   
        }
    }

/*
 * smooth the data
 */
    for (i = 0; i < piv_data_src.ny; i++) {
        for (j = 0; j < piv_data_src.nx; j++) {
            count = 0;
            for (k = -sw + 1; k < sw; k++) {
                if (i + k > 0 && i + k < pd.ny) {
                    for (l = -sw + 1; l < sw; l++) {
                        if (j + l > 0 && j + l < pd.ny) {
                            count++;
                            piv_data_dest->dx[i][j] += pd.dx[i+k][j+l];
                            piv_data_dest->dy[i][j] += pd.dy[i+k][j+l];
                        }
                    }
                }
            }
            piv_data_dest->dx[i][j] = piv_data_dest->dx[i][j] / (float)count;
            piv_data_dest->dy[i][j] = piv_data_dest->dx[i][j] / (float)count;
        }
    }

    gpiv_free_pivdata(&pd);
}



static void
cmpr_img(guint16 **img, 
         guint16 **img_cmpr, 
         GpivImagePar *image_par
         )
/*-----------------------------------------------------------------------------
 * Compressing **image** by taking mean from (-CMPR_FACT+1)/2 to CMPR_FACT/2 
 * to +CMPR_FACT
 */
{
     int row, column, i, j, index_i = 0, index_j = 0, count = 0;
     int index_i_max = 0, index_j_max = 0;
     float img_sum = 0.;



    for (row = 0; row < image_par->nrows; row = row + CMPR_FACT) {
/*        fprintf(stderr, "row=%d ", row); */
	for (column = 0; column < image_par->ncolumns; column = column + CMPR_FACT) {
/* 	  fprintf(stderr, "column=%d ", column); */
	    for (i = row; i < row + CMPR_FACT; i++) {
/* 	      fprintf(stderr, "i=%d ", i); */
		if (i >= 0 && i < image_par->nrows) {
		    for (j = column; j < column + CMPR_FACT ; j++) {
/* 		      fprintf(stderr, "j=%d ", j); */
			if (j >= 0 && j < image_par->ncolumns) {
			    img_sum += (float) img[i][j];
			    count++;
/* 				fprintf(stderr, "count=%d img=%d img_sum=%f",  */
/* 					count, (int)img[i][j], img_sum); */

			}
		    }
		}
	    }
	    img_cmpr[index_i][index_j] = (int) img_sum / count;
/* 	    img_cmpr[index_i][index_j] = img[row][column]; */
	    count = 0;
	    img_sum = 0.;
	    index_j++;
	    if (index_j > index_j_max) index_j_max = index_j; 
	}
/* 	fprintf(stderr, "\n"); */
	index_j = 0;
	index_i++;
	    if (index_i > index_i_max) index_i_max = index_i; 
    }

}



/*
 * Public functions
 */

char *
gpiv_piv_dxdy_at_new_grid(GpivPivData piv_data_src,
                          GpivPivData *piv_data_dest
                          )
/*-----------------------------------------------------------------------------
 * calculates dx, dy of piv_data_dest from piv_data_src
 * by bi-linear interpolation of inner points with shifted knots
 * or extrapolation of outer lying points

    dist_:      distance
    _n:         NORTH
    _s:         SOUTH
    _e:         EAST
    _w:         WEST
    _nn:        at NORTH of NORTH, etc.
-----------------------------------------------------------------------------*/
{
    char c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    char *err_msg = NULL;

    gint *index_n, *index_s, *index_e, *index_w;
    gint *index_nn, *index_ss, *index_ee, *index_ww;
    gfloat *src_point_x = NULL, *dest_point_x = NULL;
    gfloat *src_point_y = NULL, *dest_point_y = NULL;
    double *alpha_hor, *alpha_vert;

    double epsi = 0.01;
    enum VerticalPosition vert_pos;
    enum HorizontalPosition hor_pos;
    gint i = 0, j = 0;

    GpivPivData gpd;
/*
 * shift the knots of the grid for higher accuracies
 * REPAIRED BUGFIX: calling gpiv_piv_shift_grid(&piv_data_SRC, &piv_data_SRC)
 */
    gpd.nx = piv_data_src.nx;
    gpd.ny = piv_data_src.ny;
    gpiv_null_pivdata(&gpd);
    gpiv_alloc_pivdata(&gpd);
    gpiv_piv_shift_grid(piv_data_src, &gpd);

/*     g_warning("gpiv_piv_dxdy_at_new_grid:: 0"); */
    index_n = gpiv_ivector(piv_data_dest->ny);
    index_s = gpiv_ivector(piv_data_dest->ny);
    index_e = gpiv_ivector(piv_data_dest->nx);
    index_w = gpiv_ivector(piv_data_dest->nx);
    index_nn = gpiv_ivector(piv_data_dest->ny);
    index_ss = gpiv_ivector(piv_data_dest->ny);
    index_ee = gpiv_ivector(piv_data_dest->nx);
    index_ww = gpiv_ivector(piv_data_dest->nx);

    alpha_vert = gpiv_dvector(piv_data_dest->ny);
    alpha_hor = gpiv_dvector(piv_data_dest->nx);

    src_point_x = gpiv_vector(gpd.nx);
    src_point_y = gpiv_vector(gpd.ny);
    dest_point_x = gpiv_vector(piv_data_dest->nx);
    dest_point_y = gpiv_vector(piv_data_dest->ny);

/*
 * Calculate interpolation factors
 * in Horizontal direction
 */
/*     g_warning("gpiv_piv_dxdy_at_new_grid:: 1"); */
    if (gpd.nx >= NMIN_TO_INTERPOLATE) {
        for (i = 0, j = 0; j < gpd.nx; j++) {
            src_point_x[j] = gpd.point_x[i][j];
        }

        for (i = 0, j = 0; j < piv_data_dest->nx; j++) {
            dest_point_x[j] = piv_data_dest->point_x[i][j];
        }

        intpol_facts(src_point_x, 
                     dest_point_x, 
                     gpd.nx,
                     piv_data_dest->nx,
                     index_w,
                     index_e,
                     index_ww,
                     index_ee,
                     alpha_hor);
    } else {
        err_msg = "GPIV_PIV_DXDY_AT_NEW_GRID: Not enough points in horizontal direction";
        return err_msg;
    }
/*     g_warning("gpiv_piv_dxdy_at_new_grid:: 2"); */

/*
 * Calculate interpolation factors
 * in Vertical direction
 */
    if (gpd.ny >= NMIN_TO_INTERPOLATE) {
        for (i = 0, j = 0; i < gpd.ny; i++) {
            src_point_y[i] = gpd.point_y[i][j];
        }

        for (i = 0, j = 0; i < piv_data_dest->ny; i++) {
            dest_point_y[i] = piv_data_dest->point_y[i][j];
        }

        intpol_facts(src_point_y, 
                     dest_point_y, 
                     gpd.ny,
                     piv_data_dest->ny,
                     index_n,
                     index_s,
                     index_nn,
                     index_ss,
                     alpha_vert);
    } else {
        err_msg = "GPIV_PIV_DXDY_AT_NEW_GRID: Not enough points in horizontal direction";
        return err_msg;
    }
/*     g_warning("gpiv_piv_dxdy_at_new_grid:: 3"); */

/*
 * Calculate new displacements by bi-lineair interpolation
 */
    for (i = 0; i < piv_data_dest->ny; i++) {
        for (j = 0; j < piv_data_dest->nx; j++) {
            piv_data_dest->dx[i][j] = bilinear_interpol
                (alpha_hor[j],
                 alpha_vert[i],
                 gpd.dx[index_n[i]][index_w[j]],
                 gpd.dx[index_n[i]][index_e[j]],
                 gpd.dx[index_s[i]][index_w[j]],
                 gpd.dx[index_s[i]][index_e[j]]);
/* g_message("gpiv_piv_dxdy_at_new_grid:: alpha_hor[%d]=%f alpha_vert[%d]=%f dx_nw=%f dx_ne=%f dx_sw=%f dx_se=%f => dx=%f", */
/*               j, alpha_hor[j], i, alpha_vert[i], */
/*               gpd.dx[index_n[i]][index_w[j]], */
/*               gpd.dx[index_n[i]][index_e[j]], */
/*               gpd.dx[index_s[i]][index_w[j]], */
/*               gpd.dx[index_s[i]][index_e[j]], */
/*               piv_data_dest->dx[i][j] */
/*               ); */

            piv_data_dest->dy[i][j] = bilinear_interpol
                              (alpha_hor[j],
                               alpha_vert[i],
                               gpd.dy[index_n[i]][index_w[j]],
                               gpd.dy[index_n[i]][index_e[j]],
                               gpd.dy[index_s[i]][index_w[j]],
                               gpd.dy[index_s[i]][index_e[j]]);
/*             g_message("gpiv_piv_dxdy_at_new_grid:: alpha_hor[%d]=%f alpha_vert[%d]=%f dy_nw=%f dy_ne=%f dy_sw=%f dy_se=%f => dy=%f", */
/*               j, alpha_hor[j], i, alpha_vert[i], */
/*               gpd.dy[index_n[i]][index_w[j]], */
/*               gpd.dy[index_n[i]][index_e[j]], */
/*               gpd.dy[index_s[i]][index_w[j]], */
/*               gpd.dy[index_s[i]][index_e[j]], */
/*               piv_data_dest->dy[i][j] */
/*               ); */
        }
    }
    

    gpiv_free_ivector(index_n);
    gpiv_free_ivector(index_s);
    gpiv_free_ivector(index_e);
    gpiv_free_ivector(index_w);
    gpiv_free_ivector(index_nn);
    gpiv_free_ivector(index_ss);
    gpiv_free_ivector(index_ee);
    gpiv_free_ivector(index_ww);

    gpiv_free_dvector(alpha_vert);
    gpiv_free_dvector(alpha_hor);

    gpiv_free_vector(src_point_x);
    gpiv_free_vector(src_point_y);
    gpiv_free_vector(dest_point_x);
    gpiv_free_vector(dest_point_y);

    gpiv_free_pivdata(&gpd);

/*     g_warning("gpiv_piv_dxdy_at_new_grid:: 4/4"); */

    return err_msg;
}



char *
gpiv_piv_gridgen(GpivPivData *piv_data,
                 GpivImagePar  image_par,
                 GpivEvalPar piv_eval_par
                 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Generates grid by Calculating the positions of interrogation areas
 *     Substitutes gpiv_piv_select_int_point
 *
 * INPUTS:
 *     @image_par:      structure of image parameters
 *     @piv_eval_par:   structure of piv evaluation parameters
 *
 * OUTPUTS:
 *     @out_data:       output piv data from image analysis
 *
 * RETURNS:
 *     %char * NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int row, column, row_1, column_1, i, j;
    int row_max, row_min, column_max, column_min;
    int nx = piv_data->nx, ny = piv_data->ny;

    int ncolumns = image_par.ncolumns;
    int nrows = image_par.nrows;

    int int_geo = piv_eval_par.int_geo;
    int row_start = piv_eval_par.row_start;
    int row_end = piv_eval_par.row_end;
    int col_start = piv_eval_par.col_start;
    int col_end= piv_eval_par.col_end;
    int int_line_col = piv_eval_par.int_line_col;
    int int_line_col_start = piv_eval_par.int_line_col_start;
    int int_line_col_end = piv_eval_par.int_line_col_end;
    int int_line_row = piv_eval_par.int_line_row;
    int int_line_row_start = piv_eval_par.int_line_row_start;
    int int_line_row_end = piv_eval_par.int_line_row_end;
    int int_point_col = piv_eval_par.int_point_col;
    int int_point_row = piv_eval_par.int_point_row;
    int int_size_1 = piv_eval_par.int_size_1;
    int int_size_2 = piv_eval_par.int_size_2;
    int int_shift = piv_eval_par.int_shift;
    int pre_shift_row = piv_eval_par.pre_shift_row;
    int pre_shift_col = piv_eval_par.pre_shift_col;
				 
/*     float **point_x = piv_data->point_x, **point_y = piv_data->point_y; */

    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);

    row_min = gpiv_min(-int_size_1 / 2 + 1, 
                       pre_shift_row - int_size_2 / 2 + 1);
    column_min = gpiv_max(-int_size_1 / 2 + 1, 
                          pre_shift_col - int_size_2 / 2 + 1);
    row_max = gpiv_max(int_size_1 / 2, pre_shift_row + int_size_2 / 2);
    column_max = gpiv_max(int_size_1 / 2, pre_shift_col + int_size_2 / 2);


/*
 * Creates centre points for one single interrogation area
 */
    if (int_geo == GPIV_POINT) {
        piv_data->point_y[0][0] = int_point_row;
        piv_data->point_x[0][0] = int_point_col;
            

/*
 * Creates centre points for one single row
 */
    } else if (int_geo == GPIV_LINE_R) {
        if ((int_size_1 - int_size_2) / 2 + pre_shift_col < 0) {
            column_1 = int_line_col_start - 
                ((int_size_1 - int_size_2) / 2 +
                 pre_shift_col) + int_size_1 / 2 - 1;
        } else {
            column_1 = int_line_col_start + int_size_1 / 2 - 1;
        }

        for (i = 0, j = 0, row = int_line_row, column = column_1;
             j < piv_data->nx; j++, column += int_shift) {
            piv_data->point_y[i][j] = row;
            piv_data->point_x[i][j] = column;
        }


/*
 * Creates centre points for one single column
 */
    } else  if (int_geo == GPIV_LINE_C) {
        if ((int_size_1 - int_size_2) / 2 + pre_shift_row < 0) {
            row_1 = int_line_row_start - 
                ((int_size_1 - int_size_2) / 2 +
                     pre_shift_row) + int_size_1 / 2 - 1;
        } else {
            row_1 = int_line_row_start + int_size_1 / 2 - 1;
        }

        for (i = 0, j = 0, column = int_line_col, row = row_1; 
             i < piv_data->ny; i++, row += int_shift) {
            piv_data->point_y[i][j] = row;
            piv_data->point_x[i][j] = column;
        }


/*
 * Creates an array of  centre points of the Interrrogation Area's: 
 * int_ar_x and int_ar_y within the defined region of the image
 */
    } else if (int_geo == GPIV_AOI) {
	if ((int_size_1 - int_size_2) / 2 + pre_shift_row < 0) {
	    row_1 =
		row_start - ((int_size_1 - int_size_2) / 2 +
			     pre_shift_row) + int_size_1 / 2 - 1;
	} else {
	    row_1 = row_start + int_size_1 / 2 - 1;
	}

	if ((int_size_1 - int_size_2) / 2 + pre_shift_col < 0) {
	    column_1 =
		col_start - ((int_size_1 - int_size_2) / 2 +
			     pre_shift_col) + int_size_1 / 2 - 1;
	} else {
	    column_1 = col_start + int_size_1 / 2 - 1;
	}

	for (i = 0, row = row_1; i < ny; i++, row += int_shift) {
	    for (j = 0, column = column_1; j < nx;
		 j++, column += int_shift) {
		piv_data->point_y[i][j] = row;
		piv_data->point_x[i][j] = column;
	    }
	}
    } else {
        err_msg = "GPIV_PIV_GRIDGEN: should not arrive here";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }


    return err_msg;
}



void 
gpiv_piv_gridadapt(GpivImagePar *image_par, 
                   GpivEvalPar piv_eval_par_src,
                   GpivEvalPar *piv_eval_par_dest,
                   GpivPivData *piv_data,
                   int sweep, 
                   gboolean *grid_last
                   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Adjust grid nodes if zero_off or adaptive interrogation 
 *     area has been used. This is performed by modifying int_shift equal 
 *     to int_shift / GPIV_SHIFT_FACTOR , until it reaches (src)
 *     int_shift. Then, grid_last is set TRUE, which will avoid
 *     changing the interrogation shift in next calls and signal the
 *     (while loop in) the calling function.
 *     
 * INPUTS:
 *     @image_par:                 image parameters
 *     @piv_eval_par_src:          piv evaluation parameters
 *     @piv_data:                  input PIV data
 *     @sweep:                     interrogation sweep step
 *
 * OUTPUTS:
 *     @image_par:                image parameters
 *     @piv_eval_par_dest:        modified piv evaluation parameters
 *     @grid_last:                flag if final grid refinement has been 
 *                                reached
 *     @piv_data:                 modified PIV data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i = 0, j = 0, k = 0, l = 0, k_min = 0, l_min = 0;
    int delta_x_min = 10000;
    int delta_y_min = 10000;
    GpivPivData pd;
    char c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
    gint local_int_shift, local_int_size_1, local_int_size_2;
    gint LOCAL_SHIFT_FACTOR = 2;


    gpiv_null_pivdata(&pd);

    local_int_shift = piv_eval_par_dest->int_shift / LOCAL_SHIFT_FACTOR;
    if (local_int_shift <= piv_eval_par_src.int_shift) {
        *grid_last = TRUE;
    }

    if (*grid_last == FALSE) {
/*
 * - backup estimators from previous sweep
 * - renew grid of PIV dataset 
 * - calculate displacements at new grid points
 */

        pd.nx = piv_data->nx;
        pd.ny = piv_data->ny;
        gpiv_alloc_pivdata(&pd);
        gpiv_cp_pivdata(piv_data, &pd);
        
        
        gpiv_free_pivdata(piv_data);
        piv_eval_par_dest->int_shift = piv_eval_par_dest->int_shift / 
            LOCAL_SHIFT_FACTOR;
        gpiv_piv_count_pivdata_fromimage(piv_data, 
                                         *image_par, 
                                         *piv_eval_par_dest);
        gpiv_alloc_pivdata(piv_data);
        gpiv_piv_gridgen(piv_data, 
                         *image_par, 
                         *piv_eval_par_dest);
        
        gpiv_piv_dxdy_at_new_grid(pd, piv_data);
        gpiv_free_pivdata(&pd);

    } else {
/*
 * reset int_shift (and data positions) to the originally defined 
 * parameter value.
 * For the last grid adaption, the number of interrogation area's may 
 * not have been doubled by definition, as an int_size may be of 
 * arbitrary size. 
 */

        pd.nx = piv_data->nx;
        pd.ny = piv_data->ny;
        gpiv_alloc_pivdata(&pd);
        gpiv_cp_pivdata(piv_data, &pd);


        gpiv_free_pivdata(piv_data);
        piv_eval_par_dest->int_shift =  piv_eval_par_src.int_shift;
/*
 * Set int_size_1 and int_size_2 of piv_eval_par_dest temporarly to the
 * original settings, so that an identic grid will be constructued as
 * during the gpiv_gridgen call.
 */
        local_int_size_1 =  piv_eval_par_dest->int_size_1;
        local_int_size_2 =  piv_eval_par_dest->int_size_2;
        piv_eval_par_dest->int_size_1 = piv_eval_par_src.int_size_1;
        piv_eval_par_dest->int_size_2 = piv_eval_par_src.int_size_2;
        gpiv_piv_count_pivdata_fromimage(piv_data, 
                                         *image_par, 
                                         *piv_eval_par_dest);
        gpiv_alloc_pivdata(piv_data);
        gpiv_piv_gridgen(piv_data, 
                         *image_par, 
                         *piv_eval_par_dest);
        piv_eval_par_dest->int_size_1 = local_int_size_1;
        piv_eval_par_dest->int_size_2 = local_int_size_2;

        gpiv_piv_dxdy_at_new_grid(pd, piv_data);
        gpiv_free_pivdata(&pd);


    }

/*     g_warning("gpiv_piv_gridadapt:: shift=%d3/3", */
/*               piv_eval_par_dest->int_shift ); */
}



void 
gpiv_piv_speedup_cmpr(guint16 **img1, 
                      guint16  * **img1_ACT, 
                      guint16 **img2, 
                      guint16  * **img2_ACT, 
                      GpivImagePar image_par, 
                      GpivImagePar * image_par_ACT, 
                      GpivEvalPar piv_eval_par, 
                      GpivEvalPar * piv_eval_par_ACT, 
                      GpivPivData * piv_data,
                      int *cmpr_fact,
                      int *int_size_0, 
                      int sweep, 
                      int sweep_last
                      )
/*-----------------------------------------------------------------------------
 * Speeds up the evaluation process if zero_off or adaptive interrogation 
 * has been used by compressing the image and disable sub-pixel 
 * estimation for all but last interrogation sweeps
 */

{

    if (sweep == 1) {
        *cmpr_fact = CMPR_FACT;
/*         printf("\n*** warning *** SPEEDUP_CMPR:: sweep = 1\n"); */
        *img1_ACT = gpiv_matrix_guint16(image_par.nrows / CMPR_FACT , 
                                 image_par.ncolumns / CMPR_FACT);
        if (image_par.x_corr) {
            *img2_ACT = gpiv_matrix_guint16(image_par.nrows / CMPR_FACT,
                                     image_par.ncolumns / CMPR_FACT);
        } else {
            *img2_ACT = *img1_ACT;
        }
        gpiv_piv_compress(piv_data, img1, img2, *img1_ACT, *img2_ACT, 
                          image_par_ACT, piv_eval_par_ACT);
        *int_size_0 = 2 * piv_eval_par_ACT->int_size_2;



    } else if (sweep_last == 1) {
/*         printf("\n*** warning *** speedup_cmpr:: last sweep\n"); */
        *cmpr_fact = 1;
        gpiv_piv_decompress(piv_data, image_par_ACT, piv_eval_par_ACT);
        *int_size_0 = 2 * piv_eval_par_ACT->int_size_2;
        gpiv_free_matrix_guint16(*img1_ACT);
        if (image_par.x_corr) {
            gpiv_free_matrix_guint16(*img2_ACT);
        }
        *img1_ACT = img1;
        *img2_ACT = img2;
        piv_eval_par_ACT->ifit = piv_eval_par.ifit;
    }  else {
/*         printf("\n*** warning *** speedup_cmpr:: nothing will happen\n"); */
    }

}



void
gpiv_piv_compress(GpivPivData * piv_data,
		  guint16 **img_1, 
		  guint16 **img_2,
		  guint16 **img_cmpr_1, 
		  guint16 **img_cmpr_2,
		  GpivImagePar * image_par,
		  GpivEvalPar * piv_eval_par
		  )
/*-----------------------------------------------------------------------------
 * Compresses image(s) and adapts relevant variables and parameters
 */
{
     int i, j;

    cmpr_img(img_1, img_cmpr_1, image_par);
    if (image_par->x_corr == 1) {
        cmpr_img(img_2, img_cmpr_2, image_par);
    } else {
	 img_2 = img_1;
    }

/*
 * Checking for meory allocation of iput variables
 */
    assert(img_1[0] != NULL);
    assert(img_2[0] != NULL);
    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);

/*
 * Adapting relevant parameters and variables
 */
    image_par->nrows = image_par->nrows / CMPR_FACT;
    image_par->ncolumns = image_par->ncolumns / CMPR_FACT;
    piv_eval_par->row_start = piv_eval_par->row_start / CMPR_FACT;
    piv_eval_par->row_end = piv_eval_par->row_end / CMPR_FACT;
    piv_eval_par->col_start = piv_eval_par->col_start / CMPR_FACT;
    piv_eval_par->col_end = piv_eval_par->col_end / CMPR_FACT;
    piv_eval_par->int_size_1 = piv_eval_par->int_size_1 / CMPR_FACT;
    piv_eval_par->int_size_2 = piv_eval_par->int_size_2 / CMPR_FACT;
    piv_eval_par->int_shift = piv_eval_par->int_shift / CMPR_FACT;


/*
 * Eventually old PIV values from a previous analysis will also have to be 
 * compressed!
 */
/*     if (old_piv == 1 || adapt_int != 0) { */
    for (i = 0; i < piv_data->ny; i++) {
	 for (j = 0; j < piv_data->nx; j++) {
	      piv_data->point_x[i][j] =
		   piv_data->point_x[i][j] / CMPR_FACT;
	      piv_data->point_y[i][j] =
		   piv_data->point_y[i][j] / CMPR_FACT;
	      piv_data->dx[i][j] = piv_data->dx[i][j] / CMPR_FACT;
	      piv_data->dy[i][j] = piv_data->dy[i][j] / CMPR_FACT;
	 }
    }

}



void
gpiv_piv_decompress(GpivPivData * piv_data,
/* 		    guint16 **img_1,  */
/* 		    guint16 **img_2, */
                    GpivImagePar * image_par,
		    GpivEvalPar * piv_eval_par
                    )
/*-----------------------------------------------------------------------------
 * Adapts relevant variables and parameters
 */
{
     int i, j;
          
     assert(piv_data->point_x != NULL);
     assert(piv_data->point_y != NULL);
     assert(piv_data->dx != NULL);
     assert(piv_data->dy != NULL);
     assert(piv_data->snr != NULL);
     assert(piv_data->peak_no != NULL);
     
     image_par->nrows = image_par->nrows * CMPR_FACT;
     image_par->ncolumns = image_par->ncolumns * CMPR_FACT;
     piv_eval_par->row_start = piv_eval_par->row_start * CMPR_FACT;
     piv_eval_par->row_end = piv_eval_par->row_end * CMPR_FACT;
     piv_eval_par->col_start = piv_eval_par->col_start * CMPR_FACT;
     piv_eval_par->col_end = piv_eval_par->col_end * CMPR_FACT;
     piv_eval_par->int_size_1 = piv_eval_par->int_size_1 * CMPR_FACT;
     piv_eval_par->int_size_2 = piv_eval_par->int_size_2 * CMPR_FACT;
     piv_eval_par->int_shift = piv_eval_par->int_shift * CMPR_FACT;
     
     for (i = 0; i < piv_data->ny; i++) {
	  for (j = 0; j < piv_data->nx; j++) {
	       piv_data->point_x[i][j] =
		    piv_data->point_x[i][j] * CMPR_FACT;
	       piv_data->point_y[i][j] =
		    piv_data->point_y[i][j] * CMPR_FACT;
	       piv_data->dx[i][j] = piv_data->dx[i][j] * CMPR_FACT;
	       piv_data->dy[i][j] = piv_data->dy[i][j] * CMPR_FACT;
	  }
     }

}



#undef EXPANSION_FACT
#undef SMOOTH_WINDOW
#undef NMIN_TO_INTERPOLATE
