/*******************************************************************************************************************************************
 ctablelayout.c
*******************************************************************************************************************************************/

#include "ctablelayout.h"

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CTableLayout);

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
TTablePack::TTablePack 	(const SInt32 inIndex, const UInt16 inL, const UInt16 inR, const UInt16 inT, const UInt16 inB, 
			 const UInt16 inXPad, const UInt16 inYPad, const SInt8 inXOpt, const SInt8 inYOpt)
	   :m_Index	(inIndex),
	    m_L		(inL),
	    m_R		(inR),
	    m_T		(inT),
	    m_B		(inB),
	    m_XPad	(inXPad),
	    m_YPad	(inYPad),
	    m_XOpt	(inXOpt),
	    m_YOpt	(inYOpt)
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// ==
//------------------------------------------------------------------------------------------------------------------------------------------
bool TTablePack::operator == (const TTablePack &inTablePack) const
{
	return m_Index == inTablePack.m_Index;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// !=
//------------------------------------------------------------------------------------------------------------------------------------------
bool TTablePack::operator != (const TTablePack &inTablePack) const
{
	return m_Index != inTablePack.m_Index;
}


//------------------------------------------------------------------------------------------------------------------------------------------
// child pack properties reader
//------------------------------------------------------------------------------------------------------------------------------------------
TTablePack * CTableLayout::GetTablePackFor (const CWidget *inWidget)
{
	// is the widget parameter specified ?
	if (inWidget != NULL)
	{
		// go through the records and see if we find the specified widget
		for (size_t i=m_TablePacks.GetLength(); i>0; i--)
			if (m_TablePacks[i-1]->m_Index == CComponent::GetInOwnerIndex (this, inWidget, __metaclass(CWidget)))
				return m_TablePacks[i-1];
	}

	// the widget is not specified or was not found in the registered pack records, first add a record at the end of the buffer...
	if (m_TablePacks.GetLength() == 0 || m_TablePacks[m_TablePacks.GetLength()-1]->m_Index > -1)
		m_TablePacks += TTablePack (inWidget != NULL ? CComponent::GetInOwnerIndex (this, inWidget, __metaclass(CWidget)) : -1L);
	else 
	{
		m_TablePacks[m_TablePacks.GetLength()-1]->m_Index = 
				inWidget != NULL ? CComponent::GetInOwnerIndex (this, inWidget, __metaclass(CWidget)) : -1L;
		return m_TablePacks[m_TablePacks.GetLength()-1];
	}

	// keep a pointer on the newly added record
	TTablePack *newPack = m_TablePacks[m_TablePacks.GetLength()-1];

	// go through the packs and find where to insert 
	for (size_t i=0; i<m_Packs.GetLength(); i++)
	{
		for (size_t j=0; j<m_Packs[i]->GetLength(); j++)
		{
			if ((*(*m_Packs[i])[j]) == 0)
			{
				newPack->m_L = j;
				newPack->m_R = j+1;
				newPack->m_T = i;
				newPack->m_B = i+1;
				return newPack;
			}
		}
	}
	
	// ok
	return newPack;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// child pack size
//------------------------------------------------------------------------------------------------------------------------------------------
UInt16 CTableLayout::GetTablePackSize (const UInt32 inIndex) const
{
	for (size_t i=m_TablePacks.GetLength(); i>0; i--)
		if (m_TablePacks[i-1]->m_Index == inIndex)
			return (m_TablePacks[i-1]->m_R-m_TablePacks[i-1]->m_L) * 
			       (m_TablePacks[i-1]->m_B-m_TablePacks[i-1]->m_T);
	return 0;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// children pack size
//------------------------------------------------------------------------------------------------------------------------------------------
UInt16 CTableLayout::GetTablePackTotalSize () const
{
	UInt16 outSize=0;
	for (size_t i=m_Packs.GetLength(); i>0; i--)
		for (size_t j=m_Packs[i-1]->GetLength(); j>0; j--)
			outSize += (*(*m_Packs[i-1])[j-1]);
	return outSize;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CTableLayout::CTableLayout (CContainer *inOwner, const CWidgetListener *inListener)
	     :CLayout	   (inOwner, inListener),
	      m_TablePacks (),
	      m_Rows	   (2),
	      m_Cols	   (2)
{
	// launch instanciation process
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CTableLayout::~CTableLayout ()
{
	// request deletion coherence
	CWidget::DestroyWidget (this);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// gtk instanciation
//------------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CTableLayout::PerformWidgetInstanciate ()
{
	// defaults to an homogeneous table until explicitly modified
	GtkWidget *outWidget = ::gtk_table_new (m_Rows, m_Cols, true);

	// default tab packs image allocation
	if (m_Packs.GetLength() == 0) for (size_t i=0; i<m_Rows; i++)
	{
		m_Packs += TBuffer <SInt8> ();
		for (size_t j=0; j<m_Cols; j++) (*m_Packs[i]) += 0;
	}

	// ok
	return outWidget;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// gtk initialization
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::PerformWidgetInitialize ()
{
	// show our job
	Show ();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// child add on
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::PerformContainerAdd (CWidget *inChild)
{
	// get our gtk widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// check pointer
	if (inGtkWidget == NULL || inChild == NULL || inChild -> GetGtkWidget() == NULL) return;

	// get the specified child table pack properties
	TTablePack *inTablePack = GetTablePackFor (inChild);

	// check pointer, should never happen !
	if (inTablePack == NULL) return;

	// keep the image of the new pack size
	for (size_t j=0; j<(inTablePack->m_B-inTablePack->m_T); j++)
		for (size_t i=0; i<(inTablePack->m_R-inTablePack->m_L); i++)
			(*(*m_Packs[inTablePack->m_T+j])[inTablePack->m_L+i]) = 1;

	// add child to the table
	::gtk_table_attach (GTK_TABLE(inGtkWidget), inChild->GetGtkWidget(), inTablePack->m_L, inTablePack->m_R, 
			    inTablePack->m_T, inTablePack->m_B, (GtkAttachOptions)inTablePack->m_XOpt, 
			    (GtkAttachOptions)inTablePack->m_YOpt, inTablePack->m_XPad, inTablePack->m_YPad);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// child removal
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::PerformContainerRemove (CWidget *inChild)
{
	// check pointer
	if (inChild == NULL) return;

	// get the associated table pack properties
	TTablePack *inTablePack = const_cast <TTablePack *> (GetTablePack (inChild));
	
	// check pointer
	if (inTablePack == NULL) return;

	// keep the image of the lost pack size
	for (size_t j=0; j<(inTablePack->m_B-inTablePack->m_T); j++)
		for (size_t i=0; i<(inTablePack->m_R-inTablePack->m_L); i++)
			(*(*m_Packs[inTablePack->m_T+j])[inTablePack->m_L+i]) = 0;

	// keep the table pack index
	SInt32 index = inTablePack -> m_Index;

	// forget the associated pack record
	m_TablePacks -= *inTablePack;

	// go through the table packs, decrease the indices
	for (size_t i=m_TablePacks.GetLength(), j=0; i>0; i--, j++)
		if (m_TablePacks[j]->m_Index > index) m_TablePacks[j]->m_Index--;

	// generic removal handle
	CLayout::PerformContainerRemove (inChild);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// maximum children handling
//------------------------------------------------------------------------------------------------------------------------------------------
SInt16 CTableLayout::GetGtkChildrenNumberLeft () const
{
	return (m_Rows * m_Cols) - GetTablePackTotalSize();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// drop index calculation
//------------------------------------------------------------------------------------------------------------------------------------------
SInt16 CTableLayout::GetDropIndexAtPoint (const TPoint &inRelativePoint, const CControl *inCandidate) const
{
	// no particular handling, always an append specification
	return -1;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// set the table size
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::SetTableSize (const UInt16 inRows, const UInt16 inCols)
{
	// default tab packs image reallocation or part deletion
	if (inRows > m_Rows)
		for (size_t i=m_Packs.GetLength(); i<inRows; i++) 
			m_Packs += TBuffer <SInt8> ();
	else if (inRows < m_Rows)
		m_Packs.Delete (inRows, m_Packs.GetLength()-inRows);
	if (inCols > m_Cols)
		for (size_t i=0; i<m_Rows; i++) 
			for (size_t j=m_Packs[i]->GetLength(); j<inCols; j++) 
				(*m_Packs[i]) += 0;
	else if (inCols < m_Cols)
		for (size_t i=0; i<inRows; i++) 
			m_Packs[i]->Delete (inCols, m_Packs[i]->GetLength()-inCols);
	for (size_t i=m_Rows; i<inRows; i++) 
		for (size_t j=0; j<inCols; j++) 
			(*m_Packs[i]) += 0;

	// local copy of the values
	m_Rows = inRows;
	m_Cols = inCols;

	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget();

	// set the size properties
	if (inGtkWidget != NULL) ::gtk_table_resize (GTK_TABLE(inGtkWidget), inRows, inCols);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// get the rows number
//------------------------------------------------------------------------------------------------------------------------------------------
UInt16 CTableLayout::GetTableRows () const
{
	return m_Rows;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// get the columns number
//------------------------------------------------------------------------------------------------------------------------------------------
UInt16 CTableLayout::GetTableCols () const
{
	return m_Cols;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// set the table pack
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::SetTablePack (const UInt16 inLeft, const UInt16 inRight, const UInt16 inTop, const UInt16 inBottom, 
				 const UInt16 inXPad, const UInt16 inYPad, const SInt8 inXOpt, const SInt8 inYOpt)
{
	// get the next child table pack properties pointer
	TTablePack *inTablePack = GetTablePackFor ();

	// check pointer
	if (inTablePack == NULL) return;
	
	// register the pack properties
	inTablePack->m_L    = inLeft;
	inTablePack->m_R    = inRight;
	inTablePack->m_T    = inTop;
	inTablePack->m_B    = inBottom;
	inTablePack->m_XPad = inXPad;
	inTablePack->m_YPad = inYPad;
	inTablePack->m_XOpt = inXOpt;
	inTablePack->m_YOpt = inYOpt;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// get the child pack properties
//------------------------------------------------------------------------------------------------------------------------------------------
const TTablePack * CTableLayout::GetTablePack (const CWidget *inChild) const
{
	// check pointer
	if (inChild == NULL) return NULL;

	// search for the specified widget
	for (size_t i=m_TablePacks.GetLength(); i>0; i--) 
		if (m_TablePacks[i-1]->m_Index == CComponent::GetInOwnerIndex (this, inChild, __metaclass(CWidget)))
			return m_TablePacks[i-1];

	// not found
	return NULL;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// table pack from specified point
//------------------------------------------------------------------------------------------------------------------------------------------
TPoint CTableLayout::GetTablePackFromPoint (const TPoint &inRelativePoint) const
{
	// get the left and top indices values
	SInt16 l = (SInt16) (float(inRelativePoint.x) / m_Bounds.w * m_Cols);
	SInt16 t = (SInt16) (float(inRelativePoint.y) / m_Bounds.h * m_Rows);

	// ok
	return TPoint (l, t);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// table packs access
//------------------------------------------------------------------------------------------------------------------------------------------
TTablePacks CTableLayout::GetTablePacks () const
{
	return m_TablePacks;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// homogeneous set
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::SetHomogeneous (const bool inHomogeneous)
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// set the widget property
	if (inGtkWidget != NULL) ::gtk_table_set_homogeneous (GTK_TABLE(inGtkWidget), inHomogeneous);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// get homogeneous
//------------------------------------------------------------------------------------------------------------------------------------------
bool CTableLayout::GetHomogeneous () const
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// retreive the widget property
	return inGtkWidget != NULL ? ::gtk_table_get_homogeneous (GTK_TABLE(inGtkWidget)) : false;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// col spacing writer
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::SetColSpacing (const UInt16 inCol, const UInt16 inSpacing)
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// set the widget property
	if (inGtkWidget != NULL) ::gtk_table_set_col_spacing (GTK_TABLE(inGtkWidget), inCol, inSpacing);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// col spacing reader
//------------------------------------------------------------------------------------------------------------------------------------------
UInt16 CTableLayout::GetColSpacing (const UInt16 inCol) const
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// retreive the property
	return inGtkWidget != NULL ? ::gtk_table_get_col_spacing (GTK_TABLE(inGtkWidget), inCol) : 0;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// row spacing writer
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::SetRowSpacing (const UInt16 inRow, const UInt16 inSpacing)
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// set the widget property
	if (inGtkWidget != NULL) ::gtk_table_set_row_spacing (GTK_TABLE(inGtkWidget), inRow, inSpacing);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// row spacing reader
//------------------------------------------------------------------------------------------------------------------------------------------
UInt16 CTableLayout::GetRowSpacing (const UInt16 inRow) const
{
	// get our widget
	GtkWidget *inGtkWidget = GetGtkWidget ();

	// retreive the property
	return inGtkWidget != NULL ? ::gtk_table_get_row_spacing (GTK_TABLE(inGtkWidget), inRow) : 0;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
void CTableLayout::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// serialization request analyse
	switch (inMode)
	{
		// xml dump
		case XML_WRITE :
		{
			// generics serialization first
			CLayout::Serialize (ioXMLElementNode, inMode);

			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_TABLELAYOUT_ELEMENT);

			// set the element attributes
			newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_ROWS, CString((UInt32)m_Rows));
			newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_COLS, CString((UInt32)m_Cols));
			newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_HOMOGENEOUS, CString(GetHomogeneous()?"true":"false"));

			// modify the io xml element node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// foreach row, set a new xml element
			for (size_t i=m_Rows-1, j=0; i>0; i--, j++)
			{
				// instanciate the element
				newXMLElement = new CXMLElement (ioXMLElementNode, XML_TABLELAYOUT_ROW_ELEMENT);

				// add the attribute
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_SPACING, CString((UInt32)GetRowSpacing(j)));
			}
			// foreach column, set a new xml element
			for (size_t i=m_Cols-1, j=0; i>0; i--, j++)
			{
				// instanciate the element
				newXMLElement = new CXMLElement (ioXMLElementNode, XML_TABLELAYOUT_COL_ELEMENT);

				// add the attribute
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_SPACING, CString((UInt32)GetColSpacing(j)));
			}

			// get the table layout widget children
			CComponents inChildren (GetChildren (__metaclass(CWidget)));

			// foreach widget child
			for (size_t i=inChildren.GetLength(), j=0; i>0; i--, j++)
			{
				// get the associated pack
				TTablePack *inPack = const_cast <TTablePack *> (GetTablePack (static_cast <CWidget *> (*inChildren[j])));

				// check the pack pointer
				if (inPack == NULL) continue;

				// instanciate the child element
				newXMLElement = new CXMLElement (ioXMLElementNode, XML_TABLELAYOUT_CHILD_ELEMENT);

				// add the child attributes
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_LEFT,     CString((UInt32)inPack->m_L));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_RIGHT,    CString((UInt32)inPack->m_R));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_TOP, 	    CString((UInt32)inPack->m_T));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_BOTTOM,   CString((UInt32)inPack->m_B));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_XPADDING, CString((UInt32)inPack->m_XPad));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_YPADDING, CString((UInt32)inPack->m_YPad));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_XEXPAND, 
									CString(inPack->m_XOpt&GTK_EXPAND?"true":"false"));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_XSHRINK, 
									CString(inPack->m_XOpt&GTK_SHRINK?"true":"false"));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_XFILL, 
									CString(inPack->m_XOpt&GTK_FILL  ?"true":"false"));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_YEXPAND, 
									CString(inPack->m_YOpt&GTK_EXPAND?"true":"false"));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_YSHRINK, 
									CString(inPack->m_YOpt&GTK_SHRINK?"true":"false"));
				newXMLElement -> AddAttribute (XML_TABLELAYOUT_ATTR_CHILD_YFILL, 
									CString(inPack->m_YOpt&GTK_FILL  ?"true":"false"));
			}
		}
		break;

		// xml load
		case XML_READ :
		{
			// search for the node this instance is interested in
			CXMLElementNode *inXMLNode = ::xml_node_search (ioXMLElementNode, XML_TABLELAYOUT_ELEMENT); 

			// check we got an expected cboxlayout node
			if (inXMLNode == NULL)
				throw new CException (CString("CTableLayout::Serialize, specified xml node is not a \"") + 
							      XML_TABLELAYOUT_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// read in the xml node attributes
			SetTableSize   (::xml_node_get_attribute (inXMLNode, XML_TABLELAYOUT_ATTR_ROWS).       GetValue().ToULong(),
					::xml_node_get_attribute (inXMLNode, XML_TABLELAYOUT_ATTR_COLS).       GetValue().ToULong());
			SetHomogeneous (::xml_node_get_attribute (inXMLNode, XML_TABLELAYOUT_ATTR_HOMOGENEOUS).GetValue().ToBool ());
	
			// foreach child of the current xml node
			for (size_t i=0, j=0; i<xml_node_get_children_number(inXMLNode); i++)
			{
				// get the child node
				CXMLElementNode *inXMLChild = ::xml_node_get_child (inXMLNode, i);

				// if the xml node is a row specification node
				if (::xml_node_get_name (inXMLChild) == XML_TABLELAYOUT_ROW_ELEMENT)
				{
					static UInt16 Row=0;
					SetRowSpacing (Row++, 
						::xml_node_get_attribute (inXMLNode, XML_TABLELAYOUT_ATTR_SPACING).GetValue().ToULong());
				}
				// if the xml node is a col specification node
				else if (::xml_node_get_name (inXMLChild) == XML_TABLELAYOUT_COL_ELEMENT)
				{
					static UInt16 Col=0;
					SetColSpacing (Col++, 
						::xml_node_get_attribute (inXMLChild, XML_TABLELAYOUT_ATTR_SPACING).GetValue().ToULong());
				}
				// if the xml node is a child pack specification node
				else if (::xml_node_get_name (inXMLChild) == XML_TABLELAYOUT_CHILD_ELEMENT)
				{
					// get the x and y pack options properties
					bool inXExpand = 
						::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_XEXPAND).GetValue().ToBool();
					bool inYExpand = 
						::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_YEXPAND).GetValue().ToBool();
					bool inXShrink = 
						::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_XSHRINK).GetValue().ToBool();
					bool inYShrink = 
						::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_YSHRINK).GetValue().ToBool();
					bool inXFill = 
						::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_XFILL  ).GetValue().ToBool();
					bool inYFill = 
						::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_YFILL  ).GetValue().ToBool();

					// declare a new table pack structure
					TTablePack inPack (j++,
					::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_LEFT    ).GetValue().ToULong(),
					::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_RIGHT   ).GetValue().ToULong(),
					::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_TOP     ).GetValue().ToULong(),
					::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_BOTTOM  ).GetValue().ToULong(),
					::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_XPADDING).GetValue().ToULong(),
					::xml_node_get_attribute(inXMLChild,XML_TABLELAYOUT_ATTR_CHILD_YPADDING).GetValue().ToULong(),
					(inXExpand?GTK_EXPAND:0)|(inXShrink?GTK_SHRINK:0)|(inXFill?GTK_FILL:0),
					(inYExpand?GTK_EXPAND:0)|(inYShrink?GTK_SHRINK:0)|(inYFill?GTK_FILL:0));

					// add the table pack to the list
					m_TablePacks += inPack;
				}
			}

			// now we got our pack attributes and we are ready to handle container add requests, so call the generic
			// serialization process that will instanciate all of the sub components
			CLayout::Serialize (ioXMLElementNode, inMode);

			// modify the in/out element node pointer so that the serialization process will continue under the cwidget node
			ioXMLElementNode = inXMLNode;
		}
		break;
	}
}



