/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: AbstractFileCommand.java,v 1.6 2007/01/11 08:28:39 pietschy Exp $
 */
package org.pietschy.command.file;

import org.pietschy.command.ActionCommand;
import org.pietschy.command.CommandManager;

import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.awt.*;
import java.io.File;

/**
 * This is the base implementation of file commands.  It provides the basic {@link JFileChooser}
 * functionality.  Subclass must override {@link #showChooserDialog} and {@link #performFileAction}.
 *
 * @version $Revision: 1.6 $
 * @author andrewp
 */
public abstract class
AbstractFileCommand
extends ActionCommand
{
   private static final String _ID_ = "$Id: AbstractFileCommand.java,v 1.6 2007/01/11 08:28:39 pietschy Exp $";

   private JFileChooser fileChooser;
   private FileFilter[] filters;
   private FileFilter defaultFileFilter;
   private boolean rememberLastFilter = false;
   private FileFilter lastUsedFilter = null;
   private boolean acceptAllFileFilterUsed = true;
   private boolean centerOnInvoker = true;

   /**
    * Creates a new AbstractFileCommand with the specified file filters.  Filters must contain
    * at least one instance of {@link javax.swing.filechooser.FileFilter}.
    * @param filters the file filters to use.
    * @throws java.lang.NullPointerException if filters is null
    * @throws java.lang.IllegalArgumentException if filters is an empty array.
    */
   public AbstractFileCommand(CommandManager manager, String id, FileFilter[] filters)
   {
      super(manager, id);

      if (filters == null)
         throw new NullPointerException("filters is null");

      if (filters.length < 1)
         throw new IllegalArgumentException("filters is empty");

      this.filters = filters;
      defaultFileFilter = filters[0];
   }

   protected final void
   handleExecute()
   {

      try
      {
         beforeExecute();

         if (!confirmProceed())
            return;

         JFileChooser chooser = getFileChooser();
         Window invokerWindow = getInvokerWindow();
         chooser.setAcceptAllFileFilterUsed(isAcceptAllFileFilterUsed());

         if (rememberLastFilter && lastUsedFilter != null)
            chooser.setFileFilter(lastUsedFilter);
         else
            chooser.setFileFilter(defaultFileFilter);

         int result = showChooserDialog(chooser, isCenterOnInvoker() ? invokerWindow : null);

         lastUsedFilter = chooser.getFileFilter();

         if (result == JFileChooser.APPROVE_OPTION)
         {
            if (chooser.isMultiSelectionEnabled())
               performFileAction(chooser.getSelectedFiles(), chooser, invokerWindow);
            else
               performFileAction(new File[]{chooser.getSelectedFile()}, chooser, invokerWindow);
         }
         else if (result == JFileChooser.CANCEL_OPTION)
         {
            handleCancel(chooser, invokerWindow);
         }
         else if (result == JFileChooser.ERROR_OPTION)
         {
            handleError(chooser, invokerWindow);
         }
      }
      finally
      {
         afterExecute();
      }
   }

   /**
    * This method is invoked if the user cancels the file chooser dialog.  By default this
    * method doesn nothing.  Subclasses can override to perform some meaningful action.
    * @param chooser the chooser that was cancelled.
    * @param invokerWindow the window associated with the button or menu that invoked the command.
    */
   protected void
   handleCancel(JFileChooser chooser, Window invokerWindow)
   {
   }

   /**
    * This is method can be overridden to prevent the action from proceeding.  By default this
    * method simply returns true.
    * @return <code>true</code> to preceed with showing the chooser, <code>false</code> to cancel the
    * action.  This method always returns <code>true</code>.
    */
   protected boolean
   confirmProceed()
   {
      return true;
   }

   /**
    * Invoked before the command is executed.
    * @deprecated Use {@link org.pietschy.command.ActionCommandInterceptor} instead.
    */
   protected void
   beforeExecute(){}

   /**
    * Invoked after the command is executed.
    * @deprecated Use {@link org.pietschy.command.ActionCommandInterceptor} instead.
    */
   protected void
   afterExecute(){}

   /**
    * This method is invoked if the {@link #showChooserDialog} returns
    * {@link JFileChooser#ERROR_OPTION}.  The default implementation does nothing, subclasses
    * can override to handle the error.
    * @param chooser the {@link JFileChooser} that was displayed.
    * @param invoker the owner window.
    */
   protected void
   handleError(JFileChooser chooser, Window invoker)
   {
   }

   /**
    * Called to display the {@link JFileChooser}.  Subclasses override to display an appropriate
    * version of the chooser (such as an open or save dialog).  This method must return the result
    * of the {@link JFileChooser} show method. Eg.
    * <pre>
    *protected int showChooserDialog(JFileChooser chooser, Window invoker)
    *{
    *   return chooser.showOpenDialog(invoker);
    *}</pre>
    * @param chooser the chooser to display
    * @return the result of {@link JFileChooser#showDialog}, {@link JFileChooser#showOpenDialog} or
    * {@link JFileChooser#showSaveDialog}.
    */
   protected abstract int
   showChooserDialog(JFileChooser chooser, Window centerOn);

   /**
    * This method is invoked if the {@link #showChooserDialog} returns
    * {@link JFileChooser#APPROVE_OPTION}.  Subclasses must override to perform the specific file
    * operation.
    * @param files the files that were selected in the {@link JFileChooser}
    * @param chooser the {@link JFileChooser} that was displayed.
    * @param invoker the owner window.
    */
   protected abstract void
   performFileAction(File[] files, JFileChooser chooser, Window invoker);

   /**
    * Gets the file choose the command will use.  The chooser will be configured with the
    * filters specified by the command.
    * @return the commands {@link javax.swing.JFileChooser}.
    */
   protected JFileChooser
   getFileChooser()
   {
      if (fileChooser == null)
      {
         fileChooser = new JFileChooser();

         if (filters.length == 1)
         {
            fileChooser.setFileFilter(filters[0]);
         }
         else
         {
            for (int i = 0; i < filters.length; i++)
               fileChooser.addChoosableFileFilter(filters[i]);
         }
      }

      return fileChooser;
   }


   /**
    * Gets the default {@link FileFilter} that will be selected in the chooser if
    * {@link #isRememberLastFilter} is {@link false}.
    * <p>
    * If not explicitly specified, this value will be the filter specified in the constructor, or
    * the first filter if a list of filteres was specified.
    */
   public FileFilter
   getDefaultFileFilter()
   {
      return defaultFileFilter;
   }

   /**
    * Sets the default {@link FileFilter} that will be selected in the chooser if
    * {@link #isRememberLastFilter} is {@link false}.
    * <p>
    * If not explicitly specified, this value will be the filter specified in the constructor, or
    * the first filter if a list of filteres was specified.
    * @param defaultFileFilter the default file filter.
    */
   public void
   setDefaultFileFilter(FileFilter defaultFileFilter)
   {
      this.defaultFileFilter = defaultFileFilter;
   }

   /**
    * Checks if the last {@link FileFilter} used by the user will be remembered.
    * @return <tt>true</tt> if the command remebers the last {@link FileFilter} used by the
    * user, <tt>false</tt> if default is always used.
    */
   public boolean
   isRememberLastFilter()
   {
      return rememberLastFilter;
   }

   /**
    * Configures this command to remember the {@link FileFilter} last used by the user.  This
    * property defaults to <tt>false</tt>.
    * @param rememberLastFilter <tt>true</tt> to remember the last {@link FileFilter} used,
    * <tt>false</tt> to always used the default {@link FileFilter}.
    */
   public void
   setRememberLastFilter(boolean rememberLastFilter)
   {
      this.rememberLastFilter = rememberLastFilter;
   }

   /**
    * Used to configure {@link javax.swing.JFileChooser#setAcceptAllFileFilterUsed(boolean)}.
    */
   public boolean
   isAcceptAllFileFilterUsed()
   {
      return this.acceptAllFileFilterUsed;
   }

   /**
    * Used to configure {@link javax.swing.JFileChooser#setAcceptAllFileFilterUsed(boolean)}.
    */
   public void
   setAcceptAllFileFilterUsed(boolean accept)
   {
      this.acceptAllFileFilterUsed = accept;
   }

   /**
    * Checks if this command will be centering the chooser on the invoker window.  If
    * <tt>false</tt>, the command will invoke {@link #showChooserDialog} with a null window to
    * center on.
    * <p>
    * This setting will only be honoured if subclasses implement {@link #showChooserDialog} to
    * correctly utilize the <tt>centerOn</tt> parameter.
    * @return <tt>true</tt> if the chooser will be centered on the invoker window, <tt>false</tt> to
    * center on the desktop.
    */
   public boolean
   isCenterOnInvoker()
   {
      return centerOnInvoker;
   }

   /**
    * Configures if command will centre the chooser over the invoker or centre in the desktop.  If
    * <tt>false</tt>, the command will invoke {@link #showChooserDialog} with a null window to
    * center on.
    * @param centerOnInvoker <tt>true</tt> to center the chooser on the invoker window, <tt>false</tt> to
    * centre on the desktop.
    */
   public void
   setCenterOnInvoker(boolean centerOnInvoker)
   {
      this.centerOnInvoker = centerOnInvoker;
   }
}

