/***************************************************************************
 *   Copyright (C) 2006 by Bram Biesbrouck                                 *
 *   b@beligum.org                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA.             *
 *
 *   In addition, as a special exception, the copyright holders give	   *
 *   permission to link the code of portions of this program with the	   *
 *   OpenSSL library under certain conditions as described in each	   *
 *   individual source file, and distribute linked combinations		   *
 *   including the two.							   *
 *   You must obey the GNU General Public License in all respects	   *
 *   for all of the code used other than OpenSSL.  If you modify	   *
 *   file(s) with this exception, you may extend this exception to your	   *
 *   version of the file(s), but you are not obligated to do so.  If you   *
 *   do not wish to do so, delete this exception statement from your	   *
 *   version.  If you delete this exception statement from all source	   *
 *   files in the program, then also delete it here.			   *
 ***************************************************************************/

#include <cstdio>
#include <cstdlib>
#include <iostream>

#include <FTGL/FTGLTextureFont.h>

#include <libinstrudeo/isdcaptoverlay.h>
#include <libinstrudeo/isdroundedbox.h>
#include <libinstrudeo/isdtextballoon.h>
#include <libinstrudeo/isdtextballoon2D.h>
#include <libinstrudeo/isdutils.h>
#include <libinstrudeo/isdvideocanvas.h>
#include <libinstrudeo/isdlogger.h>
#include <libinstrudeo/isdglobjfile.h>
#include <libinstrudeo/isdcommentbox.h>

#undef LOG_HEADER
#define LOG_HEADER "Error while accessing a generic commentbox: \n"
#include <libinstrudeo/isdloggermacros.h>

//-----CONSTANTS-----
float ISDCommentbox::MIRROR_MATRIX_NONE[16] = 
    {
	1.0, 0.0, 0.0, 0.0,
	0.0, 1.0, 0.0, 0.0,
	0.0, 0.0, 1.0, 0.0,
	0.0, 0.0, 0.0, 1.0
    };
float ISDCommentbox::MIRROR_MATRIX_HORIZONTAL[16] = 
    {
	-1.0, 0.0, 0.0, 0.0,
	0.0, 1.0, 0.0, 0.0,
	0.0, 0.0, 1.0, 0.0,
	0.0, 0.0, 0.0, 1.0
    };
float ISDCommentbox::MIRROR_MATRIX_VERTICAL[16] = 
    {
	1.0, 0.0, 0.0, 0.0,
	0.0, -1.0, 0.0, 0.0,
	0.0, 0.0, 1.0, 0.0,
	0.0, 0.0, 0.0, 1.0
    };
float ISDCommentbox::MIRROR_MATRIX_BOTH[16] = 
    {
	-1.0, 0.0, 0.0, 0.0,
	0.0, -1.0, 0.0, 0.0,
	0.0, 0.0, 1.0, 0.0,
	0.0, 0.0, 0.0, 1.0
    };

//-----CONSTRUCTORS-----
ISDCommentbox::ISDCommentbox(ISDVideoCanvas* parent, int id)
    : ISDGLWidget(), id(id),
      mirrorMode(ISD_COMMENTBOX_MIRROR_NONE),
      mirrorMatrix(MIRROR_MATRIX_NONE),
      text(DEFAULT_COMMENTBOX_TEXT),
      language(DEFAULT_COMMENTBOX_LANGUAGE),
      startTime(DEFAULT_COMMENTBOX_STARTTIME),
      duration(DEFAULT_COMMENTBOX_DURATION),
      font(NULL)
{
    //init the light position with default values
    lightPosition[0] = 0.0;
    lightPosition[1] = 0.0;
    lightPosition[2] = 1.0;
    lightPosition[3] = 0.0;

    if (parent!=NULL) {
	this->setParent(parent);
	/*
	 * Note: don't init the size, position, colors in this constructor,
	 * some of them result in the calling of the virtual function getObjFile
	 * and because the subclass-doesn't exist yet, the pure virtual
	 * function of this class is called.
	 */

	/*
	 * Note: don't add this commentbox to the parent's child-list automatically,
	 *       because we don't know if must be shown or not. (all children are shown)
	 */
	//parent->addChild(this);
    }

    initFont();
}

//-----DESTRUCTOR-----
ISDCommentbox::~ISDCommentbox()
{
    //remove ourselves from the parent's child list
    ISDGLWidget* parent = getParent();
    if (parent!=NULL) {
	parent->removeChild(this);
    }

    if (font!=NULL) {
	delete font;
	font = NULL;
    }
}

//-----FACTORY METHODS-----
ISDCommentbox* ISDCommentbox::createCommentbox(commentboxType type, ISDVideoCanvas* parent, int id)
{
    ISDCommentbox* retVal = NULL;

    switch (type) {
    case ISD_COMMENT_TEXTBALLOON_TYPE:
	return new ISDTextballoon(parent, id);
	break;
    case ISD_COMMENT_ROUNDED_BOX_TYPE:
        return new ISDRoundedBox(parent, id);
        break;
    case ISD_COMMENT_TEXTBALLOON_2D_TYPE:
        return new ISDTextballoon2D(parent, id);
        break;
    case ISD_COMMENT_CAPTOVERLAY_TYPE:
	return new ISDCaptoverlay(parent, id);
	break;
    default:
	LOG_WARNING("Trying to create a commentbox of unknown type.");
	break;
    }

    return retVal;
}

//-----PUBLIC GETTERS-----
int ISDCommentbox::getId()
{
    return id;
}
float ISDCommentbox::getModelUnit()
{
    //this constant is located in the isdglobjfile.h file
    return MODEL_UNIT_SCALE;
}
void ISDCommentbox::getHotspot(int pos[])
{
    //defaults to 0,0
    pos[0] = 0;
    pos[1] = 0;
}
Glib::ustring& ISDCommentbox::getLanguage()
{
    return language;
}
void ISDCommentbox::getPosition(int pos[])
{   
    pos[0] = this->position[0];
    pos[1] = this->position[1];
}
void ISDCommentbox::getScaledPosition(int pos[])
{
    float pixelZoom;
    glGetFloatv(GL_ZOOM_X, &pixelZoom);

    pos[0] = (int)(this->position[0]*pixelZoom);
    pos[1] = (int)(this->position[1]*pixelZoom);
}
void ISDCommentbox::getSize(int size[])
{
    size[0] = this->size[0];
    size[1] = this->size[1];
}
void ISDCommentbox::getScaledSize(int size[2])
{
    float pixelZoom;
    glGetFloatv(GL_ZOOM_X, &pixelZoom);
    
    size[0] = (int)(this->size[0]*pixelZoom);
    size[1] = (int)(this->size[1]*pixelZoom);
}
Glib::ustring& ISDCommentbox::getText()
{
    return text;
}
ISDCommentbox::commentboxMirrorMode ISDCommentbox::getMirrorMode()
{
    return mirrorMode;
}
int ISDCommentbox::getStartTime()
{
    return startTime;
}
int ISDCommentbox::getDuration()
{
    return duration;
}
void ISDCommentbox::getColor(float color[])
{
    color[0] = this->color[0];
    color[1] = this->color[1];
    color[2] = this->color[2];
    color[3] = this->color[3];
}
void ISDCommentbox::getTextColor(float color[])
{
    color[0] = this->textColor[0];
    color[1] = this->textColor[1];
    color[2] = this->textColor[2];
    color[3] = this->textColor[3];
}

//-----PUBLIC SETTERS-----
void ISDCommentbox::setLanguage(Glib::ustring& language)
{
    this->language = language;
}
void ISDCommentbox::setPosition(int x, int y)
{
    this->position[0] = x;
    this->position[1] = y;

    recalcScales();
}
void ISDCommentbox::setScaledPosition(int x, int y)
{
    float pixelZoom;
    glGetFloatv(GL_ZOOM_X, &pixelZoom);
    /*
     * We receive a scaled width and height, but we store
     * the position unscaled, so calc the unscaled position.
     */
    this->position[0] = (int)(x/pixelZoom);
    this->position[1] = (int)(y/pixelZoom);

    recalcScales();
}
void ISDCommentbox::setSize(int width, int height)
{
    this->size[0] = width;
    this->size[1] = height;

    recalcScales();
}
void ISDCommentbox::setScaledSize(int width, int height)
{
    float pixelZoom;
    glGetFloatv(GL_ZOOM_X, &pixelZoom);
    
    /*
     * We receive a scaled width and height, but we store
     * the dimension unscaled, so calc the unscaled dimension.
     */
    this->size[0] = (int)(width/pixelZoom);
    this->size[1] = (int)(height/pixelZoom);

    recalcScales();
}
void ISDCommentbox::setText(Glib::ustring& text)
{
    this->text = text;
}
void ISDCommentbox::setMirrorMode(ISDCommentbox::commentboxMirrorMode mode)
{
    this->mirrorMode = mode;

    switch (mode) {
    case ISD_COMMENTBOX_MIRROR_NONE:
	mirrorMatrix = MIRROR_MATRIX_NONE;
	break;
    case ISD_COMMENTBOX_MIRROR_HORIZONTAL:
	mirrorMatrix = MIRROR_MATRIX_HORIZONTAL;
	break;
    case ISD_COMMENTBOX_MIRROR_VERTICAL:
	mirrorMatrix = MIRROR_MATRIX_VERTICAL;
	break;
    case ISD_COMMENTBOX_MIRROR_BOTH:
	mirrorMatrix = MIRROR_MATRIX_BOTH;
	break;
    }

    recalcScales();
}
void ISDCommentbox::setStartTime(int time)
{
    this->startTime = time;
}
void ISDCommentbox::setDuration(int duration)
{
    this->duration = duration;
}
void ISDCommentbox::setColor(float r, float g, float b, float a)
{
    this->color[0] = r;
    this->color[1] = g;
    this->color[2] = b;
    this->color[3] = a;
}
void ISDCommentbox::setTextColor(float r, float g, float b, float a)
{
    this->textColor[0] = r;
    this->textColor[1] = g;
    this->textColor[2] = b;
    this->textColor[3] = a;
}
void ISDCommentbox::display()
{
    glLoadIdentity();
    
    //draw the model first
    glPushMatrix();
        glEnable(GL_DEPTH_TEST);
	//enable back-facing polygons
	glEnable(GL_CULL_FACE);
	//enable transparency
	glEnable(GL_BLEND);

	//load the mirror matrix
	glLoadMatrixf(mirrorMatrix);

	/*
	 * If we mirror (one time, not both), we're looking at the 
	 * back of the model, so we need to Z-mirror the light to
	 * the back of the model to keep the lighting.
	 */
	if (mirrorMode==ISD_COMMENTBOX_MIRROR_HORIZONTAL ||
	    mirrorMode==ISD_COMMENTBOX_MIRROR_VERTICAL)
	    {
		float newLightPos[4] = {lightPosition[0],
					lightPosition[1],
					-lightPosition[2],
					lightPosition[3]};
		
		glLightfv(GL_LIGHT0, GL_POSITION, newLightPos);
	    }
	else {
	    glLightfv(GL_LIGHT0, GL_POSITION, lightPosition);
	}
		
	//we always Z-align to the XY plane
	glTranslatef(scaledPosition[0], scaledPosition[1], 0.0);

	/*
	 * These values give the subclass the opportunity to tweak the scaling.
	 * Default values are 1.0, 1.0, 1.0, so defaults don't do anything.
	 */
	float tweakValues[3];
	getScalesTweakValues(tweakValues);
	
	glScalef(scales[0]*tweakValues[0],
		 scales[1]*tweakValues[1],
		 scales[2]*tweakValues[2]);

	displayModel();
	
	glDisable(GL_BLEND);
	glDisable(GL_CULL_FACE);
	glDisable(GL_DEPTH_TEST);
    glPopMatrix();
    
    /*
     * By drawing the model first, we can never obscure the text with the model.
     * This way, we can't achieve text-transparency, but I prefer this method,
     * as the text is more important than the model.
     */
    
    //draw the text
    glLoadIdentity();
    glPushMatrix();
        glEnable(GL_TEXTURE_2D);
	glDisable(GL_DEPTH_TEST);
	glEnable(GL_COLOR_MATERIAL);

	glColor3f(textColor[0], textColor[1], textColor[2]);

	glTranslatef(scaledTextplaneOrigin[0],
		     scaledTextplaneOrigin[1],
		     scaledTextplaneOrigin[2]);
	
	float pixelZoom;
	glGetFloatv(GL_ZOOM_X, &pixelZoom);
	/*
	 * Don't scale the font  with scales[]; the commentbox is scaled around the text
	 * We must use the pixelZoom tough, otherwise the text will not scale when zooming
	 * and that's not what we want.
	 */
	glScalef(FONT_SCALE_SIZE*pixelZoom,
		 FONT_SCALE_SIZE*pixelZoom,
		 0.0);
	
	glNormal3f(0.0, 0.0, 1.0);
	
        displayText();

	// don't enable depth test here, it caused the mesh to disappear...
	//glEnable(GL_DEPTH_TEST);
	glDisable(GL_TEXTURE_2D);
    glPopMatrix();
}
void ISDCommentbox::recalcScales()
{
    /*
     * Get the dimensions of the whole model, scaled around a centered unit cube
     * See the isdglobjfile.h file for the size of that unit cube (MODEL_UNIT_SCALE)
     * It isn't really important here, since we don't use it; we ask for the dimensions
     * of the model.
     */
    ISDGLObjFile* objFile = this->getObjFile();
    if (objFile==NULL || objFile->error()) {
	LOG_WARNING("Couldn't get the .obj class wrapper.");
	return;
    }
    float width, height, depth;
    objFile->getDimensions(&width, &height, &depth);

    /*
     * Since the requested size of the model is nearly never the same
     * as the dimensions of the model in memory, we must calculate a scaled
     * value for every dimension the user requested that maps the dimension on
     * this requested value.
     * Because the user never specifies a depth, we assume the depth to be the 
     * mean value of the width and height of the commentbox.
     */
    scales[0] = size[0]/width;
    scales[1] = size[1]/height;
    //use the mean of the width and height for the new depth
    scales[2] = (size[0]+size[1])/2/depth;

    /*
     * It's possible that the user is zooming the entire canvas.
     * This zoom-factor is set by the OpenGL call glPixelZoom(xScale, yScale) and
     * is abstracted by the updateScales(newScale) method of the ISDVideoCanvas class,
     * so that the X and Y scales are always the same and therefore can be used
     * for the Z dimension too.
     */
    float pixelZoom;
    glGetFloatv(GL_ZOOM_X, &pixelZoom);
    scales[0] *= pixelZoom;
    scales[1] *= pixelZoom;
    scales[2] *= pixelZoom;

    /*
     * IMPORTANT: Keep in mind we are positioning on the original (unzoomed) canvas
     *            and need to use the sizes as specified by the user, not in memory.
     */

    /*
     * Now, the 3D (righthanded) OpenGL coordinate-system is different from
     * 2D coordinates; the Y axis is pointing down in 2D, up in 3D, so we need to
     * mirror the Y values if we draw in 3D.
     *
     * Since we don't know the height of the canvas here (needed to mirror the value),
     * fetch a reference to the parent of this commentbox that needs to be the videoCanvas.
     * Because the API doen't enforce this, we include a check that throws a critical error.
     */
    ISDVideoCanvas* canvas = dynamic_cast<ISDVideoCanvas*>(getParent());
    if (!canvas) {
	LOG_CRITICAL("This commentbox is not a child of the video canvas, this shouldn't happen.");
	lastError = ISD_FAILURE;
	return;
    }

    /*
     * The mesh is always positioned around it's center (remember the centered unit cube),
     * so we need to add/substract half the size in every dimension because the user specifies
     * the top left corner position.
     * Plus, we perform the above Y-mirroring (around the X axis).
     * Note: we keep the main mesh fixed at Z-position zero, so there's no need to calc/keep 3 dimensions.
     * Note2: instead of using size[0], we use width*scales[0] to include the pixelZoom.
     */
    scaledPosition[0] = (position[0]*pixelZoom) + (width*scales[0]/2);
    scaledPosition[1] = (canvas->getHeight()*pixelZoom) - (position[1]*pixelZoom) - (height*scales[1]/2);

    /*
     * If we are mirroring, mirror the positions too
     */
    scaledPosition[0] *= (mirrorMode==ISD_COMMENTBOX_MIRROR_HORIZONTAL ||
			  mirrorMode==ISD_COMMENTBOX_MIRROR_BOTH)?-1:1;
    scaledPosition[1] *= (mirrorMode==ISD_COMMENTBOX_MIRROR_VERTICAL ||
			  mirrorMode==ISD_COMMENTBOX_MIRROR_BOTH)?-1:1;

    /*
     * Depending on the mirrormode, the origin of the textbox is the top left,
     * or the mirrored (X axis) bottom left (if mirrored vertically).
     */
    float textplaneOrigin[3];
    switch (mirrorMode) {
    case ISD_COMMENTBOX_MIRROR_HORIZONTAL:
    case ISD_COMMENTBOX_MIRROR_NONE:
	textplaneOrigin[0] = textplaneTopLeft[0];
	textplaneOrigin[1] = textplaneTopLeft[1];
	textplaneOrigin[2] = textplaneTopLeft[2];
	break;
    case ISD_COMMENTBOX_MIRROR_BOTH:
    case ISD_COMMENTBOX_MIRROR_VERTICAL:
	textplaneOrigin[0] = textplaneTopLeft[0];
	textplaneOrigin[1] = -(textplaneTopLeft[1]-textplaneHeight);
	textplaneOrigin[2] = textplaneTopLeft[2];
	break;
    }

    /*
     * The origin of the textplane is different from the mesh' position.
     * The returned values in the variable textplaneTopLeft represent the values of the
     * top left point of the plane (so, not the center) in the scale, as represented
     * in memory (inside the centered unit cube).
     * The top left position of the XY-plane of the unit cube is (-width/2, height/2), since
     * the unit cube is centered on the origin.
     * By calculating the offset from that (top left unit cube) position to the origin of the
     * textplane, we obtain the offset-vector that can be added to the unit cube origin to reach
     * the textplane origin (inside the unit cube).
     * We must not forget to multiply the distance (vector) with the unitcube-to-real scale.
     */
    scaledTextplaneOrigin[0] = (position[0]*pixelZoom) + (textplaneOrigin[0]-(-width/2))*scales[0];
    scaledTextplaneOrigin[1] = (canvas->getHeight()-position[1])*pixelZoom + (textplaneOrigin[1]-(height/2))*scales[1];
    //Note: mesh is positioned at Z=0
    scaledTextplaneOrigin[2] = textplaneOrigin[2]*scales[2];
}

//-----PROTECTED METHODS-----
ISDObject::ISDErrorCode ISDCommentbox::initGL()
{
    //for transparency-support
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        
    //set-up lighting
    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);

    //make a backup of the light position
    glGetLightfv(GL_LIGHT0, GL_POSITION, lightPosition);

    GLfloat colBuf[4];
    colBuf[0] = 0.0;
    colBuf[1] = 0.0;
    colBuf[2] = 0.0;
    colBuf[3] = 1.0;
    glLightfv(GL_LIGHT0, GL_AMBIENT, colBuf);
    colBuf[0] = 1.0;
    colBuf[1] = 1.0;
    colBuf[2] = 1.0;
    colBuf[3] = 1.0;
    glLightfv(GL_LIGHT0, GL_DIFFUSE, colBuf);
    colBuf[0] = 1.0;
    colBuf[1] = 1.0;
    colBuf[2] = 1.0;
    colBuf[3] = 1.0;
    glLightfv(GL_LIGHT0, GL_SPECULAR, colBuf);

    colBuf[0] = 0.2;
    colBuf[1] = 0.2;
    colBuf[2] = 0.2;
    colBuf[3] = 1.0;
    glLightModelfv(GL_LIGHT_MODEL_AMBIENT, colBuf);

    //recalculate the surface-normals if we scale
    glEnable(GL_NORMALIZE);
    
    //read in some useful text-plane variables
    ISDGLObjFile* objFile = this->getObjFile();
    if (objFile==NULL || objFile->error()) {
	LOG_WARNING("Couldn't get the .obj class wrapper in the initGL method.");
	RETURN_ERROR(ISD_INIT_ERROR);
    }
    
    //read in the upper left corner
    if (objFile->getTextPlaneOrigin(textplaneTopLeft)!=ISD_SUCCESS) {
	LOG_WARNING("Error while fetching the textplane origin.");
	RETURN_ERROR(ISD_INIT_ERROR);
    }
    
    //read in width and height
    if (objFile->getTextPlaneDimensions(&textplaneWidth, &textplaneHeight)!=ISD_SUCCESS) {
	LOG_WARNING("Error while fetching the textplane dimensions.");
	RETURN_ERROR(ISD_INIT_ERROR);
    }

    RETURN_SUCCESS;
}
ISDObject::ISDErrorCode ISDCommentbox::reInitGL()
{
    initGL();
    recalcScales();
}
ISDObject::ISDErrorCode ISDCommentbox::initFont()
{
    font = new FTGLTextureFont(DEFAULT_FONT_FILE.c_str());
    if (font->Error()) {
	LOG_WARNING("Error while initialising font.");
	RETURN_ERROR(ISD_INIT_ERROR);
    }
    font->FaceSize(DEFAULT_FONT_FACE_SIZE);
    font->CharMap(ft_encoding_unicode);
    
    RETURN_SUCCESS;
}
void ISDCommentbox::displayModel()
{
    ISDGLObjFile* objFile = this->getObjFile();
    if (objFile==NULL || objFile->error()) {
	LOG_WARNING("Couldn't get the .obj class wrapper.");
	return;
    }
    
    GLuint meshObjList = objFile->getMeshObjectList();
    if (meshObjList==0) {
	LOG_WARNING("Error while fetching the main mesh display list in display method.");
	return;
    }
    
    glCallList(meshObjList);
}
void ISDCommentbox::displayText()
{
    float maxWidth;
    
    int line = 0;
    float x1, y1, z1, x2, y2, z2;
    //we want the first line to come one 'ascend' down,
    //because we want the TOP of the line to be aligned with the textboxOrigin
    GLfloat lineHeight = font->Ascender();

    //first, we split on newlines
    list<Glib::ustring> newlines = ISDUtils::getInstance()->explodeString(this->text, NEWLINE_DELIMITER);
    list<Glib::ustring>::iterator newlineIter;
    list<Glib::ustring>::iterator horLineIter;

    /*
     * First, we calc the total height of the text, in a quick and dirty pre-loop.
     * Note: This really should be optimized into one loop!!
     */
    float totalLines = 0;
    for(newlineIter=newlines.begin();newlineIter!=newlines.end();newlineIter++, line++) {
	totalLines += splitTextHorizontally((*newlineIter), &maxWidth).size();
    }

    //this dumps down the text with one line (seems to be better)
    totalLines--;
    
    totalLines *= (lineHeight+LINE_SPACER);
    
    glTranslatef(0.0, -((textplaneHeight*scales[1])-totalLines)/2.0, 0.0);

    for(newlineIter=newlines.begin();newlineIter!=newlines.end();newlineIter++, line++) {
	//now, split horizontally
	list<Glib::ustring> horLines = splitTextHorizontally((*newlineIter), &maxWidth);
	
	//we calculate the height of the text, and center vertically
	/*
	 * TODO: Caused problems with not-vertically aligned textplanes...
	 */
	//float linesHeight = horLines.size() * (lineHeight+LINE_SPACER);
	//glTranslatef(0.0, -((textplaneHeight*scales[1])-linesHeight)/2.0, 0.0);
	
	for(horLineIter=horLines.begin();horLineIter!=horLines.end();horLineIter++, line++) {
	    char* theLine = (char*)(*horLineIter).c_str();	    
	    glTranslatef(0.0, -(lineHeight+LINE_SPACER), 0.0);
	    glPushMatrix();
	        font->Render(theLine);
	    glPopMatrix();
	}
    }
}
void ISDCommentbox::getScalesTweakValues(float values[3])
{
    values[0] = 1.0;
    values[1] = 1.0;
    values[2] = 1.0;
}
list<Glib::ustring> ISDCommentbox::splitTextHorizontally(Glib::ustring text, float* maxWidth)
{
    float x1, y1, z1, x2, y2, z2;
    list<Glib::ustring> retVal;
    Glib::ustring line = "";
    bool firstWord = true;
    
    float mWidth = 0;
    float lineWidth;

    Glib::ustring delim = " ";
    list<Glib::ustring> words = ISDUtils::getInstance()->explodeString(text, delim);

    float pixelZoom;
    glGetFloatv(GL_ZOOM_X, &pixelZoom);

    list<Glib::ustring>::iterator iter;
    for(iter=words.begin();iter!=words.end();iter++) {
	Glib::ustring word = (*iter);
	font->BBox((line+" "+word).c_str(), x1, y1, z1, x2, y2, z2);
	float xMax = x1>x2?x1:x2;
	float xMin = x1<x2?x1:x2;
	
	lineWidth = (xMax-xMin)*FONT_SCALE_SIZE*pixelZoom;
	
	if (lineWidth<=textplaneWidth*scales[0]) {
	    //keep the maximum line-width
	    if (lineWidth > mWidth){
		mWidth = lineWidth;
	    }
	    
	    //don't append a space to the first word
	    if (firstWord)
		firstWord = false;
	    else
		line.append(" ");
	    line.append(word);
	}
	else {
	    retVal.push_back(line);
	    
	    line = word;
	}
    }
    retVal.push_back(line);
    
    if (maxWidth!=NULL)
	*maxWidth = mWidth;
    
    return retVal;
}
