/*
    Copyright (C) 2000 Steve Brown
    Copyright (C) 2000,2001,2002 Guillaume Morin, Alcve

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


    $Id: passwd.c,v 1.106 2003/05/12 14:59:18 gmorin Exp $
    $Source: /cvsroot/nss-mysql/nss-mysql/src/passwd.c,v $
    $Date: 2003/05/12 14:59:18 $
    $Author: gmorin $
*/


#include <stdlib.h>
#include <pwd.h>
#include <string.h>
#include <errno.h>

#include "nss-mysql.h"

#include "lib.h"
#include "parser.h"
#include "passwd.h"
#include "ent.h"

#define FALLBACK_GID 65534 /* if the gid column can't be read, 
                              fall back to this GID. should be nogroup */
#define FALLBACK_UID 65534

#define FALLBACK_TMP "/tmp"
#define FALLBACK_SHELL "/bin/sh"

NSS_STATUS _nss_mysql_getpwuid_r(uid_t,struct passwd *,char *, size_t,int *);
NSS_STATUS _nss_mysql_setpwent (void);
NSS_STATUS _nss_mysql_getpwent_r (struct passwd *pw, char * buffer, size_t buflen,int * errnop); 
NSS_STATUS _nss_mysql_endpwent (void);
NSS_STATUS _nss_mysql_getpwnam_r(const char *,struct passwd *,char *,size_t,int *);

static void prepare_forkhandler(void);
static void parent_forkhandler(void);
static void child_forkhandler(void);
static void atexit_handler(void);
static NSS_STATUS handle_query(struct query * q,struct mysql_auth * mysql_auth,
                struct passwd * pw, char * buffer, size_t buflen,int * errnop);
static char * get_sql_query(struct query * q,struct parse_result * pr,
                struct mysql_auth * auth);

/* EOP */

static int forkhandler_isset = 0;
static pthread_mutex_t forkhandler_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;

/* get_sql_query:
 * 
 * Returns a complete SQL query string corresponding to the query
 * 
 * q: request:
 * pr: value gathered by the conf files parsing
 * auth: connection information
 */
char * get_sql_query(struct query * q,struct parse_result * pr,
                struct mysql_auth * auth) {
        struct passwdoptions * options = pr->options_p;
        char * sql = NULL;
        char * e_name=NULL;
        char buffer[64];
        int call_free;

        if (DEBUG) {
                if (q->name) {
                        /* looking by name */
                        _nss_mysql_log(LOG_ERR,"passwd.c: get_query "
                                        "called for user %s",q->name);
                } else {
                        /* looking by uid */
                        _nss_mysql_log(LOG_ERR,"passwd.c: get_query "
                                        "called for uid %d",q->uid);
                }
        }
        
        /* escaping name */
        if (q->name) {
                e_name = _nss_mysql_escape_string(q->name,auth,buffer,
                                sizeof buffer, &call_free);
                if (! e_name)
                        return NULL;
        }

        if (q->name) {
                /* look up by name */
                sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s,%s,%s,%s from %s"
                               " where %s='%s' and %s is not null and %s order"
                               " by %s",
                        /* select */
                        options->usercolumn,
                        options->uidcolumn,
#if ! USE_SHADOW
                        options->passwdcolumn,
#else
                        "NULL",
#endif
                        options->realnamecolumn,
                        options->shellcolumn,
                        options->homedircolumn,
                        options->gidcolumn,
                        /* from */
                        options->table,
                        /* where */
                        options->usercolumn,
                        /* = */
                        e_name,
                        /* and */
                        options->uidcolumn,
                        /* is not null and */
                        options->where[0] ? options->where : "1=1",
                        options->uidcolumn);

                if (call_free)
                        free(e_name);

        } else {
                /* look up by uid */
                sql = _nss_mysql_sqlprintf("select %s,%s,%s,%s,%s,%s,%s from %s"
                                " where %s=%d and %s order by %s",
                        /* select */
                        options->usercolumn,
                        options->uidcolumn,
#if ! USE_SHADOW
                        options->passwdcolumn,
#else
                        "NULL",
#endif
                        options->realnamecolumn,
                        options->shellcolumn,
                        options->homedircolumn,
                        options->gidcolumn,
                        /* from */
                        options->table,
                        /* where */
                        /* if bulk is true, we retrieve every row */
                        q->bulk ? "1" : options->uidcolumn,
                        /* = */
                        q->bulk ? 1 : q->uid,
                        /* and */
                        options->where[0] ? options->where : "1=1",
                        options->uidcolumn
                        );
        }

        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: get_query: SQL statement is "
                                "%s",sql);
        
        return sql;
}

/* handle_query:
 *
 * generate a sql query, send it, parse the result and store it in pw according
 * to the nss-mysql query "q"
 *
 * q: query to handle
 * mysql_auth: connection information
 * pw: to be filled
 * buffer: buffer to save dynamic data
 * buflen: sizeof buffer
 * errnop: pointer to errno
 */

NSS_STATUS handle_query(struct query * q,struct mysql_auth * mysql_auth,
                        struct passwd * pw, char * buffer, size_t buflen,
                        int * errnop) {
        struct passwdoptions options;
        struct parse_result pr = {&options,NULL,NULL};
        NSS_STATUS status;
        MYSQL_RES * result = NULL;
        char * sql_query = NULL;
        /*int call_mte = 0;*/

        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: handle_query called for "
                                "name %s, uid %d, bulk %d",q->name,q->uid,
                                q->bulk);
        
        memset(&options,0,sizeof(struct passwdoptions));

        if (! _nss_mysql_read_conf_file("users",&pr)) {
                _nss_mysql_log(LOG_ERR,"passwd.c: handle_query: conf file "
                                "parsing failed");
                status = NSS_STATUS_UNAVAIL;
                *errnop = ENOENT;
                goto out;
        }

        _nss_mysql_set_fork_handler(&forkhandler_isset,&forkhandler_mutex,
                        prepare_forkhandler,
                        parent_forkhandler,
                        child_forkhandler,
                        atexit_handler);

        if (! _nss_mysql_check_connection(mysql_auth,&options.con)) {
                status = NSS_STATUS_UNAVAIL;
                *errnop = ENOENT;
                goto out;
        }
       
        /*call_mte = 1;*/
        sql_query = get_sql_query(q,&pr,mysql_auth);
        if (! sql_query) {
                *errnop = EAGAIN;
                status = NSS_STATUS_TRYAGAIN;
                /* we need to unlock the mutex because 
                 * _nss_mysql_check_connection locks it
                 */
                pthread_mutex_unlock(mysql_auth->mutex);
                goto out;
        }
        
        status = _nss_mysql_send_query(mysql_auth,sql_query,&result,errnop);
        if (status != NSS_STATUS_SUCCESS)
                goto out;

        status = _nss_mysql_passwd_result_to_struct(pw,result,errnop,buffer,
                        buflen,1);
out:
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: handle_query(): reached "
                                "\"out\" label, result %p",result);
        s_free(sql_query);
        if (result)
                mysql_free_result(result);
        _nss_mysql_free_users(&options);
        /*
        if (call_mte)
                my_thread_end();
        */
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: leaving handle_query with "
                                "status %d (errno %s)",status,
                                strerror(*errnop));
        return status;
}

/* fills a struct according to the MySQL result
 *
 * pw: to be filled
 * result: the MySQL result
 * errnop: pointer to errno
 * buffer: we'll allocate strings here
 * buflen: size of buffer
 * max: maximum numbers of rows in result, ignored if 0
 */
NSS_STATUS _nss_mysql_passwd_result_to_struct(struct passwd *pw,
                MYSQL_RES * result,int *errnop, char * buffer, size_t buflen, 
                unsigned long max) {
        unsigned long rows;
        int i;
        MYSQL_ROW sql_row;
        MYSQL_ROW_OFFSET initial_offset = mysql_row_tell(result);
        
        *errnop = ENOENT;
        rows = mysql_num_rows(result);
        
        if (rows == 0) {
                if  (DEBUG) _nss_mysql_log(LOG_ERR,"User not found...");
                return NSS_STATUS_NOTFOUND;
        }

        if (max && rows > max) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct:"
                                "Warning: lookup returned %lu rows, I "
                                "was expecting just %lu.",rows,max);
        }

        sql_row = mysql_fetch_row(result);
        if (! sql_row) {
                if (DEBUG) 
                        _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_"
                                        "struct: mysql_fetch_row has failed."
                                        " End of retrieval");
                /* the end of a bulk retrieval */
                return NSS_STATUS_NOTFOUND;
        }
                                        
        /* sanity checks */
        if (_nss_mysql_isempty(sql_row[0])) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct: "
                                "NULL or empty username. Fix your "
                                "database");
                return NSS_STATUS_UNAVAIL;
        }
                
#if ! USE_SHADOW
        if (_nss_mysql_isempty(sql_row[2])) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct: "
                                "NULL or empty password for %s. "
                                "Fix your database",sql_row[0]);
                return NSS_STATUS_UNAVAIL;
        }
#endif
                
        /* name is stored */
        pw->pw_name = _nss_mysql_copy_to_buffer(&buffer,&buflen,
                                sql_row[0]);
        if (! pw->pw_name)
                goto out_nomem;

        if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct"
                        ": username == %s", sql_row[0]);
                
        /* storing password */
#if ! USE_SHADOW
        pw->pw_passwd = _nss_mysql_copy_to_buffer(&buffer,&buflen,sql_row[2]);
        if (! pw->pw_passwd)
                goto out_nomem;
        if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct"
                        ": password == %s", sql_row[2]);
#else
        pw->pw_passwd = _nss_mysql_copy_to_buffer(&buffer,&buflen,"x");
        if (! pw->pw_passwd)
                goto out_nomem;
#endif
                
        /* storing uid */
        pw->pw_uid = _nss_mysql_strtol(sql_row[1],FALLBACK_UID,&i);
        if (i) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct: "
                                "User %s has invalid uid(%s). "
                                "Reverted to %d. Fix your database.", 
                                pw->pw_name, sql_row[1]||"NULL",
                                pw->pw_uid);
        }
        if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct"
                        ": uid == %d", pw->pw_uid);

        /* check if a GID was returned */
        pw->pw_gid = _nss_mysql_strtol(sql_row[6],FALLBACK_GID,&i);
        if (i) {
                _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct: "
                                "User %s has invalid gid(%s). "
                                "Reverted to %d. Fix your database.",
                                pw->pw_name, sql_row[6],FALLBACK_GID);
        }
        if (DEBUG) _nss_mysql_log(LOG_ERR, "_nss_mysql_passwd_result_to_struct"
                        ": gid == %d", pw->pw_gid);
        
        /* Realname, empty realname is valid, so no warning */
        pw->pw_gecos = _nss_mysql_copy_to_buffer(&buffer,&buflen,
                        sql_row[3] ? sql_row[3] : "");
        if (! pw->pw_gecos)
                goto out_nomem;

        /* home directory */
        if (_nss_mysql_isempty(sql_row[5]))	{
                _nss_mysql_log(LOG_ERR,"Empty or NULL home column for "
                                "user %s(%d). Falling back to " FALLBACK_TMP 
                                ". Fix your database.",
                                pw->pw_name,pw->pw_uid);

                pw->pw_dir =  _nss_mysql_copy_to_buffer(&buffer,&buflen,
                                         FALLBACK_TMP);
        } else {
                pw->pw_dir =  _nss_mysql_copy_to_buffer(&buffer,&buflen,
                                        sql_row[5]);
        }
        if (! pw->pw_dir)
                goto out_nomem;

        /* shell */
        if (_nss_mysql_isempty(sql_row[4])) {
                _nss_mysql_log(LOG_ERR,"Empty or NULL shell column for "
                                "user %s(%d). Falling back to " FALLBACK_SHELL 
                                ". Fix your database.",
                                pw->pw_name,pw->pw_uid);
                pw->pw_shell =  _nss_mysql_copy_to_buffer(&buffer,&buflen,
                                FALLBACK_SHELL);
        } else {
                pw->pw_shell =  _nss_mysql_copy_to_buffer(&buffer,&buflen,
                                sql_row[4]);
        }
        
        if (! pw->pw_shell)
                goto out_nomem;
        *errnop = 0;
        if (DEBUG) _nss_mysql_log(LOG_ERR,"_nss_mysql_passwd_result_to_struct"
                        " finished sucessfully");
                
        return NSS_STATUS_SUCCESS;
        
out_nomem:
        /* if we're here, that means that the buffer is too small, so
         * we return ERANGE
         */
        *errnop = ERANGE;
        /* the current row will be processed again, so we go back */
        mysql_row_seek(result,initial_offset);
        return NSS_STATUS_TRYAGAIN;
}


/* setpwent
 * Initializes data for ...pwent functions
 * NOTE this function does _NOT_ use errno
 */

NSS_STATUS _nss_mysql_setpwent (void) {
        return _nss_mysql_setent(PASSWD_ENT_TYPE,get_sql_query);
}


/* endpwent
 * Kills all data for ...pwent functions
 * NOTE this function does _NOT_ use errno
 */
NSS_STATUS _nss_mysql_endpwent (void) {
        return _nss_mysql_endent(PASSWD_ENT_TYPE);
}

/* getpwent
 * Gets info for every group
 * Arguments:
 * pw: array to fill
 * buffer: buffer, unused 
 * buflen: size of buffer
 * errnop: ptr to the application errno
 */

NSS_STATUS _nss_mysql_getpwent_r (struct passwd *pw,
                char * buffer, size_t buflen,int * errnop) {
        return _nss_mysql_getent_r(PASSWD_ENT_TYPE,(void *) pw,buffer,buflen,errnop);
}


#define getpw_quit() \
_nss_mysql_free_users(&options); 

/* getpwnam
 * looks for an user by its name
 * Arguments:
 * uid: user's uid
 * result: struct we'll fill
 * buffer:
 * buflen: sizeof(buffer)
 * errnop: ptr on the application errno
 */


static pthread_mutex_t nam_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;
static struct mysql_auth nam_auth = {NULL,0,&nam_mutex};

NSS_STATUS _nss_mysql_getpwnam_r (const char *name, struct passwd * pw,
                char *buffer, size_t buflen, int *errnop) {
        struct query q = {name,0,0,0};
        return handle_query(&q,&nam_auth,pw,buffer,buflen,errnop);
}

/* getpwuid
 * looks for an user by uid
 * Arguments:
 * uid: user's uid
 * result: struct we'll fill
 * buffer: 
 * buflen: sizeof(buffer)
 * errnop: ptr on the application errno
 */


static pthread_mutex_t uid_mutex = NSS_MYSQL_PTHREAD_MUTEX_INITIALIZER;
static struct mysql_auth uid_auth = {NULL,0,&uid_mutex};

NSS_STATUS _nss_mysql_getpwuid_r (uid_t uid, struct passwd *pw, 
        char *buffer, size_t buflen, int *errnop) {
        struct query q = {NULL,uid,0,0};
        return handle_query(&q,&uid_auth,pw,buffer,buflen,errnop);
}

void prepare_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: calling prepare_forkhandler");
        pthread_mutex_lock(nam_auth.mutex);
        pthread_mutex_lock(uid_auth.mutex);
        pthread_mutex_lock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: end of prepare_forkhandler");
}

void parent_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: calling parent_forkhandler");
        pthread_mutex_unlock(nam_auth.mutex);
        pthread_mutex_unlock(uid_auth.mutex);
        pthread_mutex_unlock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: end of parent_forkhandler");
}

void child_forkhandler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: calling child_forkhandler");
        uid_auth.pid = 0;
        nam_auth.pid = 0;
        pthread_mutex_unlock(nam_auth.mutex);
        pthread_mutex_unlock(uid_auth.mutex);
        pthread_mutex_unlock(&forkhandler_mutex);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: end of child_forkhandler");
}

void atexit_handler(void) {
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: calling atexit_handler");
        pthread_mutex_lock(nam_auth.mutex);
        _nss_mysql_db_close(&nam_auth.mysql);
        pthread_mutex_lock(uid_auth.mutex);
        _nss_mysql_db_close(&uid_auth.mysql);
        if (DEBUG)
                _nss_mysql_log(LOG_ERR,"passwd.c: end of atexit_handler");
}
/* EOF */
