/* EnhancedTableHeader.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.tables;

import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.util.List;
import javax.swing.event.MouseInputListener;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableColumnModel;

/**
 * Special table header for use with {@link EnhancedTableHeader}. Handles mouse
 * events and uses a special {@link EnhancedTableCellRenderer} for rendering.
 */
class EnhancedTableHeader extends JTableHeader implements MouseInputListener {
    
    //
    static final int SPACING = 4; // TODO: merge with SPACING in EnhancedTableRenderer
    
    /** Default constructor */
    public EnhancedTableHeader (TableColumnModel columnModel) {
        super (columnModel);
        setDefaultRenderer (new EnhancedTableCellRenderer ());
        addMouseListener (this);
        addMouseMotionListener (this);
        mouseDecoration = null;
        selectedDecoration = null;
    }
    
    
    
    // current decoration of the mouse pointer
    private TableColumnDecoration mouseDecoration;
    
    // decoration currently selected
    private TableColumnDecoration selectedDecoration;
    
    //
    public void mouseMoved (MouseEvent e) {
        TableColumnDecoration deco = mouseToDecoration (e);
        if (deco != mouseDecoration) { // NOPMD
            if (mouseDecoration != null)
                mouseDecoration.mouseExited ();
            mouseDecoration = deco;
            if (mouseDecoration != null)
                mouseDecoration.mouseEntered ();
            repaint ();
        }
    }

    //
    public void mouseEntered (MouseEvent e) {
        TableColumnDecoration deco = mouseToDecoration (e);
        mouseDecoration = deco;
        if (mouseDecoration != null)
            mouseDecoration.mouseEntered ();
        repaint ();
    }
    
    //
    public void mouseExited (MouseEvent e) {
        if (mouseDecoration != null)
            mouseDecoration.mouseExited ();
        mouseDecoration = null;
        repaint ();
    }
    
    //
    public void mouseClicked (MouseEvent e) {
        // trivial implementation
    }
    
    
    public void mouseReleased (MouseEvent e) {
        if (selectedDecoration != null) {
            selectedDecoration.mouseReleased ();
            selectedDecoration = null;
            repaint ();
            e.consume (); // [kc] has no effect... ?
        }
    }
    
    public void mousePressed (MouseEvent e) {
        selectedDecoration = mouseDecoration;
        if (selectedDecoration != null) {
            selectedDecoration.mousePressed ();
            repaint ();
            e.consume (); // [kc] has no effect... ?
        }
    }
        
    public void mouseDragged (MouseEvent e) {
        mouseMoved (e);
        if (selectedDecoration != null)
            e.consume (); // [kc] has no effect... ?
    }
    
    
    //
    private TableColumnDecoration mouseToDecoration (MouseEvent e) {
        int viewIndex = columnAtPoint (e.getPoint ());
        if (viewIndex < 0)
            return null;
        int modelIndex = columnModel.getColumn (viewIndex).getModelIndex ();
        List<TableColumnDecoration> list 
            = ((EnhancedTable)table).getDecorations (modelIndex);
        if (list == null)
            return null;
        Rectangle rect = getHeaderRect (viewIndex);
        
        int x = rect.x + rect.width;
        int mouseX = e.getX ();
        for (TableColumnDecoration deco : list) {
            x -= deco.getDecorationWidth () + SPACING;
            if (mouseX >= x && mouseX < x + deco.getDecorationWidth ()) {
                int y = (rect.height - deco.getDecorationHeight ()) / 2;
                int mouseY = e.getY () - y - rect.y;
                if (mouseY >= 0 && mouseY < deco.getDecorationHeight ())
                    return deco;
                else
                    return null;
            }
        }
        return null;
    }
}
