/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */
 
#include "support.h"

#include <libsyncml/sml_transport_internals.h>

#include <libsyncml/transports/http_client.h>
#include <libsyncml/transports/http_server.h>

START_TEST (http_client_new)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *tsp = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	fail_unless(tsp != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportFree(tsp);
}
END_TEST

START_TEST (http_client_init)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *tsp = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	fail_unless(tsp != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlTransportHttpClientConfig config;
	config.port = 80;
	config.url = "test.com";
	config.proxy = NULL;
	
	fail_unless(smlTransportInitialize(tsp, &config, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportFinalize(tsp, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportFree(tsp);
}
END_TEST

START_TEST (http_client_wrong_port)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *tsp = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	fail_unless(tsp != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlTransportHttpClientConfig config;
	config.port = -1;
	config.url = "test.com";
	config.proxy = NULL;
	
	fail_unless(!smlTransportInitialize(tsp, &config, &error), NULL);
	fail_unless(error != NULL, NULL);
	
	smlErrorDeref(&error);
	
	smlTransportFree(tsp);
}
END_TEST

START_TEST (http_client_init_no_url)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *tsp = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	fail_unless(tsp != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlTransportHttpClientConfig config;
	config.port = 80;
	config.url = NULL;
	config.proxy = NULL;
	
	fail_unless(!smlTransportInitialize(tsp, &config, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlTransportFree(tsp);
}
END_TEST

START_TEST (http_server_new)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *tsp = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	fail_unless(tsp != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportFree(tsp);
}
END_TEST

START_TEST (http_server_init)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *tsp = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	fail_unless(tsp != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlTransportHttpServerConfig config;
	config.port = 32425;
	config.url = "/test";
	config.interface = NULL;
	config.ssl_key = NULL;
	config.ssl_crt = NULL;
	
	fail_unless(smlTransportInitialize(tsp, &config, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportFinalize(tsp, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportFree(tsp);
}
END_TEST

START_TEST (http_server_wrong_port)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *tsp = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	fail_unless(tsp != NULL, NULL);
	fail_unless(error == NULL, NULL);
	
	SmlTransportHttpServerConfig config;
	config.port = -1;
	config.url = NULL;
	config.interface = NULL;
	config.ssl_key = NULL;
	config.ssl_crt = NULL;
	
	fail_unless(!smlTransportInitialize(tsp, &config, &error), NULL);
	fail_unless(error != NULL, NULL);
	smlErrorDeref(&error);
	
	smlTransportFree(tsp);
}
END_TEST

int client_connect_done = 0;
int client_disconnect_done = 0;
int client_receives = 0;
int client_errors = 0;

SmlBool _recv_client_event(SmlTransport *tsp, SmlLink *link, SmlTransportEventType type, SmlTransportData *data, SmlError *error, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s()", __func__);
	
	fail_unless(GPOINTER_TO_INT(userdata) == 1);
	
	switch (type) {
		case SML_TRANSPORT_EVENT_CONNECT_DONE:
			g_atomic_int_inc(&client_connect_done);
			break;
		case SML_TRANSPORT_EVENT_DISCONNECT_DONE:
			g_atomic_int_inc(&client_disconnect_done);
			break;
		case SML_TRANSPORT_EVENT_DATA:
			fail_unless(!strcmp(data->data, "answer"), NULL);
			fail_unless(data->size == 7, NULL);
			fail_unless(data->type == SML_MIMETYPE_WBXML, NULL);
			g_atomic_int_inc(&client_receives);
			break;
		case SML_TRANSPORT_EVENT_ERROR:
			fail_unless(error != NULL, NULL);
			g_atomic_int_inc(&client_errors);
			break;
		default:
			fail(NULL);
	}
	
	smlTrace(TRACE_EXIT, "%s()", __func__);
	return TRUE;
}

unsigned server_receives = 0;

SmlBool _recv_server_event(SmlTransport *tsp, SmlLink *link, SmlTransportEventType type, SmlTransportData *data, SmlError *error, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s()", __func__);
		
	switch (type) {
		case SML_TRANSPORT_EVENT_DATA:
			server_receives++;
			
			if (!strcmp(data->data, "test")) {
				fail_unless(data->size == 5, NULL);
				fail_unless(data->type == SML_MIMETYPE_XML, NULL);
				fail_unless(link != NULL, NULL);
				
				if (GPOINTER_TO_INT(userdata) == 1) {
					data = smlTransportDataNew("answer", 7, SML_MIMETYPE_WBXML, FALSE, &error);
					fail_unless(data != NULL, NULL);
					fail_unless(error == NULL, NULL);
				
					fail_unless(smlTransportSend(tsp, link, data, NULL), NULL);
					
					smlTransportDataDeref(data);
				} else if (GPOINTER_TO_INT(userdata) == 2) {
					SmlError *newerror = NULL;
					smlErrorSet(&newerror, SML_ERROR_GENERIC, "test");
					smlTransportSetError(tsp, link, &newerror);
					smlErrorDeref(&newerror);
				} else
					fail(NULL);
			} else if (!strcmp(data->data, "error")) {
				fail_unless(data->size == 6, NULL);
				fail_unless(data->type == SML_MIMETYPE_XML, NULL);
				fail_unless(link != NULL, NULL);
				
				SmlError *newerror = NULL;
				smlErrorSet(&newerror, SML_ERROR_GENERIC, "test2");
				smlTransportSetError(tsp, link, &newerror);
				smlErrorDeref(&newerror);
			} else
				fail(NULL);
			break;
		default:
			fail(NULL);
	}
	
	smlTrace(TRACE_EXIT, "%s()", __func__);
	return TRUE;
}

START_TEST (http_connect)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10008;
	clientConfig.url = "http://127.0.0.1:10008";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10008;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(client, _recv_client_event, GINT_TO_POINTER(1));

	fail_unless(smlTransportConnect(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (client_connect_done == 0) { usleep(50); };
	
	fail_unless(client_connect_done == 1, NULL);
	fail_unless(client_disconnect_done == 0, NULL);
	
	fail_unless(smlTransportDisconnect(client, NULL, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	while (client_disconnect_done == 0) { usleep(50); };
	
	fail_unless(client_connect_done == 1, NULL);
	fail_unless(client_disconnect_done == 1, NULL);
	
	smlTransportStop(client);
	smlTransportStop(server);

	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (http_send)
{
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10009;
	clientConfig.url = "http://127.0.0.1:10009";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10009;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(client, _recv_client_event, GINT_TO_POINTER(1));

	SmlTransportData *data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportDataDeref(data);
	
	smlTransportStop(client);
	smlTransportStop(server);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (http_receive)
{
	client_connect_done = 0;
	client_disconnect_done = 0;
	client_receives = 0;
	client_errors = 0;
	server_receives = 0;

	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10010;
	clientConfig.url = "http://127.0.0.1:10010";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10010;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(server, _recv_server_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(client, _recv_client_event, GINT_TO_POINTER(1));

	SmlTransportData *data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportDataDeref(data);
	
	while (server_receives != 1) {
		usleep(50000);
	}
	
	fail_unless(server_receives == 1, NULL);
	
	smlTransportStop(client);
	smlTransportStop(server);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (http_reply)
{
	client_connect_done = 0;
	client_disconnect_done = 0;
	client_receives = 0;
	client_errors = 0;
	server_receives = 0;
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10011;
	clientConfig.url = "http://127.0.0.1:10011";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10011;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(client, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(server, _recv_server_event, GINT_TO_POINTER(1));

	SmlTransportData *data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	smlTransportDataDeref(data);
	
	while (1) {
		usleep(100);
		if (client_receives == 1)
			break;
	}
	
	fail_unless(client_receives == 1, NULL);
	fail_unless(server_receives == 1, NULL);
	
	smlTransportStop(client);
	smlTransportStop(server);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (http_talk)
{
	client_connect_done = 0;
	client_disconnect_done = 0;
	client_receives = 0;
	client_errors = 0;
	server_receives = 0;
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10012;
	clientConfig.url = "http://127.0.0.1:10012";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10012;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(client, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(server, _recv_server_event, GINT_TO_POINTER(1));
	
	SmlTransportData *data = NULL;
	
	int i;
	for (i = 0; i < 20; i++) {
		data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
		fail_unless(data != NULL, NULL);
		fail_unless(error == NULL, NULL);
	
		fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
		fail_unless(error == NULL, NULL);
		
		smlTransportDataDeref(data);
		
		while (1) {
			usleep(100);
			if (client_receives == i+1)
				break;
		}
	}
	
	fail_unless(client_receives == 20, NULL);
	fail_unless(server_receives == 20, NULL);
	
	smlTransportStop(client);
	smlTransportStop(server);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client);
}
END_TEST

START_TEST (http_connect_error)
{
	client_connect_done = 0;
	client_disconnect_done = 0;
	client_receives = 0;
	client_errors = 0;
	server_receives = 0;
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10013;
	clientConfig.url = "http://127.0.0.1:10013";
	clientConfig.proxy = NULL;

	fail_unless(smlTransportRunAsync(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);

	smlTransportSetEventCallback(client, _recv_client_event, GINT_TO_POINTER(1));

	SmlTransportData *data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);

	smlTransportDataDeref(data);
		
	while (client_errors == 0) { usleep(100); };
	
	fail_unless(client_connect_done == 0, NULL);
	fail_unless(client_disconnect_done == 0, NULL);
	fail_unless(client_receives == 0, NULL);
	fail_unless(client_errors == 1, NULL);
	
	smlTransportStop(client);
	
	fail_unless(smlTransportFinalize(client, &error), NULL);
	
	smlTransportFree(client);
}
END_TEST

START_TEST (http_reject)
{
	client_connect_done = 0;
	client_disconnect_done = 0;
	client_receives = 0;
	client_errors = 0;
	server_receives = 0;
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10014;
	clientConfig.url = "http://127.0.0.1:10014";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10014;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(client, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(server, _recv_server_event, GINT_TO_POINTER(2));

	SmlTransportData *data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);

	fail_unless(smlTransportSend(client, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);

	smlTransportDataDeref(data);
	
	while (server_receives == 0) { usleep(100); };
	
	fail_unless(server_receives == 1, NULL);
					
	while (client_errors == 0) { usleep(100); };
	
	fail_unless(client_connect_done == 0, NULL);
	fail_unless(client_disconnect_done == 0, NULL);
	fail_unless(client_receives == 0, NULL);
	fail_unless(client_errors == 1, NULL);
	
	smlTransportStop(client);
	smlTransportStop(server);
	
	fail_unless(smlTransportFinalize(client, &error), NULL);
	fail_unless(smlTransportFinalize(server, &error), NULL);
	
	smlTransportFree(client);
	smlTransportFree(server);
}
END_TEST

START_TEST (http_multi_connect)
{
	client_connect_done = 0;
	client_disconnect_done = 0;
	client_receives = 0;
	client_errors = 0;
	server_receives = 0;
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client1 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	SmlTransport *client2 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	SmlTransport *client3 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10015;
	clientConfig.url = "http://127.0.0.1:10015";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10015;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client1, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(client2, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(client3, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client1, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(client2, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(client3, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(client1, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(client2, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(client3, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(server, _recv_server_event, GINT_TO_POINTER(1));
	
	SmlTransportData *data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportSend(client1, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlTransportDataDeref(data);
	
	data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportSend(client2, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlTransportDataDeref(data);
	
	data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportSend(client3, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlTransportDataDeref(data);
	
	while (client_receives != 3) {
		usleep(100);
	}
	
	fail_unless(client_receives == 3, NULL);
	fail_unless(server_receives == 3, NULL);
	
	smlTransportStop(client1);
	smlTransportStop(client2);
	smlTransportStop(client3);
	smlTransportStop(server);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client1, &error), NULL);
	fail_unless(smlTransportFinalize(client2, &error), NULL);
	fail_unless(smlTransportFinalize(client3, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client1);
	smlTransportFree(client2);
	smlTransportFree(client3);
}
END_TEST

START_TEST (http_multi_partial_error)
{
	client_connect_done = 0;
	client_disconnect_done = 0;
	client_receives = 0;
	client_errors = 0;
	server_receives = 0;
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client1 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	SmlTransport *client2 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	SmlTransport *client3 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10016;
	clientConfig.url = "http://127.0.0.1:10016";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10016;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client1, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(client2, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(client3, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client1, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(client2, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(client3, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(client1, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(client2, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(client3, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(server, _recv_server_event, GINT_TO_POINTER(1));
	
	SmlTransportData *data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportSend(client1, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlTransportDataDeref(data);
	
	data = smlTransportDataNew("error", 6, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportSend(client2, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlTransportDataDeref(data);
	
	
	data = smlTransportDataNew("error", 6, SML_MIMETYPE_XML, FALSE, &error);
	fail_unless(data != NULL, NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportSend(client3, NULL, data, &error), NULL);
	fail_unless(error == NULL, NULL);
	smlTransportDataDeref(data);
	
	while (client_receives != 1 || client_errors != 2) {
		usleep(100);
	}
	
	fail_unless(client_receives == 1, NULL);
	fail_unless(client_errors == 2, NULL);
	fail_unless(server_receives == 3, NULL);
	
	smlTransportStop(client1);
	smlTransportStop(client2);
	smlTransportStop(client3);
	smlTransportStop(server);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client1, &error), NULL);
	fail_unless(smlTransportFinalize(client2, &error), NULL);
	fail_unless(smlTransportFinalize(client3, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client1);
	smlTransportFree(client2);
	smlTransportFree(client3);
}
END_TEST

START_TEST (http_multi_stress)
{
	int num = 1000;
	
	client_connect_done = 0;
	client_disconnect_done = 0;
	client_receives = 0;
	client_errors = 0;
	server_receives = 0;
	setup_testbed(NULL);
	
	SmlError *error = NULL;
	SmlTransport *server = smlTransportNew(SML_TRANSPORT_HTTP_SERVER, &error);
	SmlTransport *client1 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	SmlTransport *client2 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	SmlTransport *client3 = smlTransportNew(SML_TRANSPORT_HTTP_CLIENT, &error);
	
	SmlTransportHttpClientConfig clientConfig;
	clientConfig.port = 10017;
	clientConfig.url = "http://127.0.0.1:10017";
	clientConfig.proxy = NULL;
	
	SmlTransportHttpServerConfig serverConfig;
	serverConfig.port = 10017;
	serverConfig.url = NULL;
	serverConfig.interface = NULL;
	serverConfig.ssl_key = NULL;
	serverConfig.ssl_crt = NULL;

	fail_unless(smlTransportRunAsync(client1, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(client2, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(client3, &error), NULL);
	fail_unless(error == NULL, NULL);
	fail_unless(smlTransportRunAsync(server, &error), NULL);
	fail_unless(error == NULL, NULL);
	
	fail_unless(smlTransportInitialize(client1, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(client2, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(client3, &clientConfig, &error), NULL);
	fail_unless(smlTransportInitialize(server, &serverConfig, &error), NULL);

	smlTransportSetEventCallback(client1, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(client2, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(client3, _recv_client_event, GINT_TO_POINTER(1));
	smlTransportSetEventCallback(server, _recv_server_event, GINT_TO_POINTER(1));
	
	SmlTransportData *data = NULL;
	
	int i;
	for (i = 0; i < num; i++) {
		data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
		fail_unless(data != NULL, NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlTransportSend(client1, NULL, data, &error), NULL);
		fail_unless(error == NULL, NULL);
		smlTransportDataDeref(data);
		
		data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
		fail_unless(data != NULL, NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlTransportSend(client2, NULL, data, &error), NULL);
		fail_unless(error == NULL, NULL);
		smlTransportDataDeref(data);
		
		data = smlTransportDataNew("test", 5, SML_MIMETYPE_XML, FALSE, &error);
		fail_unless(data != NULL, NULL);
		fail_unless(error == NULL, NULL);
		fail_unless(smlTransportSend(client3, NULL, data, &error), NULL);
		fail_unless(error == NULL, NULL);
		smlTransportDataDeref(data);
		
		while (1) {
			usleep(1);
			if (client_receives == (i+1)*3)
				break;
		}
	}
	
	fail_unless(client_receives == 3 * num, NULL);
	fail_unless(server_receives == 3 * num, NULL);
	
	smlTransportStop(client1);
	smlTransportStop(client2);
	smlTransportStop(client3);
	smlTransportStop(server);
	
	fail_unless(smlTransportFinalize(server, &error), NULL);
	fail_unless(smlTransportFinalize(client1, &error), NULL);
	fail_unless(smlTransportFinalize(client2, &error), NULL);
	fail_unless(smlTransportFinalize(client3, &error), NULL);
	
	smlTransportFree(server);
	smlTransportFree(client1);
	smlTransportFree(client2);
	smlTransportFree(client3);
}
END_TEST

Suite *http_suite(void)
{
	Suite *s = suite_create("Http Transports");
	//Suite *s2 = suite_create("Http Transports");
	
	create_case(s, "http_client_new", http_client_new);
	create_case(s, "http_client_init", http_client_init);
	create_case(s, "http_client_wrong_port", http_client_wrong_port);
	create_case(s, "http_client_init_no_url", http_client_init_no_url);
	
	create_case(s, "http_server_new", http_server_new);
	create_case(s, "http_server_init", http_server_init);
	create_case(s, "http_server_wrong_port", http_server_wrong_port);
	
	create_case(s, "http_connect", http_connect);
	create_case(s, "http_send", http_send);
	create_case(s, "http_receive", http_receive);
	create_case(s, "http_reply", http_reply);
	create_case(s, "http_talk", http_talk);
	create_case(s, "http_connect_error", http_connect_error);
	create_case(s, "http_reject", http_reject);
	create_case(s, "http_multi_connect", http_multi_connect);
	create_case(s, "http_multi_partial_error", http_multi_partial_error);
	create_case(s, "http_multi_stress", http_multi_stress);
	
	return s;
}

int main(void)
{
	int nf;

	Suite *s = http_suite();
	
	SRunner *sr;
	sr = srunner_create(s);
	srunner_run_all(sr, CK_NORMAL);
	nf = srunner_ntests_failed(sr);
	srunner_free(sr);
	return (nf == 0) ? 0 : 1;
}
