/* This file is part of the KDE project

   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/


#ifndef IMAPMANAGER_H
#define IMAPMANAGER_H

#include "imaplib.h"
#include <qwidget.h>
class QSocket;
class DB;


namespace Mailody {

/**
 * @class ImapManager
 * Responsible for communicating with the imaplib and storing stuff in the
 * database.
 * @author Tom Albers <tomalbers@kde.nl>
 */
class ImapManager : public QWidget
{
  Q_OBJECT

  public:
        /**
         * Contructor
         */
        explicit ImapManager( QWidget* parent,  const char* name );

        /**
         * Destructor
         */
        ~ImapManager();

        /**
         * Returns an instance of this.
         */
        static ImapManager* instance();

        /**
         * Reconnect to the Imap server. You can call this safely with
         * changing from socket to secure socket and vice versa
         * @returns true when there is a new connection, false when there
         * is no need for a new connection.
         */
        void startConnection();

        /**
         * Will return the mailbox list. It will return the signal
         * mailBoxList(const QStringList&) with the result.
         * @param sync if true, do not use the cache, but refetch.
         */
        void getMailBoxList( bool sync=false );

        /**
         * This will select the mailbox and tries to fetch only whats needed.
         * @param box the mailbox
         * @param sync if true, do not use the cache, but refetch.
         */
        void getMailBox(const QString& box, bool sync=false);

        /**
         * Get the header of @p uid from the @p mb mailbox. Assumes this data is
         * available in the database.
         */
        QString getHeader( const QString& mb, int uid);

        /**
         * Get the body of @p uid from the @p mb mailbox, goes to the server if
         * it is not available in the database.
         */
        void getMessage(const QString& mb, int uid);

        /**
         * This will not select the mailbox, but will update message count
         * and fetch new mail
         * @param box the mailbox
         */
        void checkMail(const QString& box);
            
        /**
         * This will refetch the tags.
         * @param box the mailbox
         *
         */
        void updateStatus(const QString& box);

        /**
         * returns if a message has that flag set, knowledge obtained from the
         * database
         * @param box the mailbox
         * @param uid uid of the message
         * @param flag flag to add
         */
        bool hasFlag(const QString& box, int uid, const QString& flag);

        /**
         * add a flag, goes to the server and to the database for a uid in a box
         * @param box the mailbox
         * @param uid uid of the message
         * @param flag flag to add
         */
        void addFlag(const QString& box, int uid, const QString& flag);

        /**
         * add a flag, goes to the server and to the database. for a whole box
         * @param box the mailbox
         * @param flag flag to add
         */
        void addFlag(const QString& box, const QString& flag);

        /**
         * remove a flag, goes to the server and to the database.
         * @param box the mailbox
         * @param uid uid of the message
         * @param flag flag to add
         */
        void removeFlag(const QString& box, int uid, const QString& flag);

        /**
         * removes deleted messages
         * @param box the mailbox
         */
        void expungeMailBox(const QString& box);

        /**
         * create a folder
         */
        void createMailBox(const QString& box);

        /**
         * delete a folder
         */
        void deleteMailBox(const QString& box);

        /**
         * rename folder @p oldfolder to @p newfolder
         */
        void renameMailBox(const QString& oldfolder, const QString& newfolder);

        /**
         * This will move the message with @p uid from @p origbox to the
         * @p destbox mailbox.
         */
        void moveMessage(const QString& origbox, int uid,
                         const QString& destbox);

        /**
         * This will copy the message with @p uid from @p origbox to the
         * @p destbox mailbox.
         */
        void copyMessage(const QString& origbox, int uid,
                         const QString& destbox);

        /**
         * Store message with the content @p message in the @p mb mailbox,
         * you can use @p flags to set extra flags like \\Draft, seperate
         * multiple tags by a space and use double slashes.
         */
        void saveMessage(const QString& mb, const QString& message,
                         const QString& flags=QString::null);

        /**
         * Start monitoring the @p mb mailbox
         */
        void idleStart(const QString& mb);

        /**
         * Stop monitoring
         */
        void idleStop();

 private:

        DB*                 m_db;
        Imaplib*            m_imap;
        static ImapManager*        m_instance;
        QTimer*             m_preventLogOutTimer;
        QTimer*             m_checkMailTimer;
        QString             m_received;
        QStringList         m_capabilities;

        void manualAuth(Imaplib*, const QString& );

  signals:
        /**
         * Emitted when the user wants to see the settings, to alter the
         * credentials.
         */
        void showSettings();

        /**
         * Emitted when the user is authenticated.
         */
        void loginOk();

        /**
         * Reports about the network status
         */
        void status(const QString&);

        /**
         * Reports about the network status done
         */
        void statusReady();

        /**
         * Reports about the network status, there has been an error, but
         * this class is done with it, so restore the cursor and do something
         */
        void statusError(const QString&);

        /**
         * Reports the mailboxes available, mostly as a result to a call to
         * getMailBoxList()
         */
        void mailBoxList(const QStringList&);

        /**
         * a new mailbox is found on the server
         */
        void mailBoxAdded(const QString& mb);

        /**
         * a mailbox is removed from the server
         */
        void mailBoxDeleted(const QString& mb);

        /**
         * Reports the headers for a mailbox, mostly as a result to a call to
         * getMailBox()
         * @param box the mailbox for reference.
         * @param values the headers.
         */
        void mailBox(const QString& box, const QStringList& values);

        /**
         * Reports additional headers for a mailbox, mostly as a result of a
         * mailcheck...
         * @param box the mailbox for reference.
         * @param values the headers.
         */
        void mailBoxAddition(const QString& box, const QStringList& values);

        /**
         * The body is of a message, mostly as a result to a call to
         * getMessage()
         * @param box the mailbox for reference.
         * @param uid the uid for reference.
         * @param text the body
         */
        void message(const QString& mb, int uid, const QString& text);

        /**
         * We know the amount of messages(@p amount) in the @p mb
         */
        void messageCount(const QString& mb, int amount);

        /**
         * We know the amount of unseen messages(@p amount) in the @p mb
         */
        void unseenCount(const QString& mb, int amount);

        /**
         * This signal is emitted when a message is saved
         */
        void saveDone();

        /**
         * This signal is emitted when all uids are known...
         */
        void allUidsKnown(const QString&);

    private slots:
        void slotLogin( Imaplib* );
        void slotLoginFailed( Imaplib* );
        void slotAlert(Imaplib*, const QString&);
        void slotGetMailBoxList(const QStringList&);
        void slotGetMailBox(Imaplib*, const QString&, const QStringList&);
        void slotGetMessage(Imaplib*, const QString&, int, const QString&);
        void slotMessagesInMailbox(Imaplib*, const QString&, int);
        void slotUnseenMessagesInMailbox(Imaplib*, const QString&, int);
        void slotMailBoxAdded(const QString&);
        void slotMailBoxRemoved(const QString&);
        void slotMailBoxRenamed(const QString&, const QString&);
        void slotMailBoxExpunged(Imaplib*, const QString& box);
        void slotMailBoxItems(Imaplib*, const QString&, const QStringList&);
        void slotIntegrity(const QString& mb, int totalShouldBe,
                           const QString& uidvalidity, const QString& uidnext);
        void slotError(const QString&);
        void slotDisconnected();
};

}

#endif

