/***************************************************************************
 *   Copyright (C) 2004-2005 by Andreas Ramm                                           *
 *   psychobrain@gmx.net                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#ifndef MESSAGE_H
#define MESSAGE_H
#include <qobject.h>


#ifndef QT_H
#include "qstring.h"
#endif // QT_H

class QSocket;
#ifndef QValueVector
#include <qvaluevector.h>
#endif

#ifndef QImage
#include <qimage.h>
#endif
#include <ksocketbase.h>
#include <kclientsocketbase.h>
#include <kstreamsocket.h>

#include <qsyntaxhighlighter.h>
#include <qptrvector.h>

namespace MateEdit
{

    typedef QValueVector<int> lineVector;
    typedef lineVector TextOwnerLine;
    class TextOwner;
    class SyntaxHighlighter;
    class Client;


    #define MATEEDIT_MAX_USERS 6

    /**
    * @brief Coordinate class for messages
    * @long Coordinate definitions for messages. Includes comparison operators and routines for adjusting coordinates efficiently
    * @author Andreas Ramm <psychobrain@gmx.net>
    * @version 0.1
    */
    class MessageCoords
    {
        public:
            MessageCoords();
            MessageCoords( unsigned int, unsigned int );
            MessageCoords(const MessageCoords &);
            void setCoords( unsigned int, unsigned int );
            void setCoords(const MessageCoords &);
            bool operator== ( const MessageCoords & ) const;
            bool operator!= ( const MessageCoords & ) const;
            bool operator<= ( const MessageCoords & ) const;
            bool operator< ( const MessageCoords & ) const;
            bool operator>= ( const MessageCoords & ) const;
            bool operator> ( const MessageCoords & ) const;
            int shiftUp( const MessageCoords &b, const MessageCoords &c );
            int shiftDown( const MessageCoords &b, const MessageCoords &c );
            unsigned int para() const;
            unsigned int idx() const;
        private:
            unsigned int m_para;
            unsigned int m_idx;
    };


    class Shift
    {
        public:
            enum Type
            {
                Q_LEFT_OF_S,
                S_LEFT_OF_Q
            };

            Shift( int type, unsigned int queue_sender, unsigned int queue_id, unsigned int server_sender, unsigned int server_id, MessageCoords & server_from, MessageCoords & index, int n_chars, int offset = 0 );
            int type();
            unsigned int queueSender();
            unsigned int queueId();
            unsigned int serverSender();
            unsigned int serverId();
            MessageCoords serverFrom();
            MessageCoords index();
            int nChars();
            int offset();
        private:
            int m_type;
            unsigned int m_queue_sender;
            unsigned int m_queue_id;
            unsigned int m_server_sender;
            unsigned int m_server_id;
            MessageCoords m_server_from;
            MessageCoords m_index;
            int m_n_chars;
            int m_offset;
    };

    /**
    * @brief Individual Message for network transfer and order control
    * @author Andreas Ramm <psychobrain@gmx.net>
    * @version 0.1
    */
    class Message
    {
        public:
            enum Type
            {
                Undefined,
                NoOp,
                Init,
                InitUserData,
                RemoveUser,
                Insert,
                Delete,
                InfoMessage,
                ChatMessage,
                Undo,
                Redo
            };
            Message( Type type, unsigned int sender = 0, unsigned int serverState = 0, unsigned int clientState = 0, const QString & buffer = QString::null );
            Message();
            ~Message();

            unsigned int type();
            unsigned int originalType();
            unsigned int sender();
            unsigned int serverState();
            unsigned int clientState();
            unsigned int id();
            unsigned int parentId();
            QString & text();
            QImage & image ();
            MessageCoords & from( unsigned int sender );
            MessageCoords & originalFrom();
            MessageCoords & to( unsigned int sender );
            void setType( Type );
            void setSender( unsigned int new_sender );
            void setServerState( unsigned int new_server_state );
            void setClientState( unsigned int new_client_state );
            void setText( QString & newMessage );
            void setId( unsigned int id );
            void setImage ( const QImage & );
            void setFrom( unsigned int fromPara, unsigned int unsignedfromIdx );
            void setFrom( const MessageCoords & );
            void setTo( unsigned int toPara, int unsigned toIdx );
            void setTo( const MessageCoords & );
            void shiftUpFrom ( const MessageCoords &b, const MessageCoords &c );
            void shiftUpFrom ( unsigned int, const MessageCoords &b, const MessageCoords &c );
            void shiftUpTo ( const MessageCoords &b, const MessageCoords &c );
            void shiftUpTo ( unsigned int, const MessageCoords &b, const MessageCoords &c );
            void shiftDownFrom ( const MessageCoords &b, const MessageCoords &c );
            void shiftDownFrom ( unsigned int, const MessageCoords &b, const MessageCoords &c );
            void shiftDownTo ( const MessageCoords &b, const MessageCoords &c );
            void shiftDownTo ( unsigned int, const MessageCoords &b, const MessageCoords &c );
            void addAffected( Message * message );
            QPtrList<Message> * affectedList();
            void createUndo( Message * message, int local_pc );
            void restoreType();
            bool undone();


            void send ( KNetwork::KActiveSocketBase * socket );
            void read( QBuffer * buffer  );

            void debug( const char *file, int line, const char * comment );
        private:
            unsigned int m_type;
            unsigned int m_original_type;
            MessageCoords m_fromCoords[6];
            MessageCoords m_originalFromCoords;
            MessageCoords m_toCoords[6];
            bool m_originalFromCoordsSet;
            bool m_undone;
            unsigned int m_sender;
            struct __state__
            {
                unsigned int server;
                unsigned int client;
            }m_state;
            QImage * m_image;
            QString m_message;
            unsigned int m_id;
            unsigned int m_parent_id;
            QPtrList<Message> * m_affected_list;
    };

    #ifndef QT_NO_DATASTREAM
    Q_EXPORT QDataStream &operator<<( QDataStream &, const MessageCoords & );
    Q_EXPORT QDataStream &operator>>( QDataStream &, MessageCoords & );
    #endif
    Q_EXPORT QTextStream &operator<<( QTextStream &, const MessageCoords & );

    /**
    * @brief Message Handler
    * @author Andreas Ramm <psychobrain@gmx.net>
    * @version 0.1
    */
    class MessageQueue
    {
        public:
            /**
            * Default Constructor
            */
            MessageQueue();
            /**
            * Default Destructor
            */
            ~MessageQueue();

            void clearProcessed();
            void debug(QString text);

            void setLocalPc(Q_UINT32 pc);
            Q_UINT32 localPc();
            int processMessage( Message * message );
            Message * findMessage( unsigned int id, unsigned int sender );
            Message * undoMessage( unsigned int id, unsigned int sender, QPtrList<Message> & pending_list, int server_state, int client_state = 0);
        private:
            Q_UINT32 m_local_pc;
            QPtrList<Message> * m_message_list;
            QPtrList<Message> * m_pending_message_list;
            typedef QPtrListIterator<Message> MessageIterator;
            QPtrVector<MessageIterator> m_client_it;
            unsigned int * m_current_message_ids;
            void appendProcessed( const Message * message );

            int transformMessage ( Message & queueMessage, Message & serverMessage, bool undo = false, bool ignore_ps = false );
            QPtrList<Shift> * m_shifted_ops;
    };

    /**
     * @brief Maintains a copy of the document
               * This class stores a copy of the shared document
               * @author Andreas Ramm <psychobrain@gmx.net>
               * @version 0.1
     */
    class Document
    {
        public:
            Document( int );
            ~Document();
            int size();
            QString operator[] (const unsigned int );
            QString at(const unsigned int );
            QChar at (const unsigned int, const unsigned int );

            void insert( const MessageCoords &, const MessageCoords &, int, const QString & );
            QString remove( const MessageCoords &, const MessageCoords & );

            void debug( bool, const MessageCoords &, const MessageCoords & );
        private:
            QValueVector<QString> * m_textVector;
    };


    /**
    * @brief
    * @author Andreas Ramm <psychobrain@gmx.net>
    * @version 0.2
    */
    class MateEdit : public QObject
    {
        Q_OBJECT
        public:
            /**
            * Default Constructor.
            */
            MateEdit( Client * client, QTextEdit * textedit, bool init_app = false, bool init_user = false, bool remove_user = false, bool insert_text = false, bool delete_text = false, bool chat_message = false, bool message = false );
            /**
            * Default Destructor
            */
            virtual ~MateEdit();


            int connectToServer( const QString & server, int port );

            int insertText( const QString & buffer, const int paraFrom, const int indexFrom, const int paraTo, const int indexTo );
            int insertText( const QString & buffer, const MessageCoords & from, const MessageCoords & to );
            int deleteText( const QString & buffer, const int paraFrom, const int indexFrom, const int paraTo, const int indexTo );
            int deleteText( const QString & buffer, const MessageCoords & from, const MessageCoords & to );

            int chat( const QString & buffer );

            int undoMessage( int id );
            int redoMessage( int id );

            void remoteInsertText( const QString & buffer, const MessageCoords & from, const MessageCoords & to );
            void remoteDeleteText( const QString & buffer, const MessageCoords & from, const MessageCoords & to );
            void remoteChat( const QString & text );
            void remoteMessage( const QString & text );
            void remoteInitSession( int user_number, QString & username, QImage & image );
            void remoteSessionClosed( const QString & text );
            void remoteAddUser( int user_number, const QString & name, const QImage & image );
            void remoteRemoveUser( int user_number, const QString & name );

            SyntaxHighlighter * syntaxHighlighter();
        private:
            int undoMessage( int id, int sender, int server_state = 0, int client_state = 0);
            int redoMessage( int id, int sender, int server_state = 0, int client_state = 0);

            unsigned int m_timestamp;
            MessageQueue m_messageQueue;
            KNetwork::KStreamSocket * m_client_socket;
            Q_UINT32 m_local_pc;
            TextOwner * m_owners;
            Document * m_doc;
            QTextEdit * m_textedit;
            SyntaxHighlighter * m_syntaxHighlighter;
            QString * users;
            Client * m_client;
        //    MateEditServerThread * m_serverThread;
        private slots:
            void readServer();
//            void socketStateChanged(KNetwork::KClientSocketBase::SocketState);
            void connectionClosed();
            void socketStateChanged(int state);
    };

    /**
    * @brief Maintains ownership for each individual character
    * This class stores ownership of each indiviual character entered by any user.
    * @author Andreas Ramm <psychobrain@gmx.net>
    * @version 0.1
    */
    class TextOwner
    {
        public:
            TextOwner( int );
            ~TextOwner();
            int size();
            lineVector operator[] (const unsigned int );
            lineVector at(const unsigned int );
            int at (const unsigned int, const unsigned int );

            bool insert( const MessageCoords &, const MessageCoords &, int, const QString & );
            void remove( const MessageCoords &, const MessageCoords & );

            void debug( bool, const MessageCoords &, const MessageCoords & );
        private:
            QValueVector<lineVector> * m_textVector;
    };
    /**
    * @brief Highlights ownership of text
    * @author Andreas Ramm <psychobrain@gmx.net>
    * @version 0.1
    */
    class SyntaxHighlighter : public QSyntaxHighlighter
    {
        public:
            SyntaxHighlighter( QTextEdit *, TextOwner * );
            int highlightParagraph ( const QString & text, int endStateOfLastPara );
            void setLocalPc( int );
        private:
            QColor * colors;
            TextOwner * m_owners;
            int m_local_pc;
    };

    class Client
    {
        public:
            virtual bool mateEditInit( int user_number, QString & username, QImage & image );
            virtual bool mateEditInitUser( int user_number, const QString & name, const QImage & image );
            virtual bool mateEditRemoveUser( int user_number, const QString & name );
            virtual bool mateEditInsertText( const QString & text, const MessageCoords & from, const MessageCoords & to );
            virtual bool mateEditDeleteText( const QString & text, const MessageCoords & from, const MessageCoords & to );
            virtual bool mateEditMessage( const QString & text );
            virtual bool mateEditChatMessage( const QString & text );
            virtual bool mateEditSessionClosed( const QString & reason );

    };
}
#endif
