""" A wizard to save an editor's resource in the resource tree. """


# Enthought library imports.
from enthought.envisage.ui import Editor
from enthought.naming.api import Binding
from enthought.traits.api import Any, Instance, Property, Str

# Local imports.
from new_named_resource_page import NewNamedResourcePage
from new_resource_wizard import NewResourceWizard
from resource_factory import ResourceFactory


class EditorResourceFactory(ResourceFactory):

    # The resource editor.
    editor = Instance(Editor)
    
    # The resource.
    resource = Any
    
    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, editor, resource, **traits):

        # Base class constructor.
        super(EditorResourceFactory, self).__init__(**traits)

        # Set the mandatory traits.
        self.editor = editor
        self.resource = resource
        
        return
    ###########################################################################
    # 'ResourceFactory' interface.
    ###########################################################################
    
    def create_resource(self):
        """ Creates the resource. """
        
        # Bind at the selected location in the naming system.
        self.parent_folder.obj.bind(self.name, self.resource)
        
        # Update the editor so it refers to the correct resource name
        name = self._get_resource_name(self.parent_folder, self.name)
        self.editor.resource_name = name
            
        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _get_resource_name(self, folder, name):
        """ Get the resource name for the resource. """
        
        return '/'.join((folder.namespace_name, name))

class SaveAsWizard(NewResourceWizard):
    """ A wizard to save an editor's resource in the resource tree. """
    
    #### 'Window' interface ###################################################
    
    # The window title.
    title = 'Save As'
    
    # The parent folder to save the resource in.
    parent_folder = Instance(Binding)
    
    # The name to save the resource as.
    name = Str
    
    # The name to suggest for the resource.
    suggested_prefix = Str
    
    # The full namespace name to bind the resource as.
    resource_name = Property(Str)
    
    ###########################################################################
    # 'object' interface.
    ###########################################################################

    def __init__(self, editor, resource, **traits):
        """ Creates a new wizard. """
        
        self.factory = EditorResourceFactory(editor, resource)

        self.parent = editor.control
        
        # Base class constructor.
        super(SaveAsWizard, self).__init__(**traits)

    def reset(self):
        """ Reset the wizard to the original state. """
    
        # The wizard contains a single page that allows the user to select the
        # folder in which to save the resource and to give the new resource a 
        # name.
        self.pages = [
            NewNamedResourcePage(
                text   = 'Select the parent folder',
                prefix = self.suggested_prefix,
                obj    = self.factory
            )
        ]
        
        return
        
#### EOF ######################################################################
