""" The base class for all resource types. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.naming.api import ContextAdapterFactory, ObjectSerializer
from enthought.pyface.tree.api import NodeType
from enthought.traits.api import Any, Dict, HasTraits, Instance, Str, TraitError

# Local imports.
from resource_manager import ResourceManager
from resource_node_type import ResourceNodeType
from resource_reference import ResourceReference


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class ResourceType(HasTraits):
    """ The base class for all resource types.

    A resource type represents a *kind* of resource (i.e., domain object) such
    as a log, log suite, well, mixing tank etc.  It could also represent a
    'primitive' type such as an int, float, or more probably in our case, a
    numeric array.

    A resource type describes how to manage the following aspects of its
    resources:

    1) Visualization

    Resources can be visualized in a number of ways, in trees, tables, lists
    etc.  To do this we need information such as icons, label text etc. and
    this is provided by the resource type's node type.  Although the node type
    has its roots in the tree world, it can obviously be reused across widgets
    such as tables and lists etc by simply not displaying the resource's
    children.

    2) Serialization

    A resource type can provide a serializer to control how its resources are
    persisted (e.g., to the project naming system).  By default, all objects
    are pickled using cPickle, but for example, a log suite resource type could
    provide a serializer that would read and write log suites as LAS (log ASCII
    standard) files.

    3) Naming

    Envisage uses naming systems heavily, and importantly, the project
    workspace is simply a persistent naming system that uses the file system
    (see 'enthought.naming.PyFSContext').  If required, a resource type can
    describe how naming resolution can proceed 'into' the namespace of an
    individual resource e.g., a log suite resource type could allow name
    resolution into its logs.  This comes in handy in the project workspace
    view so that the user can navigate to any item displayed in the project
    tree using the same naming mechanism.

    4) Editing

    A resource type can specify a class that will be used to create editors its
    resource.

    fixme: This should probably be the common pattern of:- a factory or a
    class where the default factory just creates an instance of the class.

    5) Traits UI handlers.

    Resource types contain a dictionary of traits UI handlers that can be used
    to create user interfaces for resources.

    """

    #### 'ResourceType' interface #############################################

    # The type's unique Id (where 'unique' means unique within the resource
    # manager that the type is in.  In Envisage we have exactly one resource
    # manager so this effectively needs to be globally unique).
    id = Str

    # The resource manager that the resource type is part of.
    resource_manager = Instance(ResourceManager)

    # A trait that describes the kind of domain object that the resource type
    # represents.
    type = Any

    # Visualization (#1)
    #
    # The node type describes how to visualize resources of this type.
    node_type = Instance(NodeType)

    # Serialization (#2)
    #
    # The (optional) serializer describes how resources of this type are
    # persisted.  If no serializer is specified then the resources will be
    # pickled using 'cPickle'.
    serializer = Instance(ObjectSerializer)

    # Naming (#3)
    #
    # The (optional) context adapter factory creates (not surprisingly)
    # adapters that describe how resources of this type participate in name
    # resolution.  If no context adapter is specified name resolution is not
    # allowed to proceed 'into' the namespace of a resource.
    context_adapter_factory = Instance(ContextAdapterFactory)

    # Editing (#4)
    #
    # The class used to create editors for resources of this type.
    editor = Any # fixme: This should be Class but it doesn't seem to work!

    # Views (#5).
    #
    # fixme: These are actually traits UI handlers at the moment. I think this
    # needs some more thought.
    views = Dict # (Str, Class)

    ###########################################################################
    # 'ResourceType' interface.
    ###########################################################################

    #### Initializers #########################################################

    def _node_type_default(self):
        """ Initializes the node type trait. """

        return ResourceNodeType(resource_type=self)

    def _id_default(self):
        """ Initializes the id trait. """

        return "%s.%s" % (self.__class__.__module__, self.__class__.__name__)

    #### Methods ##############################################################

    def is_type_for(self, obj):
        """ Returns True if the resource type 'recognizes' an object. """

        # If setting the trait succeeds then object is deemed to be of this
        # type.
        #
        # fixme: Beware coercion?
        try:
            self.type = obj
            is_type_for = True

        except TraitError:
            is_type_for = False

        return is_type_for

    def clone(self, obj):
        """ Returns a clone of an object. """

        raise NotImplementedError

    def get_name(self, obj):
        """ Returns the name of an object. """

        if hasattr(obj, 'name'):
            name = obj.name

        else:
            name = 'Anonymous'

        return name

    def set_name(self, obj, name):
        """ Sets the name of an object. """

        if hasattr(obj, 'name'):
            obj.name = name

        return

    # fixme: This was a quick hack for pasting folders....
    def copy(self, context, name, obj):
        """ Copies a resource into the specified context. """

        raise NotImplementedError

    # The following methods are used by the new workbench API.
    def get_reference(self, obj):
        """ Returns a pickleable reference to a resource.

        By default this returns the object itself. This is only acceptable
        for SMALL resources!

        """

        return ResourceReference(data=obj)

    def create_editor(self, obj, *args, **kw):
        """ Creates an editor for a resource.

        By default this just calls the class referred to by the 'editor'
        trait.

        """

        if self.editor is not None:
            editor = self.editor(resource=obj, *args, **kw)

        # If no editor is specified then we create an editor that includes
        # the resource's default traits view.
        else:
            logger.debug('no editor for resource [%s]', obj)

            from enthought.envisage.workbench.traits_ui_editor import \
                 TraitsUIEditor

            # Try to guess a sensible name for the editor!
            if hasattr(obj, 'name'):
                name = obj.name

            else:
                name = str(obj)

            editor = TraitsUIEditor(resource=obj, name=name)

        return editor

    def get_id(self, obj):
        """ Returns a unique identifier for a resource.

        Unique here means, 'unique within the current process'.

        """

        return id(obj)

#### EOF ######################################################################
