""" The resource ui plugin. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.envisage import Plugin
from enthought.envisage.core.core_plugin import CorePlugin
from enthought.naming.api import Context

from enthought.envisage.project.cookie_manager import CookieManager

# Local imports.
from resource_manager import ResourceManager
from resource_plugin import ResourcePlugin
from resource_ui_plugin_definition import CookieImplementations, \
                                          ResourceActions,ResourceWizards


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class ResourceUIPlugin(Plugin):
    """ The resource plugin. """

    # Services that we create.
    ICOOKIE_MANAGER   = 'enthought.envisage.resource.ICookieManager'
    IWIZARD_MANAGER   = 'enthought.envisage.resource.IWizardManager'

    #### 'ResourcePlugin' interface ###########################################

    ###########################################################################
    # 'Plugin' interface.
    ###########################################################################

    def start(self, application):
        """ Starts the plugin. """

        # Get a reference to the resource manager service.
        resource_manager = self.get_service(ResourcePlugin.IRESOURCE_MANAGER)

        # build resource menus
        self._build_resource_menus(resource_manager)

        # Load the cookie implemenations.
        cookie_manager = self._start_cookie_manager(application)

        # Register the cookie manager as a service.
        self.register_service(self.ICOOKIE_MANAGER, cookie_manager)

        wizards = self._start_wizards(application)
        # Register the wizards as a service.
        self.register_service(self.IWIZARD_MANAGER, wizards)

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _build_resource_menus(self, resource_manager):

        # fixme: this should be separated from menu building! actions, groups,
        #        etc. can be defined here, but the menus should be handled
        #        by the ui plugin or a similar place.

        # Action, group, menu extensions.
        extensions = self.get_extensions(ResourceActions)

        keys = {}
        menus   = {}
        groups  = {}
        actions = {}
        for extension in extensions:
            for menu in extension.menus:
                keys[menu.resource_type] = None
                type_menus = menus.setdefault(menu.resource_type, [])
                type_menus.append((menu, extension._definition_.id))

            for group in extension.groups:
                keys[group.resource_type] = None
                type_groups = groups.setdefault(group.resource_type, [])
                type_groups.append((group, extension._definition_.id))
            for action in extension.actions:
                keys[action.resource_type] = None
                type_actions = actions.setdefault(action.resource_type, [])
                type_actions.append((action, extension._definition_.id))

        from enthought.envisage.project.menu_builder import MenuBuilder
        menu_builder = MenuBuilder()

        from enthought.pyface.action.api import MenuManager
        from enthought.envisage import get_application
        application = get_application()
        for key in keys:
            # Build the resource manager context menu.
            if len(key) == 0:
                menu_manager = resource_manager.context_menu

                menu_builder.build_menu(
                    application,
                    menu_manager,
                    menus.get(key, []),
                    groups.get(key, []),
                    actions.get(key, [])
                )

            # Build the context menu for a specififc resource type.
            else:
                menu_manager = MenuManager()

                menu_builder.build_menu(
                    application,
                    menu_manager,
                    menus.get(key, []),
                    groups.get(key, []),
                    actions.get(key, [])
                )

                resource_type = resource_manager.lookup(key)
                if resource_type is not None:
                    resource_type.node_type.context_menu = menu_manager
                else:
                    logger.error('Unable to find resource type for key ' + key)

        return

    def _start_cookie_manager(self, application):
        """ Initialize the cookie manager. """

        import_manager = application.import_manager
        resource_manager = \
                   application.get_service(ResourcePlugin.IRESOURCE_MANAGER)

        cookie_manager = CookieManager()

        # Action, group, menu extensions.
        extensions = self.get_extensions(CookieImplementations)

        for extension in extensions:
            # Register each of the cookie implementations with the manager.
            for implementation in extension.implementations:
                # To register a cookie implementation with a resource type,
                # we require the resource type from the resource manager,
                # the class of the cookie interface, and an instance of the
                # implementation.  So, let's get those.
                ##resource_type_class = import_manager.import_symbol(
                ##    implementation.resource_type
                ##)
                resource_type = resource_manager.lookup(
                    ##resource_type_class
                    implementation.resource_type
                )

                cookie_class = import_manager.import_symbol(
                    implementation.cookie_interface
                )

                implementation_class = import_manager.import_symbol(
                    implementation.cookie_implementation
                )
                implementation = implementation_class()

                # Finally, we can register the type cookie!
                cookie_manager.add_type_cookie(
                    cookie_class, implementation, resource_type
                )

        return cookie_manager

    def _start_wizards(self, application):
        """ Initialize the cookie manager. """

        # Get all the resource types that we know about.
        root = Context()
        extensions = self.get_extensions(ResourceWizards)
        for extension in extensions:
            for resource_wizard in extension.resource_wizards:
                # Does the category exist?
                try:
                    category = root.lookup(resource_wizard.category)

                except: # NameNotFoundError
                    category = root.create_subcontext(resource_wizard.category)

                class_name = resource_wizard.class_name
                klass = application.import_manager.import_symbol(class_name)

                category.bind(resource_wizard.name, klass())

        return root

#### EOF ######################################################################
