#-----------------------------------------------------------------------------
#
# Copyright (c) 2007 by Enthought, Inc.
# All rights reserved.
#
#-----------------------------------------------------------------------------

"""
A base class for context adapters that want to implement the minimal number
of methods yet still operate as a context.

"""

# Standard library imports.
import logging

# Enthought library imports.
from enthought.envisage.resource.resource_context_adapter import \
    ResourceContextAdapter
from enthought.naming.api import Binding
from enthought.naming.unique_name import make_unique_name


# Setup a logger for this module
logger = logging.getLogger(__name__)


class SimpleResourceContextAdapter(ResourceContextAdapter):
    """
    A base class for context adapters that want to implement the minimal number
    of methods yet still operate as a context.

    Contexts generated by this, or derived classes, will generally not perform
    as fast as implementing all context methods directly on the adapted traits.
    This is because this implementation makes more method calls and looks up
    more information then may be strictly necessary in order to minimize the
    number of methods that need to be implemented.

    The methods that MUST be implemented by sub-classes all raise a
    NotImplementedError in this implementation.  These methods determine how
    your context behaves.

    """

    ##########################################################################
    # 'Context' interface.
    ##########################################################################

    #### protected interface #################################################

    def _bind(self, name, obj):
        """
        Bind a name to an object in this context.

        """

        raise NotImplementedError

        return


    def _get_namespace_name(self):
        """
        Return the name of the context within its own namespace.

        That is the full-path, through the namespace this context participates
        in, to get to this context.  For example, if the root context of the
        namespace was called 'Foo', and there was a subcontext of that called
        'Bar', and we were within that and called 'Baz', then this should
        return 'Foo/Bar/Baz'.

        """

        raise NotImplementedError


    def _is_bound(self, name):
        """
        Is a name bound in this context?

        """

        return name in self._list_names()


    def _lookup(self, name):
        """
        Looks up a name in this context.

        """

        raise NotImplementedError

        return None


    def _list_bindings(self):
        """
        Lists the bindings in this context.

        """

        bindings = [
            Binding(name=name, obj=self._lookup(name), context=self)
            for name in self._list_names()
            ]

        return bindings


    def _list_names(self):
        """
        Lists the names bound in this context.

        If the ordering here changes, then please also update the ordering
        within the *get_node_index()* method.

        """

        raise NotImplementedError

        return []


    def _is_context(self, name):
        """
        Returns True if a name is bound to a context.

        This implmentation assumes that all names in this context *ARE* also
        contexts.

        """

        result = False
        if name in self._list_names():
            result = True

        return result


    def _unbind(self, name):
        """
        Unbind the object with the specified name.

        FIXME: Relying on the name means that either names must be unique
        or else we may delete the wrong item.  We should either fix the
        context concept to allow deletion of a specific object within the
        context or rework all of this context and the containable objects
        to force name uniqueness.

        """

        raise NotImplementedError

        return


    def _rename(self, old_name, new_name):
        """
        Renames an object in this context.

        """

        obj = self._lookup(old_name)
        self._unbind(old_name)
        new_name = make_unique_name(new_name, self._list_names())
        self._bind(new_name, obj)

        return


#### EOF #####################################################################

