#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005-2006 by Enthought, Inc.
#  All rights reserved.
#
#-----------------------------------------------------------------------------

"""
A base class for actions that switch the current perspective.

"""


# Standard library imports.
import logging

# Enthought library imports
from enthought.traits.api import Bool, List, Str

# Local imports.
from workbench_action import WorkbenchAction


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class PerspectiveAction(WorkbenchAction):
    """
    A base class for actions that switch the current perspective.

    If the user of this action calls the refresh method, this action is
    only enabled when associated with a workbench window in which our
    configured perspective exists.

    This class is architected such that implementors can easily override
    the 'perform' and 'refresh' methods without losing the ability to
    easily reuse our implementation logic.

    """

    ##########################################################################
    # Attributes
    ##########################################################################

    #### public 'PerspectiveAction' interface ################################

    # The ID of the perspective we switch to.
    perspective_id = Str("")


    ##########################################################################
    # 'Action' interface.
    ##########################################################################

    #### public interface ####################################################

    def perform(self, event):
        """ Performs the action.

            This implementation simply switches perspectives.

            Implementors should override this to add additional work to the
            performance of this action.
        """
        self._switch_perspective(event)


    ##########################################################################
    # 'PerspectiveAction' interface
    ##########################################################################

    #### public interface ####################################################

    def refresh(self):
        """ Refresh the enabled/disabled state of this action.
        """
        self.enabled = self._check_perspective_exists()


    #### protected interface #################################################

    def _check_perspective_exists(self):
        """ Returns True if we are associated with a workbench window in which
            our perspective exists.  Otherwise, returns False.
        """
        enabled = False

        # We're only enabled if we know what window we're related to
        if self.window is not None:

            # We're only enabled if our perspective actually exists!
            perspective = self.window.get_perspective_by_id(self.perspective_id)
            if perspective is not None and perspective.enabled:
                enabled = True

        return enabled


    def _switch_perspective(self, event):
        """ Called to switch the window to our perspective.
        """
        logger.debug("PerspectiveAction [%s] switching to perspective [%s]",
            self, self.perspective_id)
        perspective = self.window.get_perspective_by_id(self.perspective_id)
        self.window.active_perspective = perspective


#### EOF #####################################################################

