""" The default view menu for a workbench window. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.pyface.action.api import Action, Group
from enthought.traits.api import HasTraits, Instance

# Local imports.
from perspective_menu import PerspectiveMenu


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class ViewMenu(HasTraits):
    """ The default view menu for a workbench window. """

    #### 'ViewMenu' interface #################################################

    # The workbench window that the menu is part of.
    window = Instance('enthought.envisage.workbench.Window')

    ###########################################################################
    # 'ViewMenu' interface.
    ###########################################################################

    def refresh(self):
        """ Refreshes the checked state of the actions in the menu. """

        logger.debug('refreshing view menu')

        view_menu = self.window.menu_bar_manager.find_item('ViewMenu')

        for view in self.window.views:
            action_item = view_menu._find_item(view.id)
            if action_item:
                action         = action_item.action
                action.checked = view.visible

        return

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _initialize(self, window):
        """ Initializes the menu.

        fixme: The view menu already exists in the tool bar, it is currently
        contributed by the workbench plugin. This seems ugly!

        """

        # Find the existing (but empty) view menu.
        view_menu = window.menu_bar_manager.find_item("ViewMenu")

        # Add a group containing the perspective menu.
        if len(window.perspectives) > 0:
            view_menu.append(Group(PerspectiveMenu(window=window)))

        # Add a group containing a 'toggler' for each view.
        view_menu.append(self._create_view_group(self.window))

        # fixme: We need this since the view menu already exists.
        view_menu.changed = True

        # Set the initial checked/unchecked state of the actions.
        self.refresh()

        return

    def _create_view_group(self, window):
        """ Creates a group containing the view 'togglers'. """

        group = Group()
        for view in window.views:
            group.append(self._create_action(window, view))

        return group

    def _create_action(self, window, view):
        """ Creates an action that toggles a view's visibility. """

        # Create an action that hides/shows the view.
        action = Action(
            id         = view.id,
            name       = view.name,
            style      = 'toggle',
            on_perform = self._create_view_toggler(window, view)
        )

        return action

    def _create_view_toggler(self, window, view):
        """ Creates a function that toggles the view visibility. """

        def view_toggler():
            """ A closure that toggles a view's visibility. """

            if view.visible:
                window.hide_view(view)

            else:
                # If the view is already in the window layout, but hidden, then
                # just show it.
                #
                # fixme: This is a little gorpy, reaching into the window
                # layout here, but currently this is the only thing that knows
                # whether or not the view exists but is hidden.
                if window.window_layout.contains_view(view):
                    window.show_view(view)

                # Otherwise, we have to add the view to the layout.
                else:
                    self._add_view(window, view)

            return

        return view_toggler

    def _add_view(self, window, view):
        """ Adds a view to a window. """

        # Is the view in the current perspectives contents list? If it is then
        # we use the positioning information in the perspective item. Otherwise
        # we will use the default positioning specified in the view itself.
        item = self._get_perspective_item(window.active_perspective, view)
        if item is None:
            item = view

        # fixme: This only works because 'PerspectiveItem' and 'View' have the
        # identical 'position', 'relative_to', 'width' and 'height' traits! We
        # need to unify these somehow!
        relative_to = window.get_view_by_id(item.relative_to)
        size = (item.width, item.height)
        window.add_view(view, item.position, relative_to, size)

        # Make it so!
        window.refresh()

        return

    def _get_perspective_item(self, perspective, view):
        """ Returns the perspective item for a view.

        Returns None if the view is not mentioned in the perspectives contents.

        """

        # fixme: Errrr, shouldn't this be a method on the window?!?
        for item in perspective.contents:
            if item.id == view.id:
                break

        else:
            item = None


        return item

    #### Trait change handlers ################################################

    def _window_changed(self, old, new):
        """ Static trait change handler. """

        if old is not None:
            old.on_trait_change(self.refresh,'active_perspective',remove=True)

        if new is not None:
            # Initialize the menu.
            self._initialize(self.window)

            # When the active perspective changes, refresh the menu to show
            # which views are visible.
            new.on_trait_change(self.refresh, 'active_perspective')

        return

#### EOF ######################################################################
