""" Abstract base class for all workbench preference pages. """


# Standard library imports.
import logging

# Enthought library imports.
from enthought.envisage import Application
from enthought.pyface.preference.api import PreferencePage
from enthought.traits.api import Dict, Instance

# Local imports.
from preference_helper import FontPreferenceHelper, PreferenceHelper


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class WorkbenchPreferencePage(PreferencePage):
    """ Abstract base class for all workbench preference pages. """

    # All preference helpers.
    PREFERENCE_HELPERS = [FontPreferenceHelper]

    #### 'WorkbenchPreferencePage' interface ##################################

    # The application that the preference page is part of.
    #
    # fixme: This trait has the '_' prefix because preference pages
    # currently use their list of traits to build their UIs. This is
    # obviously a bit rubbish!
    _application = Instance(Application)

    #### Private interface ####################################################

    # Mapping of trait names to the preference instance that contains the value
    # for the trait.
    _preference_map = Dict

    ###########################################################################
    # 'PreferencePage' interface.
    ###########################################################################

    def create_control(self, parent):
        """ Creates the toolkit-specific control for the page. """

        # Initialize the page with the preferences.
        self._initialize_preferences()

        # The control is just the page's trait sheet!
        ui = self.edit_traits(parent=parent, kind='subpanel')

        return ui.control

    def restore_defaults(self):
        """ Restore the default preference values. """

        # Get the preference value for each of this editor's traits.
        for trait_name in self.traits():
            # fixme: Ignore 'private' traits and any traits defined on the base
            # 'HasTraits' class.
            if trait_name.startswith('_') or trait_name.startswith('trait_'):
                continue

            preferences = self._preference_map[trait_name]

            value = preferences.get_default(trait_name)
            if isinstance(value, PreferenceHelper):
                value = value.get_editor_value(value)

            setattr(self, trait_name, value)

        return

    def show_help_topic(self):
        """ Show the help topic for this preference page. """
        try:
            hp = get_application().get_service('enthought.help.IHelp')
            hp.library.show_topic(self.help_id)
        except SystemError:
            pass # Do nothing if no Help service

        return

    ###########################################################################
    # Protected 'WorkbenchPreferencePage' interface.
    ###########################################################################

    def _get_preferences(self):
        """ Returns the preferences that this page is editing.

        This may be a single instance or a list of instances. The instances are
        expected to implement the 'interface' defined in:-

        'enthought.envisage.core.preferences.Preferences'

        """

        raise NotImplementedError

    def _get_preferences_as_list(self):
        """ Returns a list of the preferences that this page is editing.

        This method simply converts the return type of _get_preferences so that
        code needing preferences can expect a constant return value.

        """

        preferences = self._get_preferences()
        if not isinstance(preferences, list):
            preferences = [preferences]

        return preferences

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _initialize_preferences(self):
        """ Initializes the page with the specified preferences. """

        # Clear out any previous history of what preference instances
        # contained what traits.
        self._preference_map = {}

        # The preferences object that we are editing.
        self._preferences = self._get_preferences_as_list()

        # Get the preference value for each of this editor's traits.
        for trait_name in self.traits():
            # fixme: Ignore 'private' traits and any traits defined on the base
            # 'HasTraits' class.
            if trait_name.startswith('_') or trait_name.startswith('trait_'):
                continue

            # Use the first preferences instance to define a value.
            for preferences in self._preferences:
                if preferences.contains(trait_name):
                    value = preferences.get(trait_name)
                    self._preference_map[trait_name] = preferences
                    break
            else:
                value = None
            if isinstance(value, PreferenceHelper):
                value = value.get_editor_value(value)

            setattr(self, trait_name, value)

        # Even thought we are listening to our own trait changes, we use a
        # dynamic handler so that we don't get called on the above calls to
        # 'setattr'.
        self.on_trait_change(self._on_anytrait_changed)

        return

    ###########################################################################
    # Trait event handlers.
    ###########################################################################

    def _on_anytrait_changed(self, object, trait_name, old, new):
        """ Called when a trait has been changed. """

        for helper in self.PREFERENCE_HELPERS:
            if helper.handles(new):
                new = helper.get_preference_value(new)
                break

        # Traits with built-in shadow traits don't play well with the below
        # attempt to automatically save the current preference so we must wrap
        # it in a try block and just log exceptions.
        try:
            self._preference_map[trait_name].set(trait_name, new)
        except:
            logger.warn('Unable to save preference value for [%s]', trait_name)

        return

#### EOF ######################################################################
