""" A factory for workbench windows. """


# Enthought library imports.
from enthought.pyface.action.api import StatusBarManager
from enthought.traits.api import HasTraits

# Local imports.
from window import Window

# Plugin definition imports.
from workbench_plugin_definition import Workbench


class WindowFactory(HasTraits):
    """ A factory for workbench windows. """
    
    ###########################################################################
    # 'WindowFactory' interface.
    ###########################################################################

    def create_window(self, workbench):
        """ Creates a new workbench window.

        This does NOT open the window. To do that call its 'open' method!

        """

        perspective_name = workbench.preferences.get('default_perspective', '')
        if len(perspective_name) > 0:
            for perspective in workbench.perspectives:
                if perspective.name == perspective_name:
                    default_perspective_id = perspective.id
                    break

            # fixme: This is a quick hack so that if you set the preference
            # to be an invalid name, then we will always use the last one
            # visible!
            else:
                default_perspective_id = ''

        else:
            default_perspective_id = workbench.default_perspective_id
            for perspective in workbench.perspectives:
                if perspective.id == default_perspective_id:
                    workbench.preferences.set(
                        'default_perspective', perspective.name
                    )
                    break
            
        window = Window(
            application            = workbench.application,
            default_perspective_id = default_perspective_id,
            icon                   = workbench.branding.application_icon,
            perspectives           = workbench.perspectives,
            position               = workbench.window_position,
            resource_manager       = workbench.resource_manager,
            size                   = workbench.window_size,
            state_location         = workbench.state_location,
            status_bar_manager     = StatusBarManager(),
            title                  = workbench.branding.application_name,
            # fixme: Each window needs its *own* views. They are not factories!
            views                  = self._clone_views(workbench.views),
            workbench              = workbench
        )

        return window

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _clone_views(self, views):
        """ Clones a list of views.

        fixme: We should use views as factories.

        """

        return [self._clone_view(view) for view in views]

    def _clone_view(self, view):
        """ Clones a single view. """

        # fixme: View need to implement their own clone maybe?
        clone = view.__class__(
            id       = view.id,
            name     = view.name,
            position = view.position,
            visible  = view.visible,
        )

        # fixme: See comment above! This is to clone traits UI views!
        if hasattr(view, 'uol'):
            clone.uol  = view.uol
            clone.view = view.view

        return clone

#### EOF ######################################################################
