""" The plugin that creates the Envisage Workbench UI. """


# Enthought library imports.
from enthought.envisage.api import Plugin
from enthought.pyface.api import ImageResource
from enthought.pyface.dock.api import add_feature

# Local imports.
from branding import Branding
from user_perspective import UserPerspective
from perspective import Perspective
from perspective_item import PerspectiveItem
from services import IWORKBENCH, IWORKBENCH_UI
from traits_ui_view import TraitsUIView
from view import View
from workbench import Workbench
from workbench_ui import WorkbenchUI


class WorkbenchPlugin(Plugin):
    """ The plugin that creates the Envisage Workbench UI. """

    ###########################################################################
    # 'Plugin' interface.
    ###########################################################################

    def start(self, application):
        """ Starts the plug-in. """

        # Add all contributed dock window 'features'.
        self._add_features(application)
        
        # Register the services offered by the plug-in.
        self._register_services(application)

        return

    def save_preferences(self):
        """ Saves the plug-in's user preferences.

        This method actually saves only preferences that the user has changed
        from the plug-in defaults.

        """

        # Get a reference to the workbench service.
        workbench = self.get_service(IWORKBENCH)

        # Save the initial window size and position.
        self.preferences.set('window_position', workbench.window_position)
        self.preferences.set('window_size', workbench.window_size)

        # Save 'em!
        super(WorkbenchPlugin, self).save_preferences()

        return

    ###########################################################################
    # 'Private' interface.
    ###########################################################################

    def _add_features(self, application):
        """ Adds all contributed dock window 'features'. """
        
        from workbench_plugin_definition import Feature
        
        extensions = self.application.get_extensions(Feature)
        for extension in extensions:
            klass = self.import_symbol(extension.class_name)
            add_feature(klass)

        return
    
    def _register_services(self, application):
        """ Registers the services offered by the plug-in. """

        # This plugin offers the following services:
        #
        # 1) A workbench service.
        workbench = self._create_workbench_service(application)
        self.register_service(IWORKBENCH, workbench)

        # 2) A workbench UI service.
        workbench_ui = self._create_workbench_ui_service(workbench)
        self.register_service(IWORKBENCH_UI, workbench_ui)

        return

    def _create_workbench_service(self, application):
        """ Creates and registers the workbench service. """

        # Get all contributions to the 'Workbench' extension point.
        views, perspectives, default_perspective_id = \
               self._get_workbench_contributions()
               
        # Add any user defined perspectives:
        user_perspective = UserPerspective( 
            state_location = self.state_location )
        perspectives.extend( user_perspective.perspectives )

        # Get a reference to the resource manager.
        resource_manager = self.get_service(
            'enthought.envisage.resource.IResourceManager'
        )

        # Create and register a workbench service.
        workbench = Workbench(
            application,
            branding               = self._get_branding(application),
            default_perspective_id = default_perspective_id,
            perspectives           = perspectives,
            user_perspective       = user_perspective,
            preferences            = self.preferences,
            resource_manager       = resource_manager,
            state_location         = self.state_location,
            views                  = views,
            window_position        = self.preferences.get('window_position'),
            window_size            = self.preferences.get('window_size'),
        )

        # Bind user preferences to the workbench.
        self.bind_preference(workbench, 'prompt_on_exit', default=True)
        self.bind_preference(workbench, 'show_tool_names', default=True)

        return workbench

    def _create_workbench_ui_service(self, workbench):
        """ Creates the workbench UI service. """

        return WorkbenchUI(workbench=workbench)

    def _get_workbench_contributions(self):
        """ Returns the details of all workbench contributions. """

        # Plugin definition imports.
        from workbench_plugin_definition import Workbench

        # Collect all view and perspective contributions.
        extensions = self.application.load_extensions(Workbench)

        views = []; perspectives = []; default_perspective_id = ''
        for extension in extensions:
            for definition in extension.views:
                views.append(self._create_view(definition))

            for definition in extension.perspectives:
                perspectives.append(self._create_perspective(definition))

            if len(extension.default_perspective) > 0:
                default_perspective_id = extension.default_perspective

        return views, perspectives, default_perspective_id

    def _get_branding(self, application):
        """ Returns the application branding information. """

        branding = Branding()

        extension = self._get_branding_extension()
        if extension is not None:
            # We locate images relative to the plugin definition file that
            # the branding contribution was defined in.
            path = [extension._definition_.location]

            if len(extension.about_image) > 0:
                branding.about_image = ImageResource(
                    extension.about_image, path
                )
            if len(extension.application_icon) > 0:
                branding.application_icon = ImageResource(
                    extension.application_icon, path
                )

            branding.about_additions = extension.about_additions
            branding.application_name = extension.application_name

        return branding

    def _get_branding_extension(self):
        """ Returns the *first* branding extension.

        Returns None if no branding extension is found.

        """

        from workbench_plugin_definition import Branding

        extensions = self.application.load_extensions(Branding)
        if len(extensions) > 0:
            extension = extensions[0]

        else:
            extension = None

        return extension

    def _create_view(self, definition):
        """ Creates a view implementation. """

        # If a view implementation class was specified then use that.
        if len(definition.class_name) > 0:
            # Import the class that implements the view.
            class_name = definition.class_name
            klass = self.import_symbol(class_name)

            # Create the view implementation.
            view = klass()

            # Initalize the view.
            self._initialize_view(view, definition)

            # If the view does not have an Id then make one up.
            if len(view.id) == 0:
                view.id = class_name

        # Otherwise, use the UOL and view name.
        else:
            view = TraitsUIView(
                uol=definition.uol, view=definition.traits_ui_view
            )
            
            # Initalize the view.
            self._initialize_view(view, definition)

            # If the view does not have an Id then make one up.
            if len(view.id) == 0:
                view.id = definition.uol + '/' + definition.traits_ui_view

        return view

    def _initialize_view(self, view, definition):
        """ Initializes a view from a contribution. """

        # fixme: We need to automate this process some how (or at least
        # semi-automate!).
        view.id          = definition.id
        view.name        = definition.name
        view.visible     = definition.visible
        view.position    = definition.position
        view.relative_to = definition.relative_to
        view.width       = definition.width
        view.height      = definition.height

        return
    
    def _create_perspective(self, definition):
        """ Creates a perspective implementation. """

        # Import the class that implements the perspective. If no class name
        # was specified then we use the default perspective class.
        class_name = definition.class_name
        if len(class_name) > 0:
            klass = self.import_symbol(class_name)

        else:
            klass = Perspective

        # Create the perpective implementation.
        perspective = klass()

        # Initalize the view.
        self._initialize_perspective(perspective, definition)

        # If the perspective does not have an Id then use the class name.
        #
        # fixme: These seems a bad idea since we don't expect most people to
        # have to specify a class name, just a list of contents! If no class
        # name is specified then we should force an Id to be specified.
        if len(perspective.id) == 0:
            perspective.id = class_name

        return perspective

    def _initialize_perspective(self, perspective, definition):
        """ Initializes a perspective from a contribution. """

        # fixme: We need to automate this process some how (or at least
        # semi-automate!).
        perspective.id               = definition.id
        perspective.name             = definition.name
        perspective.editor_area_size = definition.editor_area_size
        perspective.show_editor_area = definition.show_editor_area
        perspective.enabled          = definition.enabled

        # fixme: In any automation attempt this would be an interesting case,
        # since we also have to create implementation classes for each item in
        # the list.
        perspective.contents = [
            self._create_perspective_item(item_definition)
            for item_definition in definition.contents
        ]

        return

    def _create_perspective_item(self, item_definition):
        """ Creates and initializes a perspective item from a contribution. """

        # fixme: So this is an interesting case, the definition and the
        # implementation have *exactly* the same traits, but obviously, we
        # can't really use the implementation class in the plugin definition
        # nor the definition class here. Hmmmmm... Will this be a common
        # pattern?
        item = PerspectiveItem()
        item.copy_traits(item_definition)
        
        return item

### EOF ######################################################################
