#include "gtk_epg_widget.h"
#include "application.h"
#include "channel_manager.h"
#include "exception_handler.h"
#include <math.h>

GtkEpgWidget::GtkEpgWidget()
{
	Application& application = Application::get_current();
	Glade& glade = application.get_glade();
	
	offset = 0;
	selected_event = NULL;
	
	vbox_epg			= glade.get_widget("vbox_epg");
	table_epg			= glade.get_widget("table_epg");
	scrolled_window_epg	= glade.get_widget("scrolled_window_epg");
	button_epg_previous	= glade.get_widget("button_epg_previous");
	button_epg_now		= glade.get_widget("button_epg_now");
	button_epg_next		= glade.get_widget("button_epg_next");
	
	g_signal_connect( G_OBJECT ( button_epg_previous ), "clicked", G_CALLBACK ( button_epg_previous_clicked ), this );		
	g_signal_connect( G_OBJECT ( button_epg_now ), "clicked", G_CALLBACK ( button_epg_now_clicked ), this );		
	g_signal_connect( G_OBJECT ( button_epg_next ), "clicked", G_CALLBACK ( button_epg_next_clicked ), this );		
	
	last_number_rows = 0;
	last_number_columns = 0;
}

void GtkEpgWidget::set_selected_event(xmlNodePtr event)
{
	selected_event = event;
	EpgEvent e(event);
	set_offset(e.get_start_time() - DateTime::now_utc());
}

void GtkEpgWidget::set_offset(gint value)
{
	if (value < 0)
	{
		value = 0;
	}
	
	offset = value;
	
	update();
}

void GtkEpgWidget::hide()
{
	gtk_widget_hide(vbox_epg);
}

void GtkEpgWidget::show()
{
	gtk_widget_show(vbox_epg);
}

gboolean GtkEpgWidget::is_visible()
{
	return GTK_WIDGET_VISIBLE(vbox_epg);
}

void GtkEpgWidget::update()
{
	GtkAdjustment* hadjustment = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(scrolled_window_epg));
	GtkAdjustment* vadjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(scrolled_window_epg));
	
	gdouble hvalue = gtk_adjustment_get_value(hadjustment);
	gdouble vvalue = gtk_adjustment_get_value(vadjustment); 

	update_table();
	
	gtk_adjustment_set_value(hadjustment, hvalue);
	gtk_adjustment_set_value(vadjustment, vvalue);	
}

gint epg_channel_sort_func(gconstpointer a, gconstpointer b)
{
	Channel* channel_a = (Channel*)a;
	Channel* channel_b = (Channel*)b;
	return g_ascii_strcasecmp(channel_a->name.c_str(), channel_b->name.c_str());
}

void GtkEpgWidget::update_table()
{
	Application& application = Application::get_current();
	ChannelManager& channel_manager = application.get_channel_manager();
	Epg& epg = application.get_epg();
	Configuration& configuration = application.get_configuration();

	Channel& video_channel = application.get_video_channel();
	span_hours = (gsize)configuration.get_int_value("epg_span_hours");
	span_minutes = span_hours * 60;
	span_seconds = span_minutes * 60;
	
	String next_text = String::format(_("Next %d hours >"), span_hours);
	String previous_text = String::format(_("< Previous %d hours"), span_hours);
	
	gtk_button_set_label(GTK_BUTTON(button_epg_next), next_text.c_str());
	gtk_button_set_label(GTK_BUTTON(button_epg_previous), previous_text.c_str());

	number_rows = channel_manager.get_channel_count() + 1;
	number_columns = (span_hours * 6) + 1;

	if (last_number_rows != number_rows || last_number_columns != number_columns)
	{
		gtk_table_resize(GTK_TABLE(table_epg), number_rows, number_columns);
		
		last_number_rows = number_rows;
		last_number_columns = number_columns;
	}

	GSList* unsorted_channels = channel_manager.get_channels();
	GSList* channels = NULL;
	
	while (unsorted_channels != NULL)
	{
		channels = g_slist_append(channels, unsorted_channels->data);
		unsorted_channels = g_slist_next(unsorted_channels);
	}
	
	if (configuration.get_boolean_value("sort_channels_alphabetically"))
	{
		channels = g_slist_sort(channels, epg_channel_sort_func);
	}
	
	GList* children = gtk_container_get_children(GTK_CONTAINER(table_epg));
	while (children != NULL)
	{
		gtk_widget_destroy(GTK_WIDGET(children->data));
		children = g_list_next(children);
	}
	
	guint now_utc = DateTime::now_utc();
	guint now = now_utc + offset;
	guint start_time = (now / 600) * 600;
	guint end_time = start_time + span_seconds;

	for (guint count = 0; count < span_hours; count++)
	{
		DateTime slot(start_time + 3600*count, true);		
		String text = slot.ftime("<b>%A, %x, %H:%M</b>");
		GtkWidget* button = attach_button(text.c_str(), count*6 + 1, (count+1)*6 + 1, 0, 1);
		GdkColor current_colour = {0, 0x8080, 0x8080,  0xFFFF};
		gtk_widget_modify_bg(button, GTK_STATE_NORMAL, &current_colour);
	}

	guint row = 0;
	while (channels != NULL)
	{
		Channel& channel = *(Channel*)(channels->data);
		create_channel_row(channel, ++row, channel == video_channel, epg, now, start_time, end_time);
		channels = g_slist_next(channels);
	}
	
	gtk_widget_show_all(table_epg);
}

void GtkEpgWidget::create_channel_row(Channel& channel, guint row, gboolean selected, Epg& epg, guint now, guint start_time, guint end_time)
{	
	String channel_name = encode(channel.name);
	String channel_name_text = String::format("<b>%s</b>", channel_name.c_str());
	GtkWidget* button_channel = attach_button(channel_name_text.c_str(), 0, 1, row, row+1);
	gtk_widget_set_tooltip_text(button_channel, channel.name.c_str());
	gtk_widget_set_has_tooltip(button_channel, false);
	g_signal_connect( G_OBJECT ( button_channel ), "clicked", G_CALLBACK ( button_channel_name_clicked ), this );		
	
	if (selected)
	{
		GdkColor current_colour = {0, 0x8080, 0xFFFF, 0x8080};
		gtk_widget_modify_bg(button_channel, GTK_STATE_NORMAL, &current_colour);
	}
	else
	{
		GdkColor current_colour = {0, 0x8080, 0x8080,  0xFFFF};
		gtk_widget_modify_bg(button_channel, GTK_STATE_NORMAL, &current_colour);
	}

	guint total_number_columns = 0;
	
	XPathResult results = epg.get_events(channel);
	guint count = results.get_count();
	if (count > 0)
	{
		GList* sorted_events = NULL;
		GList* iterator = NULL;

		for (guint index = 0; index < count; index++)
		{
			xmlNodePtr node = results.get_result(index);
			sorted_events = g_list_insert_sorted(sorted_events,
				node,
				(GCompareFunc)EpgEvent::compare_events);
		}

		guint last_event_end_time = 0;
		
		iterator = sorted_events;
		while (iterator != NULL)
		{
			xmlNodePtr node = (xmlNodePtr)iterator->data;
			EpgEvent event(node);
						
			guint event_start_time = event.get_start_time();
			guint event_duration = event.get_duration();
			guint event_end_time = event_start_time + event_duration;

			if (
				(event_end_time > start_time && event_end_time < end_time)
				||
				(event_start_time < end_time && event_start_time > start_time)
				||
				(event_start_time < start_time && event_end_time > end_time)
			)
			{
				String time_text = DateTime(event_start_time, true).to_time_string();
				time_text += "-" + DateTime(event_end_time, true).to_time_string();
				
				guint start_column = 0;
				if (event_start_time < start_time)
				{
					start_column = 0;
				}
				else
				{
					start_column = (guint)round((event_start_time - start_time) / 600.0);
				}
												
				guint end_column = (guint)round((event_end_time - start_time) / 600.0);
				if (end_column > number_columns-1)
				{
					end_column = number_columns-1;
				}
				
				guint column_count = end_column - start_column;
				if (start_column >= total_number_columns && column_count > 0)
				{
					// If there's a gap, plug it
					if (start_column > total_number_columns)
					{
						// If it's a small gap then just extend the event
						if (event_start_time - last_event_end_time <= 2 * 60)
						{
							start_column = total_number_columns;
							column_count = end_column - start_column;
						}
						else
						{
							guint empty_columns = start_column - total_number_columns;
							attach_button(UNKNOWN_TEXT, total_number_columns + 1, start_column + 1, row, row + 1);
							total_number_columns += empty_columns;
						}
					}
															
					if (column_count > 0)
					{
						String encoded_title = encode(event.get_title());						
						String text = String::format("<b>%s</b>\n%s", time_text.c_str(), encoded_title.c_str());
						GtkWidget* button = attach_button(text, start_column + 1, end_column + 1, row, row + 1);
						g_signal_connect( G_OBJECT ( button ), "clicked", G_CALLBACK ( button_program_clicked ), this );

						String tag = String::format("%s:%d", channel.name.c_str(), event.get_event_id());
						gtk_widget_set_tooltip_text(button, tag.c_str());
						gtk_widget_set_has_tooltip(button, false);

						if (start_column == 0 && offset == 0 && selected)
						{
							GdkColor colour = {0, 0x8080, 0xFFFF, 0x8080};
							gtk_widget_modify_bg(button, GTK_STATE_NORMAL, &colour);
						}
						else if (selected_event != NULL)
						{
							EpgEvent e(selected_event);
							if (e == event)
							{
								GdkColor colour = {0, 0xFFFF, 0xFFFF, 0x8080};
								gtk_widget_modify_bg(button, GTK_STATE_NORMAL, &colour);
							}
						}
					}

					total_number_columns += column_count;
				}
				last_event_end_time = event_end_time;
			}

			iterator = g_list_next(iterator);
		}
		
		g_list_free(sorted_events);
	}
	
	if (total_number_columns < number_columns-1)
	{		
		attach_button(UNKNOWN_TEXT, total_number_columns + 1, number_columns, row, row + 1);
	}
}

GtkWidget* GtkEpgWidget::attach_button(const String& text, guint left_attach, guint right_attach, guint top_attach, guint bottom_attach)
{
	GtkWidget* button = gtk_button_new_with_label(text.c_str());
	attach_widget(button, left_attach, right_attach, top_attach, bottom_attach);
	gtk_button_set_alignment(GTK_BUTTON(button), 0, 0.5);
	GtkWidget* label = gtk_bin_get_child(GTK_BIN(button));
	gtk_label_set_use_markup(GTK_LABEL(label), true);
	return button;
}

GtkWidget* GtkEpgWidget::attach_label(const String& text, guint left_attach, guint right_attach, guint top_attach, guint bottom_attach)
{
	GtkWidget* label = gtk_label_new(text.c_str());
	attach_widget(label, left_attach, right_attach, top_attach, bottom_attach);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_label_set_use_markup(GTK_LABEL(label), true);
	return label;
}

void GtkEpgWidget::attach_widget(GtkWidget* widget, guint left_attach, guint right_attach, guint top_attach, guint bottom_attach)
{
	gtk_table_attach(GTK_TABLE(table_epg), widget, left_attach, right_attach, top_attach, bottom_attach, GTK_FILL, GTK_FILL, 2, 2);
}

void GtkEpgWidget::button_channel_name_clicked(GtkButton* button, gpointer data)
{
	TRY;
	const gchar* channel_name = gtk_widget_get_tooltip_text(GTK_WIDGET(button));
	Application::get_current().change_channel(channel_name);
	CATCH;
}

void GtkEpgWidget::button_program_clicked(GtkButton* button, gpointer data)
{
	TRY;
	const gchar* tag = gtk_widget_get_tooltip_text(GTK_WIDGET(button));
	
	StringSplitter parameters(tag, ":", 2);
	
	Application::get_current().show_program_details_dialog(
		parameters.get_value(0),
		parameters.get_int_value(1));
	CATCH;
}

void GtkEpgWidget::button_epg_previous_clicked(GtkButton* button, gpointer data)
{
	TRY;
	GtkEpgWidget* widget = (GtkEpgWidget*)data;	
	widget->set_offset(widget->offset - widget->span_seconds);
	CATCH;
}

void GtkEpgWidget::button_epg_now_clicked(GtkButton* button, gpointer data)
{
	TRY;
	GtkEpgWidget* widget = (GtkEpgWidget*)data;
	widget->set_offset(0);
	CATCH;
}

void GtkEpgWidget::button_epg_next_clicked(GtkButton* button, gpointer data)
{
	TRY;
	GtkEpgWidget* widget = (GtkEpgWidget*)data;
	widget->set_offset(widget->offset + widget->span_seconds);
	CATCH;
}

String GtkEpgWidget::encode(const String& s)
{
	String result = s;
	
	result = result.replace("&", "&amp;");
	result = result.replace("<", "&lt;");
	result = result.replace(">", "&gt;");

	return result;
}
