#include "scheduled_recording.h"

ScheduledRecording::ScheduledRecording()
{
	node		= NULL;
	start_time	= 0;
	duration	= 0;
	type		= RECORDING_TYPE_NONE;
}

ScheduledRecording::ScheduledRecording(xmlNodePtr n)
{
	load(n);
}

void ScheduledRecording::load(xmlNodePtr n)
{
	XmlNode scheduled_recording_node(n);
	node			= n;		
	type			= scheduled_recording_node.get_int_attribute_value("type");
	channel_name	= scheduled_recording_node.get_attribute_value("channel");
	description		= scheduled_recording_node.get_attribute_value("description");
	start_time		= scheduled_recording_node.get_int_attribute_value("start_time");
	duration		= scheduled_recording_node.get_int_attribute_value("duration");
}

String ScheduledRecording::get_type_text()
{
	String result;
	
	switch(type)
	{
	case RECORDING_TYPE_ONCE_OFF:			result = "Once off"; break;
	case RECORDING_TYPE_EVERY_DAY_NOT_WE:	result = "Every day except weekends"; break;
	case RECORDING_TYPE_EVERY_DAY_WE:		result = "Every day including weekends"; break;
	case RECORDING_TYPE_EVERY_WEEK:			result = "Every week"; break;
	}
	
	return result;
}

String ScheduledRecording::get_start_time_text()
{
	String result = UNKNOWN_TEXT;
	
	switch(type)
	{
	case RECORDING_TYPE_ONCE_OFF:
		result = EpgEvent::start_time_to_string(start_time);
		break;
	
	case RECORDING_TYPE_EVERY_DAY_NOT_WE:
	case RECORDING_TYPE_EVERY_DAY_WE:
		{
			DateTime dt(get_local_start_time());
			result = dt.to_time_string();
		}
		break;
	
	case RECORDING_TYPE_EVERY_WEEK:
		{
			DateTime dt(get_local_start_time());
			
			result = dt.get_day_of_week_text();
			result += ", ";
			result += dt.to_time_string();
		}
		break;
	
	default:
		result = UNKNOWN_TEXT;
	}
	
	return result;
}

time_t ScheduledRecording::get_local_start_time() const
{
	time_t result = 0;
	
	switch(type)
	{
	case RECORDING_TYPE_ONCE_OFF:
		result = start_time - timezone;
		break;
	case RECORDING_TYPE_EVERY_DAY_WE:
	case RECORDING_TYPE_EVERY_DAY_NOT_WE:
		result = (start_time + timezone) % SECONDS_IN_DAY;
		break;
	case RECORDING_TYPE_EVERY_WEEK:
		result = (start_time + timezone + (3*SECONDS_IN_DAY)) % SECONDS_IN_WEEK;
		break;
	}

	return result;
}

gboolean ScheduledRecording::is_current()
{
	time_t now = 0;

	switch(type)
	{
	case RECORDING_TYPE_ONCE_OFF:
		now = DateTime::now_utc();
		break;			
	case RECORDING_TYPE_EVERY_DAY_WE:
	case RECORDING_TYPE_EVERY_DAY_NOT_WE:
		now = DateTime::now() - timezone + daylight * SECONDS_IN_HOUR;
		now %= SECONDS_IN_DAY;
		break;
	case RECORDING_TYPE_EVERY_WEEK:
		now = DateTime::now() - timezone + daylight * SECONDS_IN_HOUR - 3 * SECONDS_IN_DAY;
		now %= SECONDS_IN_WEEK;
		break;
	}
	
	gboolean result = now >= start_time && now < (start_time + duration);
	
	// Check for overflow
	switch(type)
	{
	case RECORDING_TYPE_EVERY_DAY_WE:
	case RECORDING_TYPE_EVERY_DAY_NOT_WE:
		if (start_time + duration  > SECONDS_IN_DAY)
		{
			result |= now < (start_time + duration - SECONDS_IN_DAY);
		}
		break;
	case RECORDING_TYPE_EVERY_WEEK:
		if (start_time + duration  > SECONDS_IN_WEEK)
		{
			result |= now < (start_time + duration - SECONDS_IN_WEEK);
		}
		break;
	}
	
	/*
	Log::write(G_LOG_LEVEL_DEBUG, "	Checking: 	'%s'", description.c_str());
	Log::write(G_LOG_LEVEL_DEBUG, "	Start Time:	'%d'", start_time);
	Log::write(G_LOG_LEVEL_DEBUG, "	End Time:	'%d'", start_time + duration);
	Log::write(G_LOG_LEVEL_DEBUG, "	Now:		'%d'", now);
	Log::write(G_LOG_LEVEL_DEBUG, "	Now Hour:	'%d'", now / SECONDS_IN_HOUR);
	Log::write(G_LOG_LEVEL_DEBUG, "	Now Minute:	'%d'", (now % SECONDS_IN_HOUR) / 60);
	Log::write(G_LOG_LEVEL_DEBUG, "	Result:		'%d'", result);
	*/
	
	return result;
}

String ScheduledRecording::get_duration_text()
{
	return EpgEvent::duration_to_string(duration);
}

