/*******************************************************************************
 * Copyright (C) 2004-2007 Intel Corp. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *   - Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright notice,
 *     this list of conditions and the following disclaimer in the documentation
 *     and/or other materials provided with the distribution.
 * 
 *   - Neither the name of Intel Corp. nor the names of its
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

#include "atchkConfig.h"
#include "ConfigFile.h"

#ifdef _WIN32
#include "ConfigWinInfo.h"
#else
#include "ConfigLinuxInfo.h"
#endif

#include <fstream>
using namespace std;


//*******************************************************************************
//
// Class:	    CATCheckConfig
//
// Description:	    This class holds configuration data for the MEStatus application.
//                  There are two configuration files - a "Machine Config" file and
//                  a "User Config" file. Machine Config is general for the machine,
//                  and User Config is user-specific.
//                  If a configuration file is not found, or the file is found but 
//                  missing some parameters, defaults will be used.
//                  
//********************************************************************************

/*Constants*/
//Constants for configuration parameters names
const string CATCheckConfig::CHECK_AMT_STATUS_TIMEOUT = "CheckAMTStatusTimeout";
const string CATCheckConfig::OEM_URL = "OemUrl";
const string CATCheckConfig::AMT_STATUS_INIT_MAX_RETRIES = "AMTStatusInitMaxRetries";
const string CATCheckConfig::MINIMIZE_AT_START = "MinimizePrivacyIconAtStart";
const string CATCheckConfig::AMT_LAST_ENABLED = "AMTLastEnabled";
const string CATCheckConfig::POPUP_FOR_CURRENT_USER = "PopupForCurrentUser"; 

//Default values for configuration parameters
const unsigned long CATCheckConfig::DEFAULT_CHECK_AMT_STATUS_TIMEOUT = 10000;
const string CATCheckConfig::DEFAULT_OEM_URL = "http://www.intel.com/vpro";
const unsigned long CATCheckConfig::DEFAULT_AMT_STATUS_INIT_MAX_RETRIES = 7;
const bool CATCheckConfig::DEFAULT_MINIMIZE_AT_START = false;
const bool CATCheckConfig::DEFAULT_POPUP_FOR_CURRENT_USER = true;
const bool CATCheckConfig::DEFAULT_AMT_LAST_ENABLED = false;

//Constants for configuration file names
const string CATCheckConfig::MACHINE_CFG_FILENAME = "IntelAMTCheck.config";
const string CATCheckConfig::USER_CFG_FILENAME = "IntelAMTCheck.user.config";

//ctor
CATCheckConfig::CATCheckConfig()
{
	LoadDefaults();
}


//dtor
CATCheckConfig::~CATCheckConfig()
{
	//Save user configuration when object is destroyed.
	SaveUserConfig();
}

void CATCheckConfig::LoadDefaults()
{
	m_nCheckAMTStatusTimeout = DEFAULT_CHECK_AMT_STATUS_TIMEOUT;
	m_OemUrl = DEFAULT_OEM_URL;
	m_nAMTStatusInitMaxRetries = DEFAULT_AMT_STATUS_INIT_MAX_RETRIES;
	m_bMinimizeAtStart = DEFAULT_MINIMIZE_AT_START;

	//User-specific params
	m_bPopupForCurrentUser = DEFAULT_POPUP_FOR_CURRENT_USER;
	m_bAMTLastEnabled = DEFAULT_AMT_LAST_ENABLED;
}

//Init(): Load the configuration from configuration files.
//Returns true on success, false if either of the files could not be opened (in which
//case the defaults will be used).
bool CATCheckConfig::Init()
{
	LoadDefaults();
#ifdef _WIN32
	ConfigWinInfo OSInfo;
#else
	ConfigLinuxInfo OSInfo;
#endif

	m_MachineConfigFileName = OSInfo.MachineConfigPrefix() + MACHINE_CFG_FILENAME;
	m_UserConfigFileName = OSInfo.UserConfigPrefix() + USER_CFG_FILENAME;
	ifstream machineConfigFile(m_MachineConfigFileName.c_str());
	ifstream userConfigFile(m_UserConfigFileName.c_str());

	//If either file couldn't be opened, we use the defaults and return false in the end.
	bool bRetVal = (machineConfigFile.is_open() && userConfigFile.is_open());

	if (machineConfigFile.is_open())
	{
		LoadMachineConfig(machineConfigFile);
	}

	if (userConfigFile.is_open())
	{
		LoadUserConfig(userConfigFile);
	}

	return bRetVal;
}

void CATCheckConfig::LoadMachineConfig(std::ifstream &input)
{
	ConfigFile MachineConfigFile;
	input >> MachineConfigFile;

	//For every parameter that is read, if it's not found, its default value is used.
	m_nCheckAMTStatusTimeout = MachineConfigFile.read(CHECK_AMT_STATUS_TIMEOUT, DEFAULT_CHECK_AMT_STATUS_TIMEOUT);
	m_OemUrl = MachineConfigFile.read(OEM_URL, DEFAULT_OEM_URL);
	m_nAMTStatusInitMaxRetries = MachineConfigFile.read(AMT_STATUS_INIT_MAX_RETRIES, DEFAULT_AMT_STATUS_INIT_MAX_RETRIES);
	m_bMinimizeAtStart = MachineConfigFile.read(MINIMIZE_AT_START, DEFAULT_MINIMIZE_AT_START);
}

void CATCheckConfig::LoadUserConfig(std::ifstream &input)
{
	ConfigFile UserConfigFile;
	input >> UserConfigFile;

	if (!UserConfigFile.readInto(m_bPopupForCurrentUser, POPUP_FOR_CURRENT_USER))
	{
		//If the PopupForCurrentUser parameter is not found, we default to MinimizeAtStart.
		m_bPopupForCurrentUser = m_bMinimizeAtStart;
	}
	m_bAMTLastEnabled = UserConfigFile.read(AMT_LAST_ENABLED, DEFAULT_AMT_LAST_ENABLED);
}

bool CATCheckConfig::SaveUserConfig()
{
	ConfigFile UserConfigFile;
	UserConfigFile.add(POPUP_FOR_CURRENT_USER, m_bPopupForCurrentUser);
	UserConfigFile.add(AMT_LAST_ENABLED, m_bAMTLastEnabled);
	ofstream OutputFile(m_UserConfigFileName.c_str());
	if (!OutputFile.is_open())
		return false; //File couldn't be opened
	OutputFile << UserConfigFile;
	return true;
}

unsigned long CATCheckConfig::CheckAMTStatusTimeout() const
{
	return m_nCheckAMTStatusTimeout;
}

bool CATCheckConfig::MinimizeAtStart() const
{
	return m_bMinimizeAtStart;
}

bool CATCheckConfig::PopupForCurrentUser() const
{
	return m_bPopupForCurrentUser;
}

bool CATCheckConfig::AMTLastEnabled() const
{
	return m_bAMTLastEnabled;
}

string CATCheckConfig::GetOemUrl() const
{
	return m_OemUrl;
}

string CATCheckConfig::GetDefaultOemUrl() const
{
	return DEFAULT_OEM_URL;
}

unsigned long CATCheckConfig::AMTStatusInitMaxRetries() const
{
	return m_nAMTStatusInitMaxRetries;
}

//Modifiers
bool CATCheckConfig::SetPopupForCurrentUser(bool bPopup)
{
	m_bPopupForCurrentUser = bPopup;
	return SaveUserConfig();
}

bool CATCheckConfig::SetAMTLastEnabled(bool bEnabled)
{
	m_bAMTLastEnabled = bEnabled;
	return SaveUserConfig();
}
