//
// =================================================================
//
//  MacroMolecular Data Base (MMDB)
//
//  File  sel_exmp6.cpp
//
//  Example:
//     Selection of residues and chains. Parameters are chosen for
//  file rnase.pdb (entry 1sar).
//
//  04 May 2001
//
//  SGI make:  f77 -o sel_exmp6 sel_exmp6.cpp mmdb.a -lm -lC
//
// =================================================================
//

#ifndef  __STRING_H
#include <string.h>
#endif

#ifndef  __MMDB_Manager__
#include "mmdb_manager.h"
#endif



void  PrintInstructions()  {
  printf ( 
    "\n"
    "A MacroMolecular Data Base (MMDB) example:\n"
    "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "Selecting residues and chains.\n"
    "\n"
    "Command line syntax:\n"
    "~~~~~~~~~~~~~~~~~~~~\n"
    "\n"
    "sel_exmp2 coor_file\n"
    "\n"
    "where 'coor_file' is a coordinate file in PDB, mmCIF or\n"
    "BINary format.\n"
    "\n"
         );
}


int main ( int argc, char ** argv, char ** env )  {
PCMMDBManager  MMDBManager;
int            RC,lcount;
char           S[500];
int            selHnd,selHnd1,nSelResidues,nSelAtoms,nSelChains,i;
PPCResidue     SelResidue;
PPCAtom        SelAtom;
PPCChain       SelChain;

  if (argc<2)  {
    PrintInstructions();
    return 1;
  }

  //  1.  Make routine initializations, which must always be done
  //      before working with MMDB
  InitMatType();

  //  2.  Create an instance of MMDB
  MMDBManager = new CMMDBManager();

  //  3.  Read coordinate file.
  //    3.1 Set all necessary read flags -- check with the top of
  //        file  mmdb_file.h  as needed
  MMDBManager->SetFlag ( MMDBF_PrintCIFWarnings );

  //    3.2 MMDB provides the following function for reading
  //        coordinate files with automatic format recognition
  //        for PDB, mmCIF and MMDB BINary files:
  RC = MMDBManager->ReadCoorFile ( argv[1] );

  //    3.3 Check for possible errors:
  if (RC) {
    //  An error was encountered. MMDB provides an error messenger
    //  function for easy error message printing.
    printf ( " ***** ERROR #%i READ:\n\n %s\n\n",RC,GetErrorDescription(RC) );
    //  Location of the error may be identified as precise as line
    //  number and the line itself (PDB only. Errors in mmCIF are
    //  located by category/item name. Errors of reading BINary files
    //  are not locatable and the files are not editable). This
    //  information is now retrieved from MMDB input buffer:
    MMDBManager->GetInputBuffer ( S,lcount );
    if (lcount>=0) 
      printf ( "       LINE #%i:\n%s\n\n",lcount,S );
    else if (lcount==-1)
      printf ( "       CIF ITEM: %s\n\n",S );
    //  dispose instance of MMDB and quit:
    delete MMDBManager;
    return 2;
  } else  {
    //  MMDB allows to identify the type of file that has been just
    //  read:
    switch (MMDBManager->GetFileType())  {
      case MMDB_FILE_PDB    :  printf ( " PDB"         );  break;
      case MMDB_FILE_CIF    :  printf ( " mmCIF"       );  break;
      case MMDB_FILE_Binary :  printf ( " MMDB binary" );  break;
      default : printf ( " Unknown (report as a bug!)" );
    }
    printf ( " file %s has been read in.\n",argv[1] );
  }


  //  4.  Select atoms.
  //    4.1 Each _new_ selection starts with creation of selection
  //        handle (a handle may be used in several selections)
  selHnd = MMDBManager->NewSelection();

  //    4.2 Select residues as needed. In this examples, we select only
  //        residues of chain A containing sulphur. Check the function
  //        and its parameters in file  mmdb_selmngr.h
  MMDBManager->Select ( selHnd,STYPE_RESIDUE, 0,"A",ANY_RES,"*",ANY_RES,"*",
                        "*","*","S","*",SKEY_NEW );

  //    4.3 Selected residues may be accessed through the selection
  //        index. Selection index is merely a vector of pointers
  //        on the selected residues. Check the function and its
  //        parameters in file  mmdb_selmngr.h
  MMDBManager->GetSelIndex ( selHnd, SelResidue,nSelResidues );

  printf ( " point 4.3  %i residues selected.\n",nSelResidues );
  for (i=0;i<nSelResidues;i++)
    if (SelResidue[i])
      printf ( " %4i. %s\n",i+1,SelResidue[i]->GetResidueID(S) );

  //    4.4 Add GLY and SER residues
  MMDBManager->Select ( selHnd,STYPE_RESIDUE, 0,"A",ANY_RES,"*",ANY_RES,"*",
                        "GLY,SER","*","*","*",SKEY_OR );
  MMDBManager->GetSelIndex ( selHnd, SelResidue,nSelResidues );
  printf ( " point 4.4  %i residues selected.\n",nSelResidues );
  for (i=0;i<nSelResidues;i++)
    if (SelResidue[i])
      printf ( " %4i. %s\n",i+1,SelResidue[i]->GetResidueID(S) );


  //    4.5 Leave those falling into a sphere
  MMDBManager->SelectSphere ( selHnd,STYPE_RESIDUE,
                              60.5370,-0.2350,17.3250,10.0,
                              SKEY_AND  );
  MMDBManager->GetSelIndex ( selHnd, SelResidue,nSelResidues );
  printf ( " point 4.5  %i residues selected.\n",nSelResidues );
  for (i=0;i<nSelResidues;i++)
    if (SelResidue[i])
      printf ( " %4i. %s\n",i+1,SelResidue[i]->GetResidueID(S) );

  //    4.6 Select all atoms in selected residues
  selHnd1 = MMDBManager->NewSelection();
  MMDBManager->Select ( selHnd1,STYPE_ATOM,selHnd,SKEY_NEW );

  MMDBManager->GetSelIndex ( selHnd1, SelAtom,nSelAtoms );
  printf ( " point 4.6  %i atoms selected.\n",nSelAtoms );
  for (i=0;i<nSelAtoms;i++)
    if (SelAtom[i])
      printf ( " %4i. %s\n",i+1,SelAtom[i]->GetAtomID(S) );

  //    4.7 Select all Calpha atoms in selected residues.
  //        This can be done in at least two ways; we
  //        choose here not the most efficient one in
  //        order to demonstrate the functionality
  //        of the selection propagation function
  MMDBManager->Select ( selHnd1,STYPE_ATOM, 0,"A",ANY_RES,"*",ANY_RES,"*",
                        "*","CA","C","*",SKEY_NEW );
  MMDBManager->Select ( selHnd1,STYPE_ATOM,selHnd,SKEY_AND );

  MMDBManager->GetSelIndex ( selHnd1, SelAtom,nSelAtoms );
  printf ( " point 4.7  %i Calpha atoms selected.\n",nSelAtoms );
  for (i=0;i<nSelAtoms;i++)
    if (SelAtom[i])
      printf ( " %4i. %s %10.4f %10.4f %10.4f\n",i+1,
               SelAtom[i]->GetAtomID(S),SelAtom[i]->x,
               SelAtom[i]->y,SelAtom[i]->z );

  //    4.8 Select all the chain which contains selected
  //        residues. Make a notice of using SKEY_NEW in
  //        the following operator.
  MMDBManager->Select ( selHnd1,STYPE_CHAIN,selHnd,SKEY_NEW );

  MMDBManager->GetSelIndex ( selHnd1, SelChain,nSelChains );
  printf ( " point 4.8  %i chains selected.\n",nSelChains );
  for (i=0;i<nSelChains;i++)
    if (SelChain[i])
      printf ( " %4i. %s\n",i+1, SelChain[i]->GetChainID(S) );

  //  5. A formally correct style of programming requires you
  //     to explicitely release any allocated memory before
  //     your application quits.
  //     NOTE: we do not dispose the selection indices.
  //     They will be disposed by their manager automatically,
  //     therefore an attempt to dispose them from the application
  //     will cause a crash.
  if (MMDBManager)  delete MMDBManager;

  return 0;

}



