#################################################################
# mmucl.tcl - base mmucl library
# 
# Copyright (C) 1997-2002 Mark Patton
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
###################################################################

namespace eval mmucl {}

source [file join $config(lib_dir) lib misc.tcl]
source [file join $config(lib_dir) lib connect.tcl]

# All user scripts are evaluated in a seperate interp. The interp name
# is $ses where $ses is a session id. In theory the Mmucl array
# shouldn't be examined by interfaces. Instead the various MCname commands
# should be used to manipulate it. Practice is somewhat different. :)
#
# $Mmucl(cur) is the current session.
# $Mmucl(sessions) is the list of current sessions.
# $Mmucl(ses,$ses,*) has per session vars can mostly be read.
# $Mmucl(cfg,*) can be read and written.
# $Mmucl(pref,*) all of these vars are saved in a per interface file on exit.
#
# Procs of form "MCname" are made available as aliases
# in the user interp. Some MC commands are stubs that the interface can
# override.
#
# The interface must define a couple procs:
#
# mmucl::interface_init - called when Mmucl has finished initialization
#
# mmucl::display ses str
#   Display str to the user in session cur.
#
# mmucl::event ses event args
#   Informs the interface of various events.
#
#   connect      - successful connection
#   attempt      - about to attempt a connection
#   disconnect   - connection closed or connection attempt failure
#   timeout      - connection attempt timed out
#   ses_new old_ses - new session created
#   ses_switch old_ses  - current session changed
#   ses_close old_ses  - session old_ses closed
#
# Optional procs an interface may define:
#
# mmucl::save_config       - called with Mmucl is about to exit
#
# Command stubs an interface may override:
#
# mmucl::MChelp ses node   - command to display help
# mmucl::MCcline           - manipulate command line
# mmucl::MCbell            - ring a bell
# mmucl::MCstatus          - manipulate status line
# mmucl::MCclear           - clear the terminal

proc mmucl::init {} {
    variable Mmucl
    variable Ansi
    global argv env config

    array set Ansi {
	reset 0 bold 1 dim 2 underline 4 blink 5 reverse 7
	
	black 30 red 31 green 32 yellow 33 brown 33 blue 34
	magenta 35 cyan 36 grey 37 gray 37 white 37 default 39
	
	bg_black 40 bg_red 41 bg_green 42 bg_yellow 43 bg_brown 43
	bg_blue 44 bg_magenta 45 bg_cyan 46 bg_grey 47 bg_gray 47
	bg_white 47 bg_default 49
    }

    set Mmucl(default_cfg) {
	cfg,reconnect      1
	cfg,actions        1
	cfg,subs           1
	cfg,error_color    red
	cfg,report_color   {bold green}
	cfg,print_color    {bold}
	cfg,script_char    /
	cfg,verbatim_char  \\
	cfg,timeout        20
	cfg,strip_ansi     1
	cfg,action_by_line 0
	cfg,echo           1
	cfg,echo_color     yellow
	cfg,hist_min       3
	cfg,hist_keep      100
	cfg,keep_line      1
	cfg,split_char     \;
	cfg,use_threads    1
	cfg,use_mccp       1
    }

    array set Mmucl {
	chars              {}
	history            {}
	rxp,buf            {^(.*\n)(.*)$}
	rxp,strip_mudout   {\r|\xFF.\W?|\0}
	rxp,strip_ansi     {\x1B\[[0-9;]*m}
	rxp,parse_alias    {^(\S*)\s*(.*)$}
	types              {alias action sub key}
    }

    array set Mmucl $Mmucl(default_cfg)

    set Mmucl(save) [array names Mmucl cfg,*]
    lappend Mmucl(save) chars history

    set rc [file join $config(rc_dir) .save]
    if {[file exists $rc] && [catch {source $rc} err]} {
	puts stderr "error loading $rc: $error"
    }
    
    set Mmucl(hist_loc) [llength $Mmucl(history)]

    set Mmucl(has,MCCP) [expr {![catch {package require MCCP 0.06}]}]
    set Mmucl(has,Thread) [expr {![catch {package require Thread 2.5}]}]

    if {$Mmucl(has,Thread)} {
	thread::errorproc mmucl::thread_error
    }

    file mkdir [file join $config(rc_dir) chars]
    file mkdir [file join $config(rc_dir) autoload]

    set ses [ses_new]

    set i [lsearch -regexp $argv {^(-e|-exec|--exec|-eval|--eval)$}]

    if {$i != -1} {
	if {[catch {$ses eval [lindex $argv [incr i]]} err]} {
	    report $ses error "command line: $err"
	}
	set argv [lreplace $argv [expr {$i - 1}] $i]
    }

    rename init ""
}

proc mmucl::ses_new {} {
    variable Mmucl
    global config

    if {[info exists Mmucl(cur)]} {
	set old_ses $Mmucl(cur)
	set ses [expr {[lindex $Mmucl(sessions) end] + 1}]
    } else {
	set ses 0
    }

    lappend Mmucl(sessions) [set Mmucl(cur) $ses]

    array set Mmucl [list        \
	    ses,$ses,host     "" \
	    ses,$ses,port     "" \
	    ses,$ses,login    {} \
	    ses,$ses,connect  0  \
	    ses,$ses,snoop    0  \
	    ses,$ses,buf      "" \
            ses,$ses,flush_id "" \
	    ses,$ses,current_key    "" \
	    ses,$ses,action,changed 0  \
	    ses,$ses,sub,changed    0  \
	    ses,$ses,action,data  {}   \
	    ses,$ses,sub,data     {}   \
	    ses,$ses,group,names {default}   \
	    ses,$ses,group,exists,default "" \
    ]

    interp create $ses

    foreach proc {array set info source regexp} {
	$ses hide $proc
	$ses alias $proc $ses invokehidden $proc
    }

    foreach proc [info commands ::mmucl::MC*] {
	$ses alias [string range $proc 11 end] $proc $ses
    }

    if {$ses == 0} {
	interface_init
	
	report $ses msg "Mmucl $config(version)"

	if {$Mmucl(has,Thread)} {
	    report $ses msg "Thread extension loaded."
	}
	if {$Mmucl(has,MCCP)} {
	    report $ses msg "MCCP extension loaded."
	}

	print $ses ""
    } else {
	event $ses ses_new $old_ses
    }

    report $ses ses_new

    set dir [file join $config(rc_dir) autoload]
    foreach rc [glob -nocomplain -directory $dir *] {
	source_user $ses $rc
    }

    set rc [file join $config(rc_dir) mmucl.rc]
    if {[file exists $rc]} {
	source_user $ses $rc
    } else {
	set default [file join $config(lib_dir) script mmucl.rc]
	report $ses msg "$rc does not exist. Using default."
	source_user $ses $default
    }

    return $ses
}

proc mmucl::source_user {ses file} {
    variable Mmucl

    report $ses load $file

    if {[catch {$ses invokehidden source $file} error]} {
	report $ses error "$error\nStack trace:\n$::errorInfo"
    }
}

proc mmucl::MCsession {ses args} {
    variable Mmucl
    
    set syntax {
	new      {}
	names    {}
	close    {{+ id}}
	switch   {{+ id}}
	snoop    {{+ id} {? bool}}
	print    {{? id}}
	eval     {{+ id} {+ script}}
	current  {}
    }

    switch -exact -- [check session $syntax $args 1] {
	new {
	    return [ses_new]
	} names {
	    return $Mmucl(sessions)
	} close {
	    if {$arg(id) == $ses && [llength $Mmucl(sessions)] == 1} {
		error "can't close last session"
	    }

	    set i [lsearch -exact $Mmucl(sessions) $arg(id)]

	    if {$i == -1} {
		error "no such session: $arg(id)"
	    }

	    set Mmucl(sessions) [lreplace $Mmucl(sessions) $i $i]
	    array unset Mmucl ses,$arg(id),*
	    interp delete $arg(id)
	    set Mmucl(cur) [lindex $Mmucl(sessions) 0]
	    event $Mmucl(cur) ses_close $arg(id)
	} switch {
	    if {[lsearch -exact $Mmucl(sessions) $arg(id)] == -1} {
		error "no such session"
	    }
	    
	    if {$Mmucl(cur) != $arg(id)} {
		set old $Mmucl(cur)
		set Mmucl(cur) $arg(id)
		event $Mmucl(cur) ses_switch $old
	    }
	} snoop {
	    if {[lsearch -exact $Mmucl(sessions) $arg(id)] == -1} {
		error "no such session"
	    }

	    if {[info exists arg(bool)]} {
		set Mmucl(ses,$arg(id),snoop) \
			[string is true -strict $arg(bool)]
	    } else {
		return $Mmucl(ses,$arg(id),snoop)
	    }
	} eval {
	    if {[lsearch -exact $Mmucl(sessions) $arg(id)] == -1} {
		error "no such session"
	    }

	    $arg(id) eval $arg(script)
	} print {
	    if {[info exists arg(id)]} {
		if {[lsearch -exact $Mmucl(sessions) $arg(id)] == -1} {
		    error "no such session"
		}
		lappend names $id
	    } else {
		set names $Mmucl(sessions)
	    }

	    foreach id $names {
		set msg "$id: "
		append msg [expr {$Mmucl(ses,$id,snoop) ? \
			"snooping" : "not snooping"}] ", " \
			[expr {$Mmucl(ses,$id,connect) ? \
			$Mmucl(ses,$id,host) : "not connected"}]

		print $ses $msg
	    }
	} current {
	    return $Mmucl(cur)
	}
    }
    
    return
}

proc mmucl::MCdebug {ses args} {
    return [namespace eval ::mmucl:: [concat $args]]
}

proc mmucl::MCdebugsource {ses file} {
    uplevel \#0 source $file
}

# Groups aggregate and isolate aliases, actions, keys and subs.
#
# ses,$ses,group,exists,$name          - group $name exists
# ses,$ses,group,names                 - list of all groups
# ses,$ses,group,val,$name,$type,$key  - value of a given type in a group
# ses,$ses,group,keys,$name,$type      - list of keys of group $name
# ses,$ses,group,forkey,$type,$key     - list of groups which have a key
# ses,$ses,group,$which,$name,$type,$key - other data associated with a key

proc mmucl::group_member_set {ses name type key val args} {
    variable Mmucl

    ${type}_set_hook $ses $name $key $val $args

    set x ses,$ses,group

    if {![info exists Mmucl($x,val,$name,$type,$key)]} {
	lappend Mmucl($x,keys,$name,$type) $key
	lappend Mmucl($x,forkey,$type,$key) $name
    }

    set Mmucl($x,val,$name,$type,$key) $val

    if {![info exists Mmucl($x,exists,$name)]} {
	lappend Mmucl($x,names) $name
	set Mmucl($x,exists,$name) ""
    }
}

proc mmucl::group_member_get {ses name type key} {
    variable Mmucl

    set x ses,$ses,group

    if {![info exists Mmucl($x,exists,$name)]} {
	error "no such group: $name"
    }

    if {![info exists Mmucl($x,val,$name,$type,$key)]} {
	error "no such $type: $key"
    }

    return $Mmucl($x,val,$name,$type,$key)
}

# FIXME if last member deleted, delete group?

proc mmucl::group_member_delete {ses name type key} {
    variable Mmucl

    set x ses,$ses,group

    if {[info exists Mmucl($x,val,$name,$type,$key)]} {
	unset Mmucl($x,val,$name,$type,$key)
	lremove Mmucl($x,keys,$name,$type) $key
	lremove Mmucl($x,forkey,$type,$key) $name
	    
	if {[llength $Mmucl($x,forkey,$type,$key)] == 0} {
	    unset Mmucl($x,forkey,$type,$key)
	}

	${type}_delete_hook $ses $name $key
    }
}

proc mmucl::group_member_names {ses name type {glob *}} {
    variable Mmucl

    set x ses,$ses,group

    if {![info exists Mmucl($x,exists,$name)]} {
	error "no such group: $name"
    }

    if {[info exists Mmucl($x,keys,$name,$type)]} {
	return [lmatch $Mmucl($x,keys,$name,$type) $glob]	
    }

    return
}

# The default group "default" can never be deleted.

proc mmucl::group_delete {ses name} {
    variable Mmucl

    set x ses,$ses,group

    if {![info exists Mmucl($x,exists,$name)]} {
	return
    }

    foreach type $Mmucl(types) {
	if {[info exists Mmucl($x,keys,$name,$type)]} {
	    foreach key $Mmucl($x,keys,$name,$type) {
		unset Mmucl($x,val,$name,$type,$key)
		lremove Mmucl($x,forkey,$type,$key) $name

		if {[llength $Mmucl($x,forkey,$type,$key)] == 0} {
		    unset Mmucl($x,forkey,$type,$key)
		}

		${type}_delete_hook $ses $name $key
	    }
	    unset Mmucl($x,keys,$name,$type)
	}
    }
    
    if {$name ne "default"} {
	unset Mmucl($x,exists,$name)
	lremove Mmucl($x,names) $name
    }
}

proc mmucl::MCgroup {ses args} {
    variable Mmucl
    
    set syntax {
        delete {{- exact} {+ glob}}
        names {{? glob *}}
        print {{? glob *}}
    }
    
    set x ses,$ses,group

    switch -exact [check group $syntax $args 1] {
       delete {
	    if {[info exists arg(-exact)]} {
		group_delete $ses $arg(glob)
	    } else {
		foreach name [lmatch $Mmucl($x,names) $arg(glob)] {
		    group_delete $ses $name
		}
	    }
	} names {
	    return [lmatch $Mmucl($x,names) $arg(glob)]
        } print {
	    foreach name [lmatch $Mmucl($x,names) $arg(glob)] {
		print $ses $name
		foreach type $Mmucl(types) {
		    if {[info exists Mmucl($x,keys,$name,$type)]} {
			print $ses " $type"
			foreach key $Mmucl($x,keys,$name,$type) {
			    set val $Mmucl($x,val,$name,$type,$key)
			    print $ses "    $key"
			}
		    }
		}		
	    }
	}
    }

    return
}

# mostly a stub
proc mmucl::MCcline {ses args} {
    variable Mmucl

    set syntax {
	delete {{? first 0} {? last end}}
	get {{? first 0} {? last end}} 
	insert {{+ first} {+ str}}
	history {}
    }

    switch -exact [check cline $syntax $args 1] {
        delete {
	} get {
	} insert {
	} history {
	    return $Mmucl(history)
	}
    }
    
    return
}

# more stubs
proc mmucl::MChelp {ses args} {}
proc mmucl::MCbell {ses args} {}
proc mmucl::MCstatus {ses args} {}
proc mmucl::MCclear {ses args} {}
proc mmucl::save_config {} {}

# Called when there is mud output.Buffers an incomplete line and flushes
# the line if read_mud is not called again soon.

proc mmucl::read_mud {ses} {
    variable Mmucl

    append mudout $Mmucl(ses,$ses,buf) [read $Mmucl(ses,$ses,sock)]
    regsub -all $Mmucl(rxp,strip_mudout) $mudout "" mudout

    after cancel $Mmucl(ses,$ses,flush_id)
    
    if {[regexp $Mmucl(rxp,buf) $mudout x mudout Mmucl(ses,$ses,buf)]} {
	set Mmucl(flush_id) [after 500 mmucl::mudout_flush $ses]
    } else {
	set Mmucl(ses,$ses,buf) ""
    }

    handle_mudout $ses $mudout
    
    if {[eof $Mmucl(ses,$ses,sock)]}  {
	MCdisconnect $ses
    }
}

proc mmucl::mudout_flush {ses} {
    variable Mmucl

    handle_mudout $ses $Mmucl(ses,$ses,buf)
    set Mmucl(ses,$ses,buf) ""
}

# Check actions, do subs, and then display.

proc mmucl::handle_mudout {ses mudout} {
    variable Mmucl

    if {$Mmucl(cfg,actions)} {
	set mudout2 $mudout

	if {$Mmucl(ses,$ses,action,changed)} {
	    update_patterns $ses action
	}

	if {$Mmucl(cfg,strip_ansi)} {
	    regsub -all $Mmucl(rxp,strip_ansi) $mudout2 "" mudout2
	}
	
	if {$Mmucl(cfg,action_by_line)} {
	    foreach line [split $mudout2 \n] {
		match_actions $ses $line
	    }
	} else {
	    match_actions $ses $mudout2
	}
    }

    if {$Mmucl(cfg,subs)} {
	if {$Mmucl(ses,$ses,sub,changed)} {
	    update_patterns $ses sub
	}

	foreach {id rxp subspec} $Mmucl(ses,$ses,sub,data) {
	    regsub -all -- $rxp $mudout $subspec mudout
	}
    }

    meta_display $ses $mudout
}

proc mmucl::match_actions {ses str} {
    variable Mmucl

    foreach {id rxp script} $Mmucl(ses,$ses,action,data) {
	if {[$ses invokehidden regexp -- $rxp $str 0 1 2 3 4 5 6 7 8 9]} {
	    if {[catch {$ses eval $script} error]} {
		report $ses error "action $id: $error"
	    }
	}
    }
}

proc ::mmucl::meta_display {ses str} {
    variable Mmucl

    if {$Mmucl(ses,$ses,snoop) && $ses != $Mmucl(cur)} {
	foreach line [split $str \n] {
	    display $Mmucl(cur) [color "** $ses:" bold]$line\n
	}
    }

    display $ses $str
}

# inform the user about an error or other event

proc mmucl::report {ses event args} {
    variable Mmucl
    
    set color $Mmucl(cfg,report_color)

    switch -exact -- $event {
	error {
	    set color $Mmucl(cfg,error_color)
	    set msg [lindex $args 0]
	} timeout {
	    set msg "Connection timed out"
	} attempt {
	    lassign {n} $args
	    set msg "Connecting to $Mmucl(ses,$ses,host):$Mmucl(ses,$ses,port)\
		    \[attempt $n of $Mmucl(cfg,reconnect)\] ..."
	} connect {
	    lassign {ip host port} [fconfigure $Mmucl(ses,$ses,sock) -peername]
	    set msg "Connected to ${host}($ip):$port"
	} closed {
	    set msg "Connection closed"
	} stop_attempt {
	    set msg "Connection attempt stopped"
	} ses_new {
	    set msg "New session: $ses"
	} load {
	    lassign {file} $args
	    set msg "Loading $file..."
	} msg {
	    set msg [lindex $args 0]
	} default {
	    error "no handler for $event"
	}
    }
    
    foreach part [split $msg \n] {
	meta_display $ses [color "** " bold $part $color \n]
    }
}

# Display a line to the user.

proc mmucl::print {ses str} {
    variable Mmucl

    meta_display $ses [color $str $Mmucl(cfg,print_color) \n]
}

# check args to a command that may have options (subcommands)

proc mmucl::check {name syntax arglist {opts 0}} {
    if {$opts} {
	if {[llength $arglist] == 0} {
	    error "wrong # args: should be \"$name option arg...\""
	}
	
	set opt [lindex $arglist 0]
	array set opt_syntax $syntax

	if {![info exists opt_syntax($opt)]} {
	    error "bad option \"$opt\": must be\
		    [join [lsort [array names opt_syntax]] {, }]"
	}

	set arglist [lrange $arglist 1 end]
	set syntax $opt_syntax($opt)
	append name " " $opt
    } else {
	set opt ""
    }

    set i 0
    set len [llength $arglist]

    foreach part $syntax {
    	switch -exact -- [lindex $part 0] {
            ? {
		if {$i < $len} {
		    set var([lindex $part 1]) [lindex $arglist $i]
		    incr i
		} elseif {[llength $part] == 3} {
	       	    set var([lindex $part 1]) [lindex $part 2]  
		}
	    } - {
		unset -nocomplain -- need_arg
		
		foreach def [lrange $part 1 end] {
		    set switch [lindex $def 0]

		    if {[llength $def] == 1} {
			set var($switch) 0
		    } else {
			set var($switch) [lindex $def 1]
			set need_arg($switch) ""
		    }
		}

		while {$i < $len} {
		    set switch [lindex $arglist $i]

		    if {![string equal -length 1 $switch -]} {
			break
		    } elseif {$switch eq "--"} {
			incr i
			break
		    }
		    
		    set switch [string range $switch 1 end]

		    if {![info exists var($switch)]} {
			set msg "bad switch \"-$switch\": must be "

			foreach def [lrange $part 1 end] {
			    append msg - [lindex $def 0] ", "
			}

			error [append msg "or --"]
		    }
		    
		    if {[info exists need_arg($switch)]} {
			if {$i == $len} {
			    error "-$switch requires an argument"
			}

			set var($switch) [lindex $arglist [incr i]]
		    } else {
			set var($switch) 1
		    }

		    incr i
		}
	    } + {
		set var([lindex $part 1]) [lindex $arglist $i]
		incr i
	    } default {
		error "bad spec: [lindex $part 0]"
	    }
	}
    }
    
    if {$i != [llength $arglist]} {
        error [check_usage $name $syntax]
    }

    uplevel 1 [list array set arg [array get var]]
    return $opt
}

proc mmucl::check_usage {name syntax} {
    set usage "wrong # args: should be \"$name"

    foreach part $syntax {
    	switch -exact -- [lindex $part 0] {
            ? {
		append usage " " \[[lindex $part 1]\]
	    } - {
		append usage " \["
		set first 1

		foreach def [set switches [lrange $part 1 end]] {
		    if {$first} {
			set first 0
		    } else {
			append usage " "
		    }

		    if {[llength $def] == 1} {
			append usage - [lindex $def 0]
		    } else {
			append usage - [lindex $def 0] " val"
		    }
		}

		append usage " --\]"
	    } + {
		append usage " " [lindex $part 1]
	    } default {
		error "bad spec: [lindex $part 0]"
	    }
	}
    }
    
    return [append usage \"]    
}

proc mmucl::MCcheck {ses args} {
    check check {{- opts} {+ name} {+ syntax} {+ arglist}} $args

    set name $arg(name)
    set syntax $arg(syntax)
    set arglist $arg(arglist)
    set opts $arg(opts) 
    unset arg
    
    set opt [check $name $syntax $arglist $opts]
    $ses invokehidden array set arg [array get arg]
    
    return $opt
}

proc mmucl::MCecho {ses args} {
    meta_display $ses [join $args]\n
    return
}

proc mmucl::color {args} {
    variable Ansi

    if {[llength $args] == 1} {
	set args [lindex $args 0]
    }

    foreach {str attribs} $args {
	append new \x1B\[0\;
	
	foreach attrib $attribs {
	    if {[info exists Ansi($attrib)]} {
		append new  \; $Ansi($attrib)
	    }
	}
	
	append new m $str
    }

    return [append new \x1B\[0m]
}

proc mmucl::MCcolor {ses args} {
    return [color $args]
}

# Evaluate a line of input from the user.

proc mmucl::MCparse {ses str} {
    variable Mmucl

    if {[string equal -length 1 $str $Mmucl(cfg,verbatim_char)]} {
	MCwrite $ses [string range $str 1 end]
    } elseif {[string equal -length 1 $str $Mmucl(cfg,script_char)]} {
	return [$ses eval [string range $str 1 end]]
    } elseif {$str eq ""} {
	MCwrite $ses ""
    } else {
	set x ses,$ses,group

	foreach cmd [split $str $Mmucl(cfg,split_char)] {
	    regexp $Mmucl(rxp,parse_alias) $cmd "" alias arg

	    if {[info exists Mmucl($x,forkey,alias,$alias)]} {
                $ses invokehidden \
		    regexp {^(\S*)\s*(\S*)\s*(\S*).*$} $arg 0 1 2 3
		set err_msg ""

		foreach group $Mmucl($x,forkey,alias,$alias) {
		    set script $Mmucl($x,val,$group,alias,$alias)

		    if {[catch {$ses eval $script} error]} {
			if {$err_msg ne ""} {
			    append err_msg  \n
			}

			append err_msg "alias " $alias ": " $error
		    }
		}

		if {$err_msg ne ""} {
		    error $err_msg
		}
	    } else {
		MCwrite $ses $cmd
	    }
	}
    }
    
    return
}

proc mmucl::meta_parse {str} {
    variable Mmucl

    set ses $Mmucl(cur)

    if {$Mmucl(cfg,echo)} {
	meta_display $ses [color $str $Mmucl(cfg,echo_color) \n]
    }
    
    if {[catch {MCparse $ses $str} val]} {
	report $ses error $val
    } elseif {$val ne ""} {
	print $ses $val
    }

    if {([string length $str] >= $Mmucl(cfg,hist_min)) && \
	    ($str ne [lindex $Mmucl(history) end])} {

	set Mmucl(history) [lreplace $Mmucl(history) 0 \
		[expr {[llength $Mmucl(history)] - $Mmucl(cfg,hist_keep)}]]
	lappend Mmucl(history) $str
	set Mmucl(hist_loc) [llength $Mmucl(history)]
    }

    return
}

proc mmucl::MCmmucl {ses args} {
    variable Mmucl
    global config

    set syntax {
	lib_dir {}
	rc_dir  {}
	host    {}
	port    {}
	version {}
	connect {}
	interface {}
    }
    
    switch -exact -- [check mmucl $syntax $args 1] {
	lib_dir {
	    return $config(lib_dir)
	} rc_dir {
	    return $config(rc_dir)
	} host {
	    return $Mmucl(ses,$ses,host)
	} port {
	    return $Mmucl(ses,$ses,port)
	} version {
	    return $config(version)
	} connect {
	    return $Mmucl(ses,$ses,connect)
	} interface {
	    return $config(interface)
	}
    }

    return
}

proc mmucl::connect_callback {ses event args} {
    variable Mmucl

    switch -exact $event {
	success {
	    lassign {chan} $args

	    fconfigure $chan -blocking 0 -translation {binary auto}
	    fileevent $chan readable [list mmucl::read_mud $ses]

	    array set Mmucl [list             \
		    ses,$ses,connect 1        \
		    ses,$ses,sock    $chan    \
		    ]
	    
	    unset Mmucl(ses,$ses,connect_tok)

	    if {$Mmucl(has,MCCP) && $Mmucl(cfg,use_mccp)} {
	    	MCCP::init $chan
	    }

	    if {[llength $Mmucl(ses,$ses,login)]} {
		MCwrite $ses [join $Mmucl(ses,$ses,login) \n]
	    }

	    report $ses connect
	    event $ses connect
	} failure {
	    lassign {msg} $args

	    unset Mmucl(ses,$ses,connect_tok)
	    event $ses disconnect
	    report $ses error "connection failed: $msg"
	} timeout {
	    unset Mmucl(ses,$ses,connect_tok)
	    event $ses timeout
	    report $ses reconnect
	} attempt {
	    lassign {tok n} $args

	    set Mmucl(ses,$ses,connect_tok) $tok
	    report $ses attempt $n 
	    event $ses attempt
	} default {
	    error "event $event not understood"
	}
    }
}

proc mmucl::MCconnect {ses args} {
    variable Mmucl

    check connect {{+ host} {+ port} {? login {}}} $args

    if {[catch {llength $arg(login)}]} {
	error "login must be a list"
    }

    if {$Mmucl(ses,$ses,connect)} {
	error "already connected"
    }

    if {[info exists Mmucl(ses,$ses,connect_tok)]} {
	error "connection attempt in progress"
    }

    if {![string is integer -strict $arg(port)]} {
	error "bad port $arg(port), must be integer"
    }

    array set Mmucl [list              \
	    ses,$ses,host  $arg(host)  \
	    ses,$ses,port  $arg(port)  \
	    ses,$ses,login $arg(login) \
	    ]
	    
    connect $arg(host) $arg(port) mmucl::connect_callback $ses \
	    -max_attempts [expr {$Mmucl(cfg,reconnect) + 1}]   \
	    -timeout $Mmucl(cfg,timeout) -thread $Mmucl(cfg,use_threads)
    return
}

proc mmucl::MCdisconnect {ses} {
    variable Mmucl

    if {!$Mmucl(ses,$ses,connect)} {
	if {[info exists Mmucl(ses,$ses,connect_tok)]} {
	    connect_cancel $Mmucl(ses,$ses,connect_tok)
	    report $ses stop_attempt
	    event $ses disconnect
	    unset Mmucl(ses,$ses,connect_tok)
	    return
	} 
	error "not connected"
    }
    
    catch {
	close $Mmucl(ses,$ses,sock)
    }

    set Mmucl(ses,$ses,connect) 0
    report $ses closed
    event $ses disconnect $Mmucl(ses,$ses,host)

    return
}

proc mmucl::MCreconnect {ses } {
    variable Mmucl
    
    if {$Mmucl(ses,$ses,host) eq ""} {
	error "no previous connection"
    }
    
    MCconnect $ses $Mmucl(ses,$ses,host) $Mmucl(ses,$ses,port) \
	    $Mmucl(ses,$ses,login)
    
    return
}

proc mmucl::MCconfig {ses args} {
    variable Mmucl
    
    set syntax {
	set    {{+ option} {? value}}
	names  {{? pattern *}}
	print  {{? pattern *}}
	state  {}
	reset {}
    }

    switch -exact -- [check config $syntax $args 1] {
	set {
	    if {![info exists Mmucl(cfg,$arg(option))]} {
		error "no such option"
	    } elseif {![info exists arg(value)]} {
		return $Mmucl(cfg,$arg(option))
	    } else {
		if {[string match *_char $arg(option)]} {
		    if {[string length $arg(value)] != 1} {
			error "value must be a char"
		    }
		} elseif {[string match *_color $arg(option)]} {
		    if {[catch {concat $arg(value)}]} {
			error "value must be a list"
		    }
		} elseif {![string is integer -strict $arg(value)]} {
		    error "value must be a number"
		}

		set Mmucl(cfg,$arg(option)) $arg(value)
	    }
	} names {
	    return [ltrimleft [array names Mmucl cfg,$arg(pattern)] 4]
	} print {
	    foreach name [lsort [MCconfig $ses names $arg(pattern)]] {
		print $ses "$name: $Mmucl(cfg,$name)"
	    }
	} state {
	    set res ""

	    foreach name [ltrimleft [array names Mmucl cfg,*] 4] {
		append res [list config set $name $Mmucl(cfg,$name)] \n
	    }

	    return $res
	} reset {
	    array set Mmucl $Mmucl(default_cfg)
	}
    }
    
    return
}

proc mmucl::MCwrite {ses args} {
    variable Mmucl
    
    if {!$Mmucl(ses,$ses,connect)} {
	error "not connected"
    }

    foreach str $args {
	puts $Mmucl(ses,$ses,sock) $str
    }
    flush $Mmucl(ses,$ses,sock)
    
    return
}

proc mmucl::alias_set_hook {ses group name val args} {
    if {[regexp -- {\s} $name]} {
	error "bad alias name \"$name\": cannot contain white space"
    }
}

proc mmucl::alias_delete_hook {ses group name} {}

proc mmucl::MCalias {ses args} {
    variable Mmucl

    set syntax {
	set    {{- {group default}} {+ name} {? script}}
	names  {{- {group default}} {? glob *}}
	print  {{- {group default}} {? glob *}}
	delete {{- exact {group default}} {+ glob}}
	state  {{- {group default}}}
    }

    switch -exact -- [check alias $syntax $args 1] {
	set {
	    if {[info exists arg(script)]} {
		group_member_set $ses $arg(group) alias $arg(name) $arg(script)
	    } else {
		return [group_member_get $ses $arg(group) alias $arg(name)]
	    }
	} names {
	    return [group_member_names $ses $arg(group) alias $arg(glob)]
	} delete {
	    if {$arg(exact)} {
		group_member_delete $ses $arg(group) alias $arg(glob)
	    } else {
		foreach name [group_member_names $ses $arg(group) alias \
			$arg(glob)] {
		    group_member_delete $ses $arg(group) alias $name
		}
	    }
	} print {
	    set names [group_member_names $ses $arg(group) alias $arg(glob)]

	    foreach name [lsort $names] {
		set val [group_member_get $ses $arg(group) alias $name]
		print $ses "\"$name\" set to {$val}"
	    }
	} state {
	    set names [group_member_names $ses $arg(group) alias]
	    set res ""

	    foreach name [lsort $names] {
		set val [group_member_get $ses $arg(group) alias $name]
		append res [list alias set -group $arg(group) -- $name $val] \n
	    }

	    return $res
	}
    }
    
    return
}

proc mmucl::key_validate {key} {
    if {![regexp {^(\w+-)*\w+$} $key]} {
	error "bad key: $key, should be keysym or mod-keysym"
    }
}

proc mmucl::key_set_hook {ses group name val arglist} {
    key_validate $name
}

proc mmucl::key_delete_hook {ses group name} {}

proc mmucl::MCkey {ses args} {
    variable Mmucl

    set syntax {
	set     {{- {group default}} {+ name} {? script}}
	names   {{- {group default}} {? glob *}}
	print   {{- {group default}} {? glob *}}
	delete  {{- exact {group default}} {+ glob}}
	current {}
	state   {{- {group default}}}
    }

    switch -exact -- [check key $syntax $args 1] {
	set {
	    if {[info exists arg(script)]} {
		group_member_set $ses $arg(group) key $arg(name) $arg(script)
	    } else {
		return [group_member_get $ses $arg(group) key $arg(name)]
	    }
	} names {
	    return [group_member_names $ses $arg(group) key $arg(glob)]
	} delete {
	    if {$arg(exact)} {
		group_member_delete $ses $arg(group) key $arg(glob)
	    } else {
		foreach name [group_member_names $ses $arg(group) key \
			$arg(glob)] {
		    group_member_delete $ses $arg(group) key $name
		}
	    }
	} print {
	    set names [group_member_names $ses $arg(group) key $arg(glob)]
	    foreach name [lsort $names] {
		set val [group_member_get $ses $arg(group) key $name]

		print $ses "\"$name\" bound to {$val}"
	    }
	} current {
	    return $Mmucl(ses,$ses,current_key)
	}  state {
	    set names [group_member_names $ses $arg(group) key]
	    set res ""

	    foreach name [lsort $names] {
		set val [group_member_get $ses $arg(group) key $name]
		append res [list key set -group $arg(group) -- $name $val] \n
	    }

	    return $res
	}
    }
    
    return
}

# convert format pattern to a regexp

proc mmucl::format2regexp {fmt} {
    set charmap {
	%% %
	** *
	^^ ^
	$$ $
	|| |
	\\  \\\\
	[  \\[
	]  \\]
	(  \\(
	)  \\)
	\{ \\\}
	\} \\\{
	+  \\+
	.  \\.
	?  \\?
	%a \x1B\\[([0-9;]*)m
	%w (\\S+)
	%s (.+)
	%c (.)
	%d (\\d+)
	* (?:.|\n)*
    }

    return [append rxp (?n) [string map $charmap $fmt]]
}

# convert exact pattern to a regexp

proc mmucl::exact2regexp {exact} {
    set charmap {
	\\  \\\\
	[  \\[
	]  \\]
	(  \\(
	)  \\)
	\{ \\\}
	\} \\\{
	+  \\+
	.  \\.
	?  \\?
	*  \\*
    }

    return [append rxp (?n) [string map $charmap $exact]]
}

proc mmucl::action_or_sub_set_hook {ses group name val arglist cmd} {
    variable Mmucl

    set type [lindex $arglist 1]

    if {$type eq "regexp"} {
	set rxp $name
    } else {
	set rxp [${type}2regexp $name]
    }

    regexp -- $rxp ""

    set pri [lindex $arglist 0]

    if {![string is integer $pri]} {
	error "bad priority: \"$pri\" must be an integer"
    }

    set x ses,$ses,group

    set Mmucl($x,priority,$group,$cmd,$name) $pri
    set Mmucl($x,type,$group,$cmd,$name) $type
    set Mmucl($x,rxp,$group,$cmd,$name) $rxp
    set Mmucl(ses,$ses,$cmd,changed) 1
}

proc mmucl::action_set_hook {ses group name val arglist} {
    action_or_sub_set_hook $ses $group $name $val $arglist action
}

proc mmucl::sub_set_hook {ses group name val arglist} {
    action_or_sub_set_hook $ses $group $name $val $arglist sub
}

proc mmucl::action_or_sub_delete_hook {ses group name cmd} {
    variable Mmucl

    set x ses,$ses,group

    unset Mmucl($x,priority,$group,$cmd,$name) \
	    Mmucl($x,rxp,$group,$cmd,$name)   \
	    Mmucl($x,type,$group,$cmd,$name)

    set Mmucl(ses,$ses,$cmd,changed) 1
}

proc mmucl::action_delete_hook {ses group name} {
    action_or_sub_delete_hook $ses $group $name action
}

proc mmucl::sub_delete_hook {ses group name} {
    action_or_sub_delete_hook $ses $group $name sub
}

proc mmucl::update_patterns {ses cmd} {
    variable Mmucl

    set Mmucl(ses,$ses,$cmd,data) [list]
    set x ses,$ses,group
    set info [list]

    foreach group $Mmucl(ses,$ses,group,names) {
	foreach name [group_member_names $ses $group $cmd] {
	    if {$group eq "default"} {
		set id $name
	    } else {
		set id "$name \[$group\]"
	    }

	    set rxp $Mmucl($x,rxp,$group,$cmd,$name)
	    set data $Mmucl($x,val,$group,$cmd,$name)
	    set pri $Mmucl($x,priority,$group,$cmd,$name)

	    lappend info [list $id $rxp $data $pri]
	}
    }

    set var Mmucl(ses,$ses,$cmd,data)
    foreach el [lsort -decreasing -integer -index 3 $info] {
	lassign {id rxp data pri} $el
	lappend $var $id $rxp $data
    }

    set Mmucl(ses,$ses,$cmd,changed) 0
}

proc mmucl::MCaction {ses args} {
    variable Mmucl

    set syntax {
	set    {{- {group default} regexp exact format {priority 0}} \
		{+ pattern} {? script}}
	names  {{- {group default}} {? glob *}}
	print  {{- {group default}} {? glob *}}
	delete {{- exact {group default}} {+ glob}}
	priority {{- {group default}} {+ pattern} {? priority}}
	type   {{- {group default}} {+ pattern}}
	match  {{- {group default}} {+ str} {? glob *}}
	state  {{- {group default}}}
    }

    switch -exact -- [check action $syntax $args 1] {
	set {
	    if {[info exists arg(script)]} {
		if {$arg(regexp)} {
		    set type regexp
		} elseif {$arg(exact)} {
		    set type exact
		} else {
		    set type format
		}

		group_member_set $ses $arg(group) action $arg(pattern) \
			$arg(script) $arg(priority) $type
	    } else {
		return [group_member_get $ses $arg(group) action $arg(pattern)]
	    }
	} names {
	    return [group_member_names $ses $arg(group) action $arg(glob)]
	} delete {
	    if {$arg(exact)} {
		group_member_delete $ses $arg(group) action $arg(glob)
	    } else {
		foreach name [group_member_names $ses $arg(group) action \
			$arg(glob)] {
		    group_member_delete $ses $arg(group) action $name
		}
	    }
	} print {
	    set names [group_member_names $ses $arg(group) action $arg(glob)]
	    foreach name [lsort $names] {
		set val [group_member_get $ses $arg(group) action $name]
		set p $Mmucl(ses,$ses,group,priority,$arg(group),action,$name)
		set t $Mmucl(ses,$ses,group,type,$arg(group),action,$name)

		print $ses "($p) {$name} \[$t\] triggers {$val}"
	    }
	} priority {
	    group_member_get $ses $arg(group) action $arg(pattern)
	    set x ses,$ses,group,priority,$arg(group),action,$arg(pattern)

	    if {[info exists arg(priority)]} {
		if {![string is integer $arg(priority)]} {
		    error "bad priority: \"$arg(priority)\" must be an integer"
		}

		set Mmucl($x) $arg(priority)
		set Mmucl(ses,$ses,action,changed) 1
	    } else {
		return $Mmucl($x)
	    }
	} type {
	    group_member_get $ses $arg(group) action $arg(pattern)
	    set x ses,$ses,group,type,$arg(group),action,$arg(pattern)
	    return $Mmucl($x)
	} match {
	    set res [list]

	    set names [group_member_names $ses $arg(group) action $arg(glob)]
	    set x ses,$ses,group,rxp,$arg(group),action

	    foreach name $names {
		if {[regexp -- $Mmucl($x,$name) $arg(str)]} {
		    lappend res $name
		}
	    }

	    return $res
	}  state {
	    set names [group_member_names $ses $arg(group) action]
	    set res ""
	    
	    foreach name [lsort $names] {
		set val [group_member_get $ses $arg(group) action $name]
		set p $Mmucl(ses,$ses,group,priority,$arg(group),action,$name)
		set t $Mmucl(ses,$ses,group,type,$arg(group),action,$name)

		append res [list action set -group $arg(group) \
				-$t -priority $p -- $name $val] \n
	    }

	    return $res
	}
    }

    return
}

proc mmucl::MCsub {ses args} {
    variable Mmucl

    set syntax {
	set    {{- {group default} regexp exact format {priority 0}} \
		{+ pattern} {? subspec}}
	names  {{- {group default}} {? glob *}}
	print  {{- {group default}} {? glob *}}
	delete {{- exact {group default}} {+ glob}}
	priority {{- {group default}} {+ pattern} {? priority}}
	type   {{- {group default}} {+ pattern}}
	state  {{- {group default}}}
	match   {{- {group default}} {+ str} {? glob *}}
    }

    switch -exact -- [check sub $syntax $args 1] {
	set {
	    if {[info exists arg(subspec)]} {
		if {$arg(regexp)} {
		    set type regexp
		} elseif {$arg(exact)} {
		    set type exact
		} else {
		    set type format
		}

		group_member_set $ses $arg(group) sub $arg(pattern) \
			$arg(subspec) $arg(priority) $type
	    } else {
		return [group_member_get $ses $arg(group) sub $arg(pattern)]
	    }
	} names {
	    return [group_member_names $ses $arg(group) sub $arg(glob)]
	} delete {
	    if {$arg(exact)} {
		group_member_delete $ses $arg(group) sub $arg(glob)
	    } else {
		foreach name [group_member_names $ses $arg(group) sub \
			$arg(glob)] {
		    group_member_delete $ses $arg(group) sub $name
		}
	    }
	} print {
	    set names [group_member_names $ses $arg(group) sub $arg(glob)]
	    foreach name [lsort $names] {
		set val [group_member_get $ses $arg(group) sub $name]
		set p $Mmucl(ses,$ses,group,priority,$arg(group),sub,$name)
		set t $Mmucl(ses,$ses,group,type,$arg(group),sub,$name)
		print $ses "($p) {$name} \[$t\] changed to {$val}"
	    }
	} priority {
	    group_member_get $ses $arg(group) sub $arg(pattern)
	    set x ses,$ses,group,priority,$arg(group),sub,$arg(pattern)

	    if {[info exists arg(priority)]} {
		if {![string is integer $arg(priority)]} {
		    error "bad priority: \"$arg(priority)\" must be an integer"
		}

		set Mmucl($x) $arg(priority)
		set Mmucl(ses,$ses,subs,changed) 1
	    } else {
		return $Mmucl($x)
	    }
	} type {
	    group_member_get $ses $arg(group) sub $arg(pattern)
	    set x ses,$ses,group,type,$arg(group),sub,$arg(pattern)
	    return $Mmucl($x)
	} match {
	    set res [list]

	    set names [group_member_names $ses $arg(group) sub $arg(glob)]
	    set x ses,$ses,group,rxp,$arg(group),sub

	    foreach name $names {
		if {[regsub -all -- $Mmucl($x,$name) $arg(str) x str]} {
		    lappend res $name $str
		}
	    }

	    return $res
	} state {
	    set names [group_member_names $ses $arg(group) sub]
	    set res ""
	    
	    foreach name [lsort $names] {
		set val [group_member_get $ses $arg(group) sub $name]
		set p $Mmucl(ses,$ses,group,priority,$arg(group),sub,$name)
		set t $Mmucl(ses,$ses,group,type,$arg(group),sub,$name)

		append res [list sub set -group $arg(group) \
				-$t -priority $p -- $name $val] \n
	    }

	    return $res
	}
    }

    return
}

proc mmucl::MCchar {ses args} {
    variable Mmucl
    global config

    set syntax {
	set    {{+ name} {? info}}
	names  {{? pattern *}}
	print  {{? pattern *}}
	delete {{- exact} {+ pattern}}
	load   {{+ name}}
	state  {}
    }
    
    array set char $Mmucl(chars)
    
    switch -exact -- [check char $syntax $args 1] {
	set {
	    if {[info exists arg(info)]} {
		set n [llength $arg(info)]

		if {$n < 2 || $n > 3} {
		    error "info is list of form: {host port ?login?}"
		}
		
		catch {close [open \
			[file join $config(rc_dir) chars $arg(name)] a+]}

		set char($arg(name)) $arg(info)
	    } elseif {![info exists char($arg(name))]} {
		error "no such char"
	    } else {
		return $char($arg(name))
	    }
	} names {	
	    return [array names char $arg(pattern)]
	} delete {
	    if {$arg(exact)} {
		unset -nocomplain -- char($arg(pattern))
	    } else {
		array unset char $arg(pattern)
	    }
	} load {
	    if {![info exists char($arg(name))]} {
		error "no such char"
	    }
	    
	    lassign {host port login} $char($arg(name))
	    source_user $ses [file join $config(rc_dir) chars $arg(name)]
	    MCconnect $ses $host $port $login
	} print {
	    foreach {name info} [array get char $arg(pattern)] {
		lassign {host port login} $info

		print $ses $name
		print $ses "  Host:  $host:$port"
		print $ses "  Login: $login"
	    }
	} state {
	    set res ""

	    foreach {name val} [array get char] {
		append res [list char set $name $val] \n
	    }

	    return $res
	}
    }
    
    set Mmucl(chars) [array get char]
    return
}

proc mmucl::MCdump {ses args} {
    variable Mmucl
    
    set group_cmds {action alias sub key}
    set simple_cmds {config char}

    eval lappend switches $group_cmds $simple_cmds
    lappend switches var proc

    check dump "{- append {group default} $switches} {+ file}" $args
    
    set fd [open $arg(file) [expr {$arg(append) ? "a+" : "w"}]]
    
    set all 1
    foreach switch $switches {
	if {$arg($switch)} {
	    set all 0
	    break
	}
    }

    foreach cmd $group_cmds {
	if {$all || $arg($cmd)} {
	    puts $fd [MC$cmd $ses state -group $arg(group)]
	}
    }

    foreach cmd $simple_cmds {
	if {$all || $arg($cmd)} {
	    puts $fd [MC$cmd $ses state]
	}
    }

    if {$all || $arg(proc)} {
	foreach func [$ses invokehidden info procs] {
	    set func_args [list]

	    foreach func_arg [$ses invokehidden info args $func] {
		if {[$ses invokehidden info default $func $func_arg __def]} {
		    set def [$ses invokehidden set __def]
		    lappend func_args [list $func_arg $def]
		} else {
		    lappend func_args $func_arg
		}
	    }
	    
	    set body [$ses invokehidden info body $func]
	    puts $fd [list proc $func $func_args $body]
	}
    }


    if {$all || $arg(var)} {
	foreach var [$ses invokehidden info globals] {
	    if {[$ses invokehidden array exists $var]} {
		set data [$ses invokehidden array get $var]
		puts $fd [list array set $var $data]
	    } else {
		puts $fd [list set $var [$ses invokehidden set $var]]
	    }
	}
    }


    close $fd
    return
}

proc mmucl::MCtextin {ses args} {
    variable Mmucl

    check textin {{+ file}} $args

    if {!$Mmucl(ses,$ses,connect)} {
	error "not connected"
    }
    
    set fd [open $arg(file) r]
    fcopy $fd $Mmucl(ses,$ses,sock)
    close $fd
   
    return
}

proc bgerror {msg} {
    global errorInfo

    puts stderr "This is a BUG. Please report."
    puts stderr "background error: $msg"
    puts stderr $errorInfo
}

# Provide a reasonable default bgerror to user.

proc mmucl::MCbgerror {ses msg} {
    global errorInfo

    report $ses error "Background error in user script: $msg\nStack trace:\n$::errorInfo"
}

proc mmucl::thread_error {id msg} {
    puts stderr "BUG: Error in thread $id"
    puts stderr $msg
}

proc mmucl::MCexit {ses} {
    variable Mmucl
    global config

    mmucl::save_config

    set error [catch {
	set fd [open [file join $config(rc_dir) .pref.$config(interface)] w] 
	puts $fd [list array set Mmucl [array get Mmucl pref,*]]
	close $fd

	set fd [open [file join $config(rc_dir) .save] w]
	foreach name $Mmucl(save) {
	    lappend save $name $Mmucl($name)
	}
	puts $fd [list array set Mmucl $save]
	close $fd
    } msg]
	
    if {$error} {
	puts stderr "error saving state: $msg"
    }

    exit
}
