#ifndef MORTARS_H
#define MORTARS_H

#include <list>

#include "SinusTable.h"
#include "Tools.h"

#include "XMLVisitors.h"

#include "StaticDecorationBase.h"
#include "Grenade.h"


//----------------------------------------------------------------------------
class MortarBase;


//----------------------------------------------------------------------------
#undef DECLARE_UPDATE_STRATEGY
#define DECLARE_UPDATE_STRATEGY(c) \
    class c; \
    friend class c; \
    class c : public UpdateStrategy \
    { \
        STRATEGY_OBJECT(c); \
      public: \
        c() {} \
        ~c() {} \
        void update(MortarBase *m, MortarBarrel *b); \
    }


//----------------------------------------------------------------------------
class MortarBarrel
{
    //------------------------------------------------------------------------
    friend class MortarTest;

    //------------------------------------------------------------------------
    enum EnumUpdateStrategy
    {
        US_FIXED,
        US_RANDOM,
        US_SMART
    };

    //------------------------------------------------------------------------
    class UpdateStrategy
    {
      public:
        UpdateStrategy() {}
        virtual ~UpdateStrategy() {}

        static UpdateStrategy *get(EnumUpdateStrategy strategy);

        virtual void update(MortarBase *t, MortarBarrel *b) = 0;
    };

    DECLARE_UPDATE_STRATEGY(Fixed);
    DECLARE_UPDATE_STRATEGY(Random);
    DECLARE_UPDATE_STRATEGY(Smart);

  public:

    //------------------------------------------------------------------------
    class InitializationData
    {
      public:
        InitializationData(const XMLNode *barrelNode);
        ~InitializationData();

        EnumUpdateStrategy updateStrategy;
        int angle;
        unsigned speed;
        unsigned delay;
        unsigned exploderDelay;
        Grenade::EnumWarheadStrategy grenadeWarheadStrategy;
    };

    ~MortarBarrel();

    //------------------------------------------------------------------------
    inline int getAngle() const
    {
        return m_angle;
    }
    inline void setAngle(int angle)
    {
        m_angle = SinusTable::normalize(angle);
    }

    //------------------------------------------------------------------------
    inline unsigned getGrenadeSpeed() const
    {
        return m_grenadeSpeed;
    }

    inline unsigned getGrenadeExploderDelay() const
    {
        return m_grenadeExploderDelay;
    }

    inline Grenade::EnumWarheadStrategy getGrenadeWarheadStrategy() const
    {
        return m_grenadeWarheadStrategy;
    }

    //------------------------------------------------------------------------
    void update(MortarBase *mortar);

    //------------------------------------------------------------------------
    inline void setStrategy(UpdateStrategy *strategy)
    {
        m_strategy = strategy;
    }

    //------------------------------------------------------------------------
    static MortarBarrel *create(const XMLNode *barrelNode);

    //------------------------------------------------------------------------
    Grenade *createGrenade(MortarBase *mortar) const;

  protected:

    //------------------------------------------------------------------------
    MortarBarrel(const InitializationData &init);

    //------------------------------------------------------------------------
    int m_angle;
    int m_angleStep;

    unsigned m_grenadeSpeed;
    unsigned m_grenadeExploderDelay;
    Grenade::EnumWarheadStrategy m_grenadeWarheadStrategy;

    unsigned m_frameDelay;
    unsigned m_frameCounter;

  private:

    //------------------------------------------------------------------------
    UpdateStrategy *m_strategy;
};

typedef std::list<MortarBarrel*> MortarBarrelList;
typedef MortarBarrelList::iterator MortarBarrelListIter;
typedef MortarBarrelList::const_iterator MortarBarrelListCIter;



//----------------------------------------------------------------------------
class MortarBase : public StaticDecorationBase,
                   public OrientatingDecorationBase
{
    //------------------------------------------------------------------------
    friend class MortarTest;

    //------------------------------------------------------------------------
    class BarrelVisitor;
    friend class BarrelVisitor;
    class BarrelVisitor : public XMLConstVisitor
    {
      public:
        //--------------------------------------------------------------------
        BarrelVisitor(MortarBase *mortar) : m_mortar(mortar) {}
        ~BarrelVisitor() { m_mortar = NULL; }

      private:
        //--------------------------------------------------------------------
        void do_visit(const XMLProperty *a);
        void do_visit(const XMLNode *n);

        //--------------------------------------------------------------------
        MortarBase *m_mortar;
    };

  public:
    //------------------------------------------------------------------------
    class InitializationData
        : public StaticDecorationBase::InitializationData,
          public OrientatingDecorationBase::InitializationData
    {
      public:
        InitializationData();
        InitializationData(const XMLNode *mortarNode);
        ~InitializationData();

        unsigned hitPoints;
    };


    //------------------------------------------------------------------------
    virtual ~MortarBase();


    /// Returns, whether the mortar can fire to the given angle.
    bool isAngleInRange(int angle) const;
    virtual int getCenterAngle() const = 0;

    virtual OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const = 0;

    //------------------------------------------------------------------------
    /**
     * Called for a projectile collision to determine,
     * if the mortar's hitpoints are reached.
     */
    inline bool decAndTestHitPoints()
    {
        return --m_hitPoints == 0;
    }

    //------------------------------------------------------------------------
    void update();

    //------------------------------------------------------------------------
    void updateSurface();

    //------------------------------------------------------------------------
    static MortarBase *create(const InitializationData &init);
    static MortarBase *create(const XMLNode *mortarNode);

  protected:
    //------------------------------------------------------------------------
    MortarBase(const InitializationData &init);

    //------------------------------------------------------------------------
    Uint16 m_tipOffset;

  private:
    //------------------------------------------------------------------------
    DECLARE_OBJECT_VISITOR_API();

    //------------------------------------------------------------------------
    MortarBarrelList m_barrels;

    unsigned m_hitPoints;
};


//----------------------------------------------------------------------------
class TopMortar : public MortarBase
{
  public:
    //------------------------------------------------------------------------
    TopMortar(const InitializationData &init);
    ~TopMortar();

    //------------------------------------------------------------------------
    int getCenterAngle() const;
    OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const;
};

//----------------------------------------------------------------------------
class BottomMortar : public MortarBase
{
  public:
    //------------------------------------------------------------------------
    BottomMortar(const InitializationData &init);
    ~BottomMortar();

    //------------------------------------------------------------------------
    int getCenterAngle() const;
    OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const;
};

//----------------------------------------------------------------------------
class LeftMortar : public MortarBase
{
  public:
    //------------------------------------------------------------------------
    LeftMortar(const InitializationData &init);
    ~LeftMortar();

    //------------------------------------------------------------------------
    int getCenterAngle() const;
    OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const;
};

//----------------------------------------------------------------------------
class RightMortar : public MortarBase
{
  public:
    //------------------------------------------------------------------------
    RightMortar(const InitializationData &init);
    ~RightMortar();

    //------------------------------------------------------------------------
    int getCenterAngle() const;
    OrientatingDecorationBase::Orientation getTipPosition(
        Sint16 &x, Sint16 &y) const;
};

#endif //MORTARS_H
