/***************************************************************************
 *   Copyright (C) 2007 by Kevron Rees                                     *
 *   tripzero@nextabyte.com                                                *
 *
 *   Danny Varner <danny.varner@intel.com>
 *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************
 * simple and generic touchscreen calibration program using the linux 2.6
 * input event interface
 * compile with command: 
 *	"g++ -L/usr/X11R6/lib/ -lX11 -o calibrator calibrator.cpp"
 * V 1.0RC4
 * Maintained by Kevron Rees tripzero@nextabyte.com
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <moblin-system-daemon/moblin-system-client.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <gtk/gtk.h>
#include <stdio.h>
#include <signal.h>
#include <termios.h>
#include <stdlib.h>
#include <sys/time.h>
#include <fstream>
#include <iostream>

///added
using namespace std;

/*****************************************************************************/
/* stuff for event interface */
struct input_event {
	struct timeval time;
	unsigned short type;
	unsigned short code;
	int value;
};

/* event types */
#define EV_SYN			0x00
#define EV_KEY			0x01
#define EV_REL			0x02
#define EV_ABS			0x03

#define SYN_REPORT             0
#define BTN_LEFT               0x110
#define BTN_RIGHT              0x111
#define BTN_TOUCH              0x14a

/*****************************************************************************/
/* some constants */
#define FONT_NAME		"9x15"
#define IDLETIMEOUT		15
#define BLINKPERD		0.16        

#define ROUND_SYMBOL
#define NumRect			5

#if 0 
#define Background		cCYAN
#define	TouchedCross		cYELLOW
#define BlinkCrossBG		cRED
#define BlinkCrossFG		cWHITE
#define nTouchedCross		cBLUE
#define Cross			cWHITE
#define DrawGrid		cWHITE
#define DrawLine		cYELLOW
#define DrawUp			cRED
#define DrawDown		cBLUE
#define TimerLine		cRED
#define PromptText		cBLUE
#else
#define Background		cBLACK
#define	TouchedCross		cYELLOW
#define BlinkCrossBG		cRED
#define BlinkCrossFG		cWHITE
#define nTouchedCross		cBLUE
#define Cross			cYELLOW
#define DrawGrid		cGREEN
#define DrawLine		cYELLOW
#define DrawUp			cRED
#define DrawDown		cGREEN
#define TimerLine		cRED
#define PromptText		cWHITE
#endif

#define N_Colors		10

static char colors[N_Colors][10] =
{ "BLACK", "WHITE", "RED", "YELLOW", "GREEN", "BLUE", "#40C0C0" };
				
static unsigned long pixels[N_Colors];

#define cBLACK			(pixels[0])
#define cWHITE			(pixels[1])
#define cRED			(pixels[2])
#define cYELLOW			(pixels[3])
#define cGREEN			(pixels[4])
#define cBLUE			(pixels[5])
#define cCYAN			(pixels[6])

/* some stupid loops */
#define SYS_1( zzz... ) do {	\
	while ( (zzz) != 1 );	\
} while (0)

#define SYS_0( zzz... ) do {	\
	while ( (zzz) != 0 );	\
} while (0)

/* where the calibration points are placed */
#define SCREEN_DIVIDE	16
#define SCREEN_MAX	0x1000
#define M_POINT		(SCREEN_MAX/SCREEN_DIVIDE)
int MARK_POINT[] = { M_POINT, SCREEN_MAX - 1 - M_POINT };

#define FLAG_SWAPX              0x1
#define TST_SWAPX(x)            ((x)&FLAG_SWAPX)
#define SET_SWAPX(x)            (x) |= FLAG_SWAPX
#define CLR_SWAPX(x)            (x) &= ~FLAG_SWAPX
#define FLAG_SWAPY              0x2
#define TST_SWAPY(x)            ((x)&FLAG_SWAPY)
#define SET_SWAPY(x)            (x) |= FLAG_SWAPY
#define CLR_SWAPY(x)            (x) &= ~FLAG_SWAPY
#define FLAG_ROTCW              0x4
#define TST_ROTCW(x)            ((x)&FLAG_ROTCW)
#define SET_ROTCW(x)            (x) |= FLAG_ROTCW
#define CLR_ROTCW(x)            (x) &= ~FLAG_ROTCW
#define FLAG_ROTCCW             0x8
#define TST_ROTCCW(x)           ((x)&FLAG_ROTCCW)
#define SET_ROTCCW(x)           (x) |= FLAG_ROTCCW
#define CLR_ROTCCW(x)           (x) &= ~FLAG_ROTCCW
#define CLR_ROT(x)              (x) &= ~(FLAG_ROTCW | FLAG_ROTCCW)

/*****************************************************************************/
/* globals */
int job_done = 0;
int points_touched = 0;
int points_x[4], points_y[4];
int apply_values[4] = {0, 0, 0, 0};
unsigned int apply_flags = 0;
Display *display;
int screen;
int leaving = 0;
GC gc;
Window root;
Window win;
XFontStruct *font_info;
unsigned int width, height;	/* window size */

static DBusHandlerResult dbus_filter_callback(DBusConnection*, DBusMessage*, void*);

static DBusGConnection *connection = NULL;
static DBusGProxy *proxy = NULL;

static void
complete_shutdown(int ret)
{
GError *gerror = NULL;
DBusError error;
DBusConnection *cnct;

	if(connection != NULL)
	{
		if(!org_moblin_SystemDaemon_cancel_touch(proxy, &gerror))
		{
			g_warning ("cancel_touch error: %s", gerror->message);
			g_error_free (gerror);
		}
		cnct = dbus_g_connection_get_connection(connection);
		dbus_error_init(&error);
		dbus_bus_remove_match (cnct,
			"type='signal',"
			"interface='org.moblin.SystemDaemon',"
			"sender='org.moblin.SystemDaemon',"
			"path='/org/moblin/SystemDaemon'", &error);
		if(dbus_error_is_set(&error)) 
			dbus_error_free(&error);
		dbus_connection_remove_filter (cnct, dbus_filter_callback, NULL);
		dbus_g_connection_unref(connection);
	}
	exit(ret);
}

gboolean 
activate_system_daemon (void)
{
    GError *error = NULL;

    connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
    if (connection == NULL)
    {
	g_warning("Failed to get dbus connection\n");
	g_error_free (error);
	return FALSE;
    }
    proxy = dbus_g_proxy_new_for_name (connection,
                                     "org.moblin.SystemDaemon",
                                     "/org/moblin/SystemDaemon",
                                     "org.moblin.SystemDaemon");
    if (proxy == NULL)
    {
	g_warning("Failed to establish DBUS connection\n");
	return FALSE;
    }
    return TRUE;
}

/*****************************************************************************/
void calibrate_complete()
{
	GtkWidget *dialog;
	GError *error;
	int response;
	dialog = gtk_message_dialog_new (NULL,
					GTK_DIALOG_MODAL,	
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_YES_NO,
					"Click yes to restart GUI and apply\nClick no to discard changes");
	gtk_window_set_title(GTK_WINDOW(dialog), "Apply Calibration?");
	response = gtk_dialog_run(GTK_DIALOG(dialog));
	if (response == GTK_RESPONSE_YES)
	{
		g_debug("Apply Calibration: MinX=%d, MaxX=%d, MinY=%d, MaxY=%d\n",
			apply_values[0], apply_values[1], apply_values[2],
			apply_values[3]);
		if(!org_moblin_SystemDaemon_apply_calibration(proxy, 
			apply_values[0], apply_values[1], apply_values[2], 
			apply_values[3], apply_flags, &error))
		{
			g_warning ("apply_calibration error: %s", error->message);
			g_error_free (error);
			complete_shutdown(-1);
		}
		system("killall xinit");
	}
	gtk_widget_destroy(dialog);
}

void cleanup_exit()
{
	GError *error;

	leaving = 1;
	SYS_1(XUnloadFont(display, font_info->fid));
	XUngrabServer(display);
	XUngrabKeyboard(display, CurrentTime);
	SYS_1(XFreeGC(display, gc));
	SYS_0(XCloseDisplay(display));
	if(job_done == 1)
		calibrate_complete();
	complete_shutdown(0);
}


void load_font(XFontStruct **font_info)
{
	if ((*font_info = XLoadQueryFont(display, FONT_NAME)) == NULL) {
		printf("Cannot open %s font\n", FONT_NAME);
		complete_shutdown(-1);
	}
}


void draw_point(int x, int y, int width, int size, unsigned long color)
{
	XSetForeground(display, gc, color);
	XSetLineAttributes(display, gc, width, LineSolid,
			   CapRound, JoinRound);
	XDrawLine(display, win, gc, x - size, y, x + size, y);
	XDrawLine(display, win, gc, x, y - size, x, y + size);
}


void point_blink(unsigned long color)
{
	int i, j;
	int cx, cy;
	static int shift = 0;

	if (points_touched != 4) {
		int RectDist = width / 200;
		i = points_touched / 2;
		j = points_touched % 2;
		cx = (MARK_POINT[j] * width) / SCREEN_MAX;
		cy = (MARK_POINT[i] * height) / SCREEN_MAX;

		XSetLineAttributes(display, gc, 1, LineSolid, CapRound, JoinRound);
		for (i = 0; i < NumRect; i++) {
			if ((i + shift) % NumRect == 0)
				XSetForeground(display, gc, BlinkCrossBG);
			else
				XSetForeground(display, gc, BlinkCrossFG);

#ifdef ROUND_SYMBOL
			XDrawArc(display, win, gc,
			         cx - i * RectDist, cy - i * RectDist,
			         i * (2 * RectDist), i * (2 * RectDist),
			         0, 359 * 64);
#else
			XDrawRectangle(display, win, gc,
				       cx - i * RectDist, cy - i * RectDist,
				       i * (2 * RectDist), i * (2 * RectDist));
#endif
		}
		shift++;
	}
}


void draw_message(const char *msg)
{
	char buf[300];
	char *prompt[] = { buf };
#define num	(sizeof(prompt) / sizeof(prompt[0]))
	static int init = 0;
	static int p_len[num];
	static int p_width[num];
	static int p_height;
	static int p_maxwidth = 0;
	int i, x, y;
	int line_height;

	strncpy(buf, msg, sizeof buf);

	for (i = 0; i < num; i++) {
		p_len[i] = strlen(prompt[i]);
		p_width[i] = XTextWidth(font_info, prompt[i], p_len[i]);

		if (p_width[i] > p_maxwidth)
			p_maxwidth = p_width[i];
	}

	p_height = font_info->ascent + font_info->descent;
	init = 1;

	line_height = p_height + 5;
	x = (width - p_maxwidth) / 2;
	y = height / 2 - line_height;

	XSetForeground(display, gc, PromptText);
	XSetLineAttributes(display, gc, 3, LineSolid, CapRound, JoinRound);
	XClearArea(display, win, x - 8, y - 8 - p_height, p_maxwidth + 8 * 2,
	           num * line_height + 8 * 2, False);
	XDrawRectangle(display, win, gc, x - 8, y - 8 - p_height,
	               p_maxwidth + 8 * 2, num * line_height + 8 * 2);

	for (i = 0; i < num; i++) {
		XDrawString(display, win, gc, x, y + i * line_height, prompt[i],
			    p_len[i]);
	}
#undef num
}


void draw_text()
{
	static const char *prompt[] = {
		"              4-Pt Calibration",
		"Tap the blinking target to calibrate your screen.",
		"               (ESC to Abort)",
	};
#define num	(sizeof(prompt) / sizeof(prompt[0]))
	static int init = 0;
	static int p_len[num];
	static int p_width[num];
	static int p_height;
	static int p_maxwidth = 0;
	int i, x, y;
	int line_height;

	if (!init) {
		for (i = 0; i < num; i++) {
			p_len[i] = strlen(prompt[i]);
			p_width[i] = XTextWidth(font_info, prompt[i], p_len[i]);
			if (p_width[i] > p_maxwidth)
				p_maxwidth = p_width[i];
		}
		p_height = font_info->ascent + font_info->descent;
		init = 1;
	}
	line_height = p_height + 5;
	x = (width - p_maxwidth) / 2;
	y = height / 2 - 6 * line_height;

	XSetForeground(display, gc, PromptText);
	XClearArea(display, win, x - 11, y - 8 - p_height,
		   p_maxwidth + 11 * 2, num * line_height + 8 * 2, False);
	XSetLineAttributes(display, gc, 3, FillSolid,
			   CapRound, JoinRound);
	XDrawRectangle(display, win, gc, x - 11, y - 8 - p_height,
		       p_maxwidth + 11 * 2, num * line_height + 8 * 2);

	for (i = 0; i < num; i++) {
		XDrawString(display, win, gc, x, y + i * line_height, prompt[i],
			    p_len[i]);
	}
#undef num
}


void draw_graphics()
{
	int i, j;
	unsigned cx, cy;
	unsigned long color;

	draw_text();

	for (i = 0; i < 2; i++) {
		for (j = 0; j < 2; j++) {
			int num = 2 * i + j;

			if (num == points_touched)
				continue;
			
			if (num > points_touched)
				color = nTouchedCross;
			else
				color = TouchedCross;

			cx = (MARK_POINT[j] * width) / SCREEN_MAX;
			cy = (MARK_POINT[i] * height) / SCREEN_MAX;
			draw_point(cx, cy, width / 200, width / 64, color);
		}
	}
}


void get_gc(Window win, GC *gc, XFontStruct *font_info)
{
	unsigned long valuemask = 0;	/* ignore XGCvalues and use defaults */
	XGCValues values;
	unsigned int line_width = 5;
	int line_style = LineSolid;
	int cap_style = CapRound;
	int join_style = JoinRound;

	*gc = XCreateGC(display, win, valuemask, &values);

	XSetFont(display, *gc, font_info->fid);

	XSetLineAttributes(display, *gc, line_width, line_style,
			   cap_style, join_style);
}


int get_color()
{
	int default_depth;
	Colormap default_cmap;
	XColor my_color;
	int i;

	default_depth = DefaultDepth(display, screen);
	default_cmap = DefaultColormap(display, screen);

	for (i = 0; i < N_Colors; i++) {
		XParseColor(display, default_cmap, colors[i], &my_color);
		XAllocColor(display, default_cmap, &my_color);
		pixels[i] = my_color.pixel;
	}

	return 0;
}


Cursor create_empty_cursor()
{
	char nothing[] = { 0 };
	XColor nullcolor;
	Pixmap src = XCreateBitmapFromData(display, root, nothing, 1, 1);
	Pixmap msk = XCreateBitmapFromData(display, root, nothing, 1, 1);
	Cursor mycursor = XCreatePixmapCursor(display, src, msk,
					      &nullcolor, &nullcolor, 0, 0);
	XFreePixmap(display, src);
	XFreePixmap(display, msk);

	return mycursor;
}


void process_event()
{
	XEvent event;

	while (XCheckWindowEvent(display, win, -1, &event) == True) {
		switch (event.type) {
		case KeyPress:
			{
				KeySym keysym = XKeycodeToKeysym(display,
							     event.xkey.keycode, 0);

				if (keysym == XK_Escape) {
					puts("Aborted");
					cleanup_exit();
				}
			}
			break;

		case Expose:
			draw_graphics(/*win, gc, width, height*/);
			break;

		default:
			break;
		}
	}
}


double idle_time = 0;
double tick = 0;

void set_timer(double interval /* in second */ )
{
	struct itimerval timer;
	long sec = interval;
	long usec = (interval - sec) * 1.0e6;

	timer.it_value.tv_sec = sec;
	timer.it_value.tv_usec = usec;
	timer.it_interval = timer.it_value;
	setitimer(ITIMER_REAL, &timer, NULL);
	tick = interval;
}


void update_timer(void)
{
	int current = width * idle_time / IDLETIMEOUT;

	XSetLineAttributes(display, gc, 2, LineSolid, CapRound, JoinRound);
	XSetForeground(display, gc, Background);
	XDrawLine(display, win, gc, 0, height - 1, current, height - 1);
	XSetForeground(display, gc, TimerLine);
	XDrawLine(display, win, gc, current, height - 1, width, height - 1);
}

void sig_handler(int num)
{
	static int is_busy = 0;

	if (is_busy)
		return;
	is_busy = 1;

	if(num == SIGALRM) {
		if (!job_done)
			point_blink(BlinkCrossFG);

		update_timer();
		if (idle_time >= IDLETIMEOUT)
			cleanup_exit();

		idle_time += tick;
		XFlush(display);
		process_event();
	}

	is_busy = 0;

	return;
}

static void
capture_touch(gint x, gint y)
{
GError *error = NULL;

	if(leaving) return;

	idle_time = 0;

	points_x[points_touched] = x;
	points_y[points_touched] = y;
		
	points_touched++;
	draw_graphics();

	/* do the math */
	if (points_touched == 4 && !job_done) {
		int x_low, y_low, x_hi, y_hi;
		int x_min, y_min, x_max, y_max;
		int x_seg, y_seg;
		unsigned int flags = 0;

		if((abs(points_x[0] - points_x[1])<100)&&
		   (abs(points_x[2] - points_x[3])<100))
    		{
		  /* cw rotated */
		  x_low = (points_x[2] + points_x[3]) / 2;
    		  y_low = (points_y[0] + points_y[2]) / 2;
    		  x_hi = (points_x[0] + points_x[1]) / 2;
    		  y_hi = (points_y[1] + points_y[3]) / 2;
		  SET_ROTCW(flags);
		}
		else
    		{
		  /* normal */
		  x_low = (points_x[0] + points_x[2]) / 2;
    		  y_low = (points_y[0] + points_y[1]) / 2;
   		  x_hi = (points_x[1] + points_x[3]) / 2;
    		  y_hi = (points_y[2] + points_y[3]) / 2;
		}

		/* if x is inverted, use SwapX */
		if(x_low > x_hi)
		{
		  int tmp = x_hi;
		  x_hi = x_low;
		  x_low = tmp;
		  SET_SWAPX(flags);
		}

		/* if y is inverted, use SwapY */
		if(y_low > y_hi)
		{
		  int tmp = y_hi;
		  y_hi = y_low;
		  y_low = tmp;
		  SET_SWAPY(flags);
		}

		/* if cw and both axes inverted, convert to ccw */
		if(TST_ROTCW(flags)&&TST_SWAPX(flags)&&TST_SWAPY(flags)) {
		  CLR_SWAPX(flags);
		  CLR_SWAPY(flags);
		  CLR_ROTCW(flags);
		  SET_ROTCCW(flags);
		}

		/* calc the min and max values */
		x_seg = abs(x_hi - x_low) / (SCREEN_DIVIDE - 2);
		x_min = x_low - x_seg;
		x_max = x_hi + x_seg;

		y_seg = abs(y_hi - y_low) / (SCREEN_DIVIDE - 2);
		y_min = y_low - y_seg;
		y_max = y_hi + y_seg;

		draw_message("   Calibration Successful.  Restart X for new changes.   ");
		XFlush(display);
		apply_values[0] = x_min;
		apply_values[1] = x_max;
		apply_values[2] = y_min;
		apply_values[3] = y_max;
		apply_flags = flags;
		job_done = 1;
		idle_time = IDLETIMEOUT;
		update_timer();
	} else if (!job_done) {
		/* wait .2 seconds to ensure no double taps on the same point */
		g_usleep(200000);
		if(!org_moblin_SystemDaemon_capture_touch(proxy, &error))
		{
			g_warning ("capture_touch %d error: %s", 
				points_touched, error->message);
			g_error_free (error);
			complete_shutdown(-1);
		}
	}
}

static DBusHandlerResult
dbus_filter_callback (DBusConnection * connection, DBusMessage * message, void *user_data)
{
DBusError error;
gint x, y;

	if (dbus_message_is_signal (message, 
		"org.moblin.SystemDaemon", "TouchScreenData"))
	{
		dbus_error_init(&error);
		if (dbus_message_get_args (message, &error,
			DBUS_TYPE_INT32, &x, DBUS_TYPE_INT32, &y,
			DBUS_TYPE_INVALID)) {
			capture_touch(x, y);
		} else if(dbus_error_is_set(&error)) {
			dbus_error_free(&error);
		}
	}

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

int main(int argc, char *argv[], char *env[])
{
	char *display_name = NULL;
	XSetWindowAttributes xswa;
	char answer;
	GError *gerror = NULL;
	DBusError error;
	DBusConnection *cnct;

	gtk_init(&argc, &argv);

	if(!activate_system_daemon()) exit(-1);

	/* connect to X server */
	if ((display = XOpenDisplay(display_name)) == NULL) {
		g_warning("cannot connect to X server %s\n",
			XDisplayName(display_name));
		return FALSE;
	}

	screen = DefaultScreen(display);
	root = RootWindow(display, screen);

	/* setup window attributes */
	xswa.override_redirect = True;
	xswa.background_pixel = BlackPixel(display, screen);
	xswa.event_mask = ExposureMask | KeyPressMask;
	xswa.cursor = create_empty_cursor();

	/* get screen size from display structure macro */
	
	width = DisplayWidth(display, screen);
	height = DisplayHeight(display, screen);
	g_debug("Trying to calibrate screen to: %dx%d resolution\n",
		width, height);

	win = XCreateWindow(display, RootWindow(display, screen),
	                    0, 0, width, height, 0,
	                    CopyFromParent, InputOutput, CopyFromParent,
	                    CWOverrideRedirect | CWBackPixel | CWEventMask |
	                    CWCursor, &xswa);
	XMapWindow(display, win);
	XGrabKeyboard(display, win, False, GrabModeAsync, GrabModeAsync,
	              CurrentTime);
	XGrabServer(display);
	load_font(&font_info);
	get_gc(win, &gc, font_info);
	get_color();

	XSetWindowBackground(display, win, Background);
	XClearWindow(display, win);

	signal(SIGALRM, sig_handler);
	set_timer(BLINKPERD);

	cnct = dbus_g_connection_get_connection(connection);
	if (!dbus_connection_add_filter (cnct, dbus_filter_callback, NULL, NULL)) {
		g_warning("dbus_connection_add_filter failed\n");
		complete_shutdown(-1);
	}
	dbus_error_init(&error);
	dbus_bus_add_match (cnct,
			"type='signal',"
			"interface='org.moblin.SystemDaemon',"
			"sender='org.moblin.SystemDaemon',"
			"path='/org/moblin/SystemDaemon'", &error);
	if(dbus_error_is_set(&error)) 
	{
		g_warning("dbus_bus_add_match failed\n");
		dbus_error_free(&error);
	}
	if(!org_moblin_SystemDaemon_capture_touch(proxy, &gerror))
	{
		g_warning ("init capture_touch error: %s", gerror->message);
		g_error_free (gerror);
		complete_shutdown(-1);
	}

	gtk_main();

	complete_shutdown(0);
	return 0;
}
