#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
#include <apr_errno.h>
#include <string.h>

#include <libbtutil/const.h>
#include <libbtutil/types/bt_bcode.h>
#include <libbtpeer/types/btp_torrent.h>

static const char* btp_torrent_events[] = {
    "started",
    "stopped",
    "completed",
    NULL
};

apr_status_t btp_torrent_announce(btp_torrent* t) {
    apr_pool_t* p;
    int want = t->peers->max - t->peers->n;
    apr_size_t buf_len = BT_SHORT_STRING + (want * BT_PEERSTR_LEN * 2);
    char* buf;
    int event;
    apr_status_t ret;
    bt_bcode* reply;
    time_t now = time(NULL);
    
    apr_pool_create(&p, t->pool);
    buf = apr_palloc(p, buf_len);
    
    if(t->status & BTP_TORRENT_STATUS_QUIT)
        event = BTP_TORRENT_EVENT_STARTED;
    if(!(t->status & BTP_TORRENT_STATUS_ANNOUNCED))
        event = BTP_TORRENT_EVENT_STOPPED;
    else if(
        (btp_torrent_bytes_left(t) == 0) &&
        !(t->status & BTP_TORRENT_STATUS_SEED)
    )
        event = BTP_TORRENT_EVENT_COMPLETED;
    else
        event = BTP_TORRENT_EVENT_NONE;

    t->l_announce_t = now;
    t->status |= BTP_TORRENT_STATUS_ANNOUNCED;
        
    if(
        (ret = btp_torrent_send_announce(
            t, btp_torrent_events[event], want, buf, &buf_len
        )) == APR_SUCCESS
    ) {
        reply = apr_palloc(p, sizeof(bt_bcode));
        if((ret = bt_bcode_decode(p, buf, reply, NULL)) == BT_BCODE_SUCCESS) {
            bt_bcode* interval;
            bt_bcode* data;
            if((data = bt_bcode_find(reply, "failure reason"))) {
                BT_STRCPY(t->error, data->val.s.s);
                t->n_announce_t = now + BTP_TORRENT_ANNOUNCE_RETRY;
            } else if((interval = bt_bcode_find(reply, "interval"))) {
                t->n_announce_t = now + interval->val.i;
                
                if((data = bt_bcode_find(reply, "peers")))
                    ret = btp_torrent_receive_peerlist(t, data);
                else
                    ret = APR_SUCCESS;
                
            } else {
                BT_STRCPY(
                    t->error,
                    "Couldn't find \"interval\" in response from tracker!"
                );
                t->n_announce_t = now + BTP_TORRENT_ANNOUNCE_RETRY;
                /* TODO: fix this error return code */
                ret = APR_EGENERAL;
            }
        }
    } else {
        t->n_announce_t = now + BTP_TORRENT_ANNOUNCE_RETRY;
    }        

    return ret;
}

apr_status_t btp_torrent_send_announce(
    btp_torrent* t, const char* event, int numwant,
    char* result, apr_size_t* result_len
) {
    apr_status_t ret;
    apr_pool_t* temp;
    char* query;
    char* retbuf;
    apr_size_t retbuf_len;

    apr_pool_create(&temp, t->pool);
    
    query = apr_psprintf(
        temp,
        "%s?"
        "info_hash=%s&"
        "peer_id=%s&"
        "port=%i&"
        "numwant=%i&"
        "uploaded=%llu&"
        "downloaded=%llu&"
        "left=%llu"
        "%s%s%s%s",
        t->info->tracker_announce,
        bt_uri_escape(temp, (char*)t->info->hash, sizeof(t->info->hash)),
        bt_uri_escape(temp, t->id, sizeof(t->id)),
        t->port,
        numwant,
        t->uploaded,
        t->downloaded,
        btp_torrent_bytes_left(t),
        (t->ip[0] ? "&ip=" : ""), t->ip,
        ((event && *event) ? "&event=" : ""), event ? event : ""
    );
    
    ret = bt_http_get(
        temp, t->info->tracker_address, t->info->tracker_port, query,
        &retbuf, &retbuf_len
    );
    
    if(ret == APR_SUCCESS) {
        if(retbuf_len > *result_len) {
            ret = APR_ENOMEM;
            memcpy(result, retbuf, *result_len);
        } else {
            memcpy(result, retbuf, retbuf_len);
            *result_len = retbuf_len;
        }
    }
    
    apr_pool_destroy(temp);
    return ret;
}
