/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* other libs */
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
/* local */
#include <libbttracker.h>

char btt_error_msg[BT_SHORT_STRING];

const btt_tracker_flag btt_tracker_flags[] = {
    {
        BTT_TRACKER_REQUIRE_REG,
        "Infohash Registration Required",
        "RequireReg"
    },
    {
        BTT_TRACKER_ALLOW_SCRAPE_FULL,
        "Full Scrape Allowed",
        "AllowScrapeFull"
    },
    {
        BTT_TRACKER_ALLOW_SCRAPE_VERBOSE,
        "Verbose Scrape Allowed",
        "AllowScrapeVerbose"
    },
    {
        BTT_TRACKER_ALLOW_SCRAPE_XML,
        "XML Scrape Allowed",
        "AllowScrapeXML"
    },
    {
        BTT_TRACKER_ALLOW_SCRAPE_GENERAL,
        "General Scrape Allowed",
        "AllowScrapeGeneral"
    },
    {
        0,
        NULL,
        NULL
    }
};

const btt_tracker_config new_btt_tracker_config = {
    BTT_CSS,    /* stylesheet */
    BTT_HOME,   /* db_dir */
    "",         /* detail_url */
    "",         /* root_include */
    (           /* flags */
        BTT_TRACKER_ALLOW_SCRAPE_FULL |
        BTT_TRACKER_ALLOW_SCRAPE_VERBOSE |
        BTT_TRACKER_ALLOW_SCRAPE_XML |
        BTT_TRACKER_ALLOW_SCRAPE_GENERAL  | 
        BTT_TRACKER_BTREE
    ),
    50,         /* random_retry */
    50,         /* return_peers */
    500,        /* return_max */
    600,        /* return_interval */
    1000,       /* return_peer_factor */
    500,        /* hash_watermark */
    1800,       /* hash_min_age */
    3600,       /* hash_max_age */
    NULL        /* parent_server */
};

const btt_tracker_stats new_btt_tracker_stats = {
    0,          /* num_children */
    0,          /* num_requests */
    0,          /* num_hashes */
    0,          /* num_peers */
    0,          /* announces */
    0,          /* scrapes */
    0,          /* full_scrapes */
    0,          /* bad_announces */
    0,          /* bad_scrapes */
    0,          /* start_t */
    0,          /* server_time */
    0           /* panic */
};

const btt_tracker new_btt_tracker = {
    NULL,               /* m(emory) */
    new_btt_db_d,       /* db */
    NULL,               /* p(ool) */
    NULL,               /* c(onfig) */
    NULL,               /* s(stats) */
    0,                  /* myapr */
    NULL,               /* shlock */
    BTT_HOME            /* homedir */
};

const char* tracker_abilities =
    " <Abilities>\n"
    "  <Ability ID=\"Compact\">Compact Announce</Ability>\n"
    "  <Ability ID=\"NoPeerID\">No Peer ID Announce</Ability>\n"
    "  <Ability ID=\"ScrapeFul\">Full Scrape</Ability>\n"
    "  <Ability ID=\"ScrapeXML\">XML Scrape</Ability>\n"
    "  <Ability ID=\"ScrapeSpecific\">Specific InfoHash Scrape</Ability>\n"
    "  <Ability ID=\"ScrapeMulti\">Multiple InfoHash Scrape</Ability>\n"
    "  <Ability ID=\"Shield\">Peer Shielding</Ability>\n"
    " </Abilities>\n"
;

int btt_tracker2info_trs(apr_pool_t* p, btt_tracker* tracker, char** result) {
    char *version;
    char *server_time;
    char *start_time;
    char *return_interval;
 
    if(tracker->c->parent_server)
        version = apr_psprintf(
            p, "mod_bt/"VERSION" (%s)", tracker->c->parent_server
        );
    else
        version = "mod_bt/"VERSION;

    server_time = apr_pstrdup(p, ctime(&(tracker->s->server_time)));
    start_time = apr_pstrdup(p, ctime(&(tracker->s->start_t)));

    if(tracker->c->return_peer_factor)
        return_interval = apr_psprintf(
            p, "((%"PRIu32" / peers) + 1) * "BT_TIME_T_FMT"s",
            tracker->c->return_peer_factor, tracker->c->return_interval
        );
    else
        return_interval = apr_psprintf(
            p, BT_TIME_T_FMT"s", tracker->c->return_interval
        );

    char* rv = apr_psprintf(
        p,
        " <TR><TH>Server Version:</TH><TD>%s</TD></TR>\n"
        " <TR><TH>Server Time:</TH><TD>%s</TD></TR>\n"
        " <TR><TH>Startup Time:</TH><TD>%s</TD></TR>\n"
        " <TR><TH>Hashes:</TH><TD>%"PRIu32"</TD></TR>\n"
        " <TR><TH>Peers:</TH><TD>%"PRIu32"</TD></TR>\n"
        " <TR><TH>Announces:</TH><TD>%"PRIu64" (%.02f/s)</TD></TR>\n"
        " <TR><TH>Faulty Announces:</TH><TD>%"PRIu64"</TD></TR>\n"
        " <TR><TH>Peers Returned:</TH><TD>0 < %"PRIu32" < %"PRIu32"</TD></TR>"
        "\n <TR><TH>Return Interval:</TH><TD>%s</TD></TR>\n"
        " <TR><TH>Hash Expiry:</TH><TD>"BT_TIME_T_FMT
        "s - "BT_TIME_T_FMT"s (Watermark %"PRIu32")</TD></TR>\n"
        " <TR><TH>Registration Required:</TH><TD>%s</TD></TR>\n",
        version, server_time, start_time,
        tracker->s->num_hashes, tracker->s->num_peers,
        tracker->s->announces,
        (double)(
            (double)tracker->s->announces /
            (double)(
                (double)tracker->s->server_time - (double)tracker->s->start_t
            )
        ),
        tracker->s->bad_announces, tracker->c->return_peers,
        tracker->c->return_max,
        return_interval, tracker->c->hash_min_age, tracker->c->hash_max_age,
        tracker->c->hash_watermark,
        tracker->c->flags & BTT_TRACKER_REQUIRE_REG ? "Yes" : "No"
    );
 
    *result = rv;
    return strlen(rv);
}

int btt_tracker2xml(apr_pool_t* p, btt_tracker* tracker, char** result) {
    char* ib;
    char* ob;
    int f = 0;
    char* rv = apr_psprintf(
        p,
        "<Tracker>\n"
        " <Homedir>%s</Homedir>\n"
        "%s"
        " <Config>\n"
        "  <StyleSheet>%s</StyleSheet>\n"
        "  <DbDir>%s</DbDir>\n"
        "  <DetailURL>%s</DetailURL>\n"
        "  <RootInclude>%s</RootInclude>\n",
        tracker->homedir,
        tracker_abilities,
        tracker->c->stylesheet, tracker->c->db_dir,
        tracker->c->detail_url, tracker->c->root_include
    );
 
    if(tracker->c->flags) {
        ib = apr_psprintf(p, "  <Flags VALUE=\"%"PRIu16"\">\n", tracker->c->flags);
        for(f=0;btt_tracker_flags[f].flag;f++) {
            if(tracker->c->flags & btt_tracker_flags[f].flag) {
                ob = apr_psprintf(
                    p, "   <Flag VALUE=\"%u\" ID=\"%s\">%s</Flag>\n",
                    btt_tracker_flags[f].flag,
                    btt_tracker_flags[f].config_name,
                    btt_tracker_flags[f].description
                );
                ib = apr_pstrcat(p, ib, ob, NULL);
            }
        }
        rv = apr_pstrcat(p, rv, ib, "  </Flags>\n", NULL);
    }

    ib = apr_psprintf(
        p,
        "  <RandomRetry>%"PRIu32"</RandomRetry>\n"
        "  <ReturnPeers>%"PRIu32"</ReturnPeers>\n"
        "  <ReturnMax>%"PRIu32"</ReturnMax>\n"
        "  <ReturnInterval>"BT_TIME_T_FMT"</ReturnInterval>\n"
        "  <ReturnPeerFactor>%"PRIu32"</ReturnPeerFactor>\n"
        "  <HashWatermark>%"PRIu32"</HashWatermark>\n"
        "  <HashMinAge>"BT_TIME_T_FMT"</HashMinAge>\n"
        "  <HashMaxAge>"BT_TIME_T_FMT"</HashMaxAge>\n"
        " </Config>\n"
        " <Stats>\n"
        "  <NumChildren>%"PRIu32"</NumChildren>\n"
        "  <NumRequests>%"PRIu32"</NumRequests>\n"
        "  <NumHashes>%"PRIu32"</NumHashes>\n"
        "  <NumPeers>%"PRIu32"</NumPeers>\n"
        "  <Announces>%"PRIu64"</Announces>\n"
        "  <BadAnnounces>%"PRIu64"</BadAnnounces>\n"
        "  <StartT>"BT_TIME_T_FMT"</StartT>\n"
        "  <MasterPID>%u</MasterPID>\n"
        "  <ServerTime>"BT_TIME_T_FMT"</ServerTime>\n"
        " </Stats>\n"
        " <DB>\n"
        "  <Open>%u</Open>\n"
        "  <Dir>%s</Dir>\n"
        "  <Cache>%"APR_SIZE_T_FMT"</Cache>\n"
        " </DB>\n"
        "</Tracker>\n",
        tracker->c->random_retry,
        tracker->c->return_peers, tracker->c->return_max,
        tracker->c->return_interval, tracker->c->return_peer_factor,
        tracker->c->hash_watermark,
        tracker->c->hash_min_age, tracker->c->hash_max_age,
        tracker->s->num_children, tracker->s->num_requests,
        tracker->s->num_hashes, tracker->s->num_peers,
        tracker->s->announces, tracker->s->bad_announces,
        tracker->s->start_t, tracker->s->master_pid,
        tracker->s->server_time,
        tracker->db.open, tracker->db.dir, tracker->db.cache
    );
 
    *result = apr_pstrcat(p, rv, ib, NULL);
    return strlen(*result);
}
