#include <stdio.h>
#include <unistd.h>
#include <glib/gmacros.h>
#include "includes.h"
#include "math.h"

/* Default option values.
 */
char	*gammaVal = "1.0";
char	*balance1 = "camera";
char	*balance2 = "camera";
double	balanceMix = 0.0;	/* Use only balance1. */
double	 contrast = 0.0;
char	*profilePath;		/* Initialised below. */
char	*colorMapPath;		/* Initialised below. */
char	*toneCurvePath;		/* Initialised below. */
char	*cameraProfileName = NULL; /* Overrides default based on camera. */
char	*outputProfileName = "__sRGB__";
int	rotate = 0;
int	outputBits = 8;
int	shrink = 1;
char    *imethod = "vmedian";
double  vmedian_tolerance = 1.0e-4;
int	useLab = TRUE;
char	*toneCurve = "standard";
int	applyVMF = FALSE;
int	applyMedianFilter = FALSE;
double	blur = 2.0;
double	iSharpen = 0.5;
double	cSharpen = -0.5;
char	*cameraDetails = "none";
char	*cameraDetailsPath;		/* Initialised below. */
char	*stuckPixelFile = NULL;
int	previewShrink = 8;
int	previewWidth = 514;
int	previewHeight = 387;
int	histWidth = 256;
int	histHeight = 128;
int	xvmini = 0;
int	filterColorOnly = TRUE;
int	radius = 2;
double  darkness = 0.0;
double  shadows = 0.0;
double  saturation = 0.0;
int     fixShortSide = TRUE;
double  fixedSideLength = 4.0;
int	debugFlag = 0;

void
InitOptions()
{
	char	buffer[2048];
	char	*s;
	char	*home = getenv ("HOME");

	strcpy (buffer, ".:profiles");
	s = buffer + strlen (buffer);  /* Why should I count? */
	if (home) {
		sprintf (s, ":%s/.mrwtoppm/profiles", home);
		s += strlen (s);
	}
        sprintf (s, ":/usr/share/liblcms/profiles:%s/profiles", DATAPREFIX);
	profilePath = strdup (buffer);

	strcpy (buffer, ".:colormaps");
	s = buffer + strlen (buffer); /* So how smart is your C compiler? */
	if (home) {
		sprintf (s, ":%s/.mrwtoppm/colormaps", home);
		s += strlen (s);
	}
	sprintf (s, ":%s/colormaps", DATAPREFIX);
	colorMapPath = strdup (buffer);

	strcpy (buffer, ".:tonecurves");
	s = buffer + strlen (buffer);
	if (home) {
		sprintf (s, ":%s/.mrwtoppm/tonecurves", home);
		s += strlen (s);
	}
	sprintf (s, ":%s/tonecurves", DATAPREFIX);
	toneCurvePath = strdup (buffer);

	strcpy (buffer, ".:cameradetails");
	s = buffer + strlen (buffer);
	if (home) {
		sprintf (s, ":%s/.mrwtoppm/cameradetails", home);
		s += strlen (s);
	}
	sprintf (s, ":%s/cameradetails", DATAPREFIX);
	cameraDetailsPath = strdup (buffer);
}

void
OptionFileUsage()
{
	extern char *progname;
	void PrintInitFileOptions (FILE *f);

	fprintf (stderr, "%s: Error parsing initialization file.\n", progname);
	fprintf (stderr, "Initialization options:\n");
	PrintInitFileOptions (stderr);
	exit (1);
}

static gboolean forMe;
static char *mySection;
static char *currentSection;

static
gboolean do_section (char *name)
{
	if (currentSection) free (currentSection);
	currentSection = strdup (name);
	return TRUE;
}

static gboolean
SetGamma (const char *arg)
{
	gammaVal = strdup (arg);
	return TRUE;
}

static gboolean
SetColorBalance1 (const char *arg)
{
	balance1 = strdup (arg);
	return TRUE;
}

static gboolean
SetColorBalance2 (const char *arg)
{
	balance2 = strdup (arg);
	return TRUE;
}

static gboolean
SetBalanceMix (const char *arg)
{
	balanceMix = atof (arg);
	if (balanceMix < 0.0 || balanceMix > 100.0)
		return FALSE;
	return TRUE;
}

static gboolean
SetContrast (const char *arg)
{
	contrast = atof (arg);
	return contrast >= -10.0 && contrast <= 100.0;
}

static gboolean
SetDarkness (const char *arg)
{
	darkness = atof (arg);
	return darkness >= 0.0 && darkness <= 10.0;
}

static gboolean
SetShadows (const char *arg)
{
	shadows = atof (arg);
	return shadows >= -10.0 && shadows <= 10.0;
	return TRUE;
}

static gboolean
SetSaturation (const char *arg)
{
	saturation = atof (arg);
	return saturation >= -5.0 && saturation <= 10.0;
	return TRUE;
}

static gboolean
SetBlur (const char *arg)
{
	blur = atof (arg);
	if (fabs(blur) < 1.0e-5)
		return FALSE;
	blur = 1.0 / blur;
	return TRUE;
}

static gboolean
SetShortSideLength (const char *arg)
{
	fixedSideLength = atof (arg);
	if (fixedSideLength < 1.0e-5)
		return FALSE;
	fixShortSide = TRUE;
	return TRUE;
}

static gboolean
SetLongSideLength (const char *arg)
{
	fixedSideLength = atof (arg);
	if (fixedSideLength < 1.0e-5)
		return FALSE;
	fixShortSide = FALSE;
	return TRUE;
}

static gboolean
SetScale (const char *arg)
{
	if (sscanf (arg, "1/%d", &shrink) == 1)
		return TRUE;
	else if (strcmp (arg, "1") == 0) {
		shrink = 1;
		return TRUE;
	}
	return FALSE;
}

static gboolean
SetPreviewScale (const char *arg)
{
	if (sscanf (arg, "1/%d", &previewShrink) == 1)
		return TRUE;
	else if (strcmp (arg, "1") == 0) {
		previewShrink = 1;
		return TRUE;
	}
	return FALSE;
}

static gboolean
SetISharpen (const char *arg)
{
	iSharpen = -atof (arg);
	return TRUE;
}

static gboolean
SetCSharpen (const char *arg)
{
	cSharpen = -atof (arg);
	return TRUE;
}

static gboolean
SetToneCurve (const char *arg)
{
	toneCurve = strdup (arg);
	return TRUE;
}

static gboolean
SetProfilePath (const char *arg)
{
	profilePath = strdup (arg);
	return TRUE;
}

static gboolean
SetColorMapPath (const char *arg)
{
	colorMapPath = strdup (arg);
	return TRUE;
}

static gboolean
SetToneCurvePath (const char *arg)
{
	toneCurvePath = strdup (arg);
	return TRUE;
}

static gboolean
SetCProfile (const char *arg)
{
	cameraProfileName = strdup (arg);
	return TRUE;
}

static gboolean
SetOProfile (const char *arg)
{
	outputProfileName = strdup (arg);
	return TRUE;
}

static gboolean
SetInterpolationMethod (const char *arg)
{
	imethod = strdup (arg);
	return TRUE;
}

static gboolean
SetCameraDetails (const char *arg)
{
	cameraDetails = strdup (arg);
	return TRUE;
}

static gboolean
SetCameraDetailsPath (const char *arg)
{
	cameraDetailsPath = strdup (arg);
	return TRUE;
}

static gboolean
SetStuckPixelFile (const char *arg)
{
	stuckPixelFile = strdup (arg);
	return TRUE;
}

static gboolean
SetVTolerance (const char *arg)
{
	vmedian_tolerance = atof (arg);
	if (vmedian_tolerance <= 0.0)
		vmedian_tolerance = 1.0 / (0xFFF*0xFFF);
	return TRUE;
}

static gboolean
SetVMF (const char *arg)
{
	applyVMF = TRUE;
	return TRUE;
}

static gboolean
SetXVMini (const char *arg)
{
	xvmini = TRUE;
	return TRUE;
}

static gboolean
SetNoVMF (const char *arg)
{
	applyVMF = FALSE;
	return TRUE;
}

static gboolean
SetMedianFilter (const char *arg)
{
	applyMedianFilter = TRUE;
	filterColorOnly = FALSE;
	return TRUE;
}

static gboolean
SetCMedianFilter (const char *arg)
{
	applyMedianFilter = TRUE;
	filterColorOnly = TRUE;
	return TRUE;
}

static gboolean
SetNoMedianFilter (const char *arg)
{
	applyMedianFilter = FALSE;
	return TRUE;
}

static gboolean
SetLab (const char *arg)
{
	useLab = TRUE;
	return TRUE;
}

static gboolean
SetNoLab (const char *arg)
{
	useLab = FALSE;
	return TRUE;
}

static gboolean
SetRotate (const char *arg)
{
	rotate = atoi (arg);
	if (rotate == 0 || rotate == 90 || rotate == 180 || rotate == 270)
		return TRUE;
	else
		return FALSE;
}

static gboolean
SetRadius (const char *arg)
{
	radius = atoi (arg);
	if (radius > 0 && radius < 6)
		return TRUE;
	else
		return FALSE;
}

static gboolean
SetHistWidth (const char *arg)
{
	histWidth = atoi (arg);
	if (histWidth > 0)
		return TRUE;
	else
		return FALSE;
}

static gboolean
SetHistHeight (const char *arg)
{
	histHeight = atoi (arg);
	if (histHeight > 0)
		return TRUE;
	else
		return FALSE;
}

static gboolean
Set8 (const char *arg)
{
	outputBits = 8;
	return TRUE;
}

static gboolean
Set16 (const char *arg)
{
	outputBits = 16;
	return TRUE;
}

typedef struct {
	char	*param;
	char	*paramhelp;	/* Textual description for help. */
	char	*arghelp;	/* Textual description for help. */
	gboolean   has_arg;
	gboolean	(*func) (const char *arg);
} handler;

handler htable[] = {
	{ "cmapPath", "set colormap search path", "pathlist", TRUE, SetColorMapPath },
	{ "wb", "set white balance", "colormapname | rgain:bgain", TRUE, SetColorBalance1 },
	{ "wb2", "set second white balance", "colormapname | rgain:bgain", TRUE, SetColorBalance2 },
	{ "balanceMix", "set percentage of second white balance", "amount (>= 0 && <= 100)", TRUE, SetBalanceMix },
	{ "cameradetails", "set camera details", "mrwfile", TRUE, SetCameraDetails },
	{ "cameradetailsPath", "set camera details search path", "pathlist", TRUE, SetCameraDetailsPath },
	{ "stuck", "set stuck pixels", "filename", TRUE, SetStuckPixelFile },
	{ "profilePath", "set profile search path", "pathlist", TRUE, SetProfilePath },
	{ "cprofile", "set camera color space", "profilename", TRUE, SetCProfile },
	{ "scale", "scale image (N = 1 or 1/X, X even)", "N", TRUE, SetScale },
	{ "previewScale", "scale preview (N = 1/X, X even)", "N", TRUE, SetPreviewScale },
	{ "imethod", "set interpolation method", "name", TRUE, SetInterpolationMethod },
	{ "vtol", "set vector median tolerance", "val", TRUE, SetVTolerance },
	{ "Lab", "do most operations in CIE Lab color space", "", FALSE, SetLab },
	{ "noLab", "do most operations median in camera's color space", "", FALSE, SetNoLab },
	{ "vmf", "apply vector median filter", "", FALSE, SetVMF },
	{ "novmf", "don't apply vector median filter", "", FALSE, SetNoVMF },
	{ "mfilter", "apply median filter", "", FALSE, SetMedianFilter },
	{ "cmfilter", "apply median filter to color only", "", FALSE, SetCMedianFilter },
	{ "nomfilter", "don't apply median filter", "", FALSE, SetNoMedianFilter },
	{ "blur", "set blur factor", "val", TRUE, SetBlur },
	{ "isharp", "set intensity sharpen", "val", TRUE, SetISharpen },
	{ "csharp", "set color sharpen (Lab mode only)", "val", TRUE, SetCSharpen },

	{ "rotate", "rotate image", "angle", TRUE, SetRotate },
	{ "radius", "blur radius", "size", TRUE, SetRadius },
	{ "hwidth", "histogram width", "size", TRUE, SetHistWidth },
	{ "hheight", "histogram height", "size", TRUE, SetHistHeight },
	{ "shortside", "set short side length", "size", TRUE, SetShortSideLength },
	{ "longside", "set long side length", "size", TRUE, SetLongSideLength },
	{ "oprofile", "set output color space", "profilename", TRUE, SetOProfile },
	{ "contrast", "set contrast", "val", TRUE, SetContrast },
	{ "darkness", "set darkness", "val", TRUE, SetDarkness },
	{ "shadows",  "set shadows",  "val", TRUE, SetShadows },
	{ "saturation", "set saturation", "val", TRUE, SetSaturation },
	{ "gamma", "set gamma correction", "val where 0 < val", TRUE, SetGamma },
	{ "tone", "set tone curve", "tone curve name", TRUE, SetToneCurve },
	{ "tcPath", "set tone curve search path", "pathlist", TRUE, SetToneCurvePath },
	{ "8", "output 8 bits per color per pixel", "", FALSE, Set8 },
	{ "16", "output 16 bits per color per pixel", "", FALSE, Set16 },
	{ "xvmini", "generate XV thumbnail", "", FALSE, SetXVMini },
};

gboolean
ProcessCmdLineOptions (int count, char *arg[])
{
	int	i;

	while (count > 0) {
		if (arg[0][0] != '-')
			return FALSE;
		for (i = 0; i < sizeof(htable)/sizeof(htable[0]); i++) {
			if (strcmp (htable[i].param, &arg[0][1]) == 0) {
				if (htable[i].has_arg) {
					if (count == 1)
						return FALSE;
					else if (!(*htable[i].func) (arg[1]))
						return FALSE;
					arg += 2;
					count -= 2;
				}
				else {
					if (!(*htable[i].func) (NULL))
						return FALSE;
					arg ++;
					count --;
				}
				break;
			}
		}
		if (i == sizeof(htable)/sizeof(htable[0]))
			return FALSE;
	}
	return TRUE;
}

gboolean
SetParamFileOption (char *option, char *value)
{
	int	i;

	if (!forMe)
		return TRUE;
	for (i = 0; i < sizeof(htable)/sizeof(htable[0]); i++) {
		if (strcmp (htable[i].param, option) == 0) {
			if (htable[i].has_arg) {
				if (value == NULL || *value == '\0')
					return FALSE;
				else if (!(*htable[i].func) (value))
					return FALSE;
			}
			else {
				if (!(*htable[i].func) (NULL))
					return FALSE;
			}
			return TRUE;
		}
	}
	return FALSE;
}

void
ProcessOptionFile (char *filename, char *sect)
{
	if (access (filename, R_OK) != 0)
		return;
	currentSection = NULL;
	mySection = strdup (sect);
	forMe = TRUE;
	if (!pm_process (filename, do_section, SetParamFileOption))
		OptionFileUsage();
	free (mySection);
	if (currentSection) free (currentSection);
}


void
PrintOptions (FILE *f)
{
	int i;
	for (i = 0; i < sizeof(htable)/sizeof(htable[0]); i++) {
		fprintf (f, "\t-%s %s // %s\n", htable[i].param,
			htable[i].arghelp, htable[i].paramhelp);
	}
}

void
PrintInitFileOptions (FILE *f)
{
	int i;
	for (i = 0; i < sizeof(htable)/sizeof(htable[0]); i++) {
		fprintf (f, "    %s=%s // %s\n", htable[i].param,
			htable[i].arghelp, htable[i].paramhelp);
	}
}
