/********************************************************************
 *                                                                  *
 * pam_smartcard.c                                                  *
 *                                                                  *
 ********************************************************************
 *                                                                  *
 * Authors:                                                         *
 *   Mario Strasser       <mast@gmx.net>                            *
 *   David Corcoran       <corcoran@linuxnet.com>                   *
 *   Eirik A. Herskedal   <ehersked@cs.purdue.edu>                  *
 *   Bruce Barnett        <muscle040302@grymoire.com>
 *                                                                  *
 ********************************************************************
 *                                                                  *
 * The authentication module.                                       *
 *                                                                  *
 ********************************************************************/


/* We have to make this definitions before we include the pam header file! */

#define PAM_SM_AUTH
#define PAM_SM_ACCOUNT
#define PAM_SM_SESSION
#define PAM_SM_PASSMSCULong32

#include <stdlib.h>
#include <security/pam_modules.h>
#include <syslog.h>
#include <musclecard.h>
#include "cardtools.h"
#include "certutils.h"
#include <openssl/evp.h>
#include <string.h>
#include <preferences.h>
#include <x509v3.h>
#include <err.h>

#define PRV_KEY_NUMBER               3
#define LOGNAME "PAM-SmartCard"     /* Name for logfile entries */
#define RAND_SIZE 128               /* Size of the random value */
#define MAX_USERNAME_LEN 128
#define DEBUG 1

/**
 * This struct contains secure data and should be overwritten
 * before releasing!
 */
struct secure_data {
  char rand[RAND_SIZE];
  char cipher[RAND_SIZE];
  char plain[RAND_SIZE];
  char pin[PIN_SIZE+1];
  char new1[PIN_SIZE+1];
  char new2[PIN_SIZE+1];
  char cardholder[257];
  const char *user;
};

void pam_release_data(struct secure_data *data)
{
  /* overwrite data */
  memset(data, 0, sizeof(struct secure_data));
  /* release it */
  free(data);
}

int pam_get_pin(pam_handle_t *pamh, char *pin, char* text, int oitem, int nitem)
{
  int rv;
  const char *old_pin;
  struct pam_conv *conv;
  struct pam_message msg;
  struct pam_message *(msgp[1]) = { &msg };
  struct pam_response *resp;

  /* use stored password if variable oitem is set */
  if ((oitem == PAM_AUTHTOK) || (oitem == PAM_OLDAUTHTOK)) {
    /* try to get stored item */
    rv = pam_get_item(pamh, oitem, (const void**)&old_pin);
    if (rv != PAM_SUCCESS) return rv;
    if (old_pin != NULL) {
      strncpy(pin, old_pin, PIN_SIZE);
      pin[PIN_SIZE] = 0;
      return PAM_SUCCESS;
    }
  }

  /* ask the user for the password if variable text ist set */
  if (text != NULL) {
    msg.msg_style = PAM_PROMPT_ECHO_OFF;
    msg.msg = text;
    rv = pam_get_item(pamh, PAM_CONV, (const void**)&conv);
    if (rv != PAM_SUCCESS) return rv;
    if ((conv == NULL) || (conv->conv == NULL)) return PAM_CRED_INSUFFICIENT;
    rv = conv->conv(1, (const struct pam_message **)msgp, &resp, conv->appdata_ptr);
    if (rv != PAM_SUCCESS) return rv;
    if ((resp == NULL) || (resp[0].resp == NULL)) return PAM_CRED_INSUFFICIENT;
    strncpy(pin, resp[0].resp, PIN_SIZE);
    pin[PIN_SIZE] = 0;
    /* overwrite memory and release it */
    memset(resp[0].resp, 0, strlen(resp[0].resp));
    free(&resp[0]);

    /* save password if variable nitem is set */
    if ((nitem == PAM_AUTHTOK) || (nitem == PAM_OLDAUTHTOK)) {
      rv = pam_set_item(pamh, nitem, pin);
      if (rv != PAM_SUCCESS) return rv;
    }

    return PAM_SUCCESS;
  }

  return PAM_CRED_INSUFFICIENT;
}

int pam_show_message(pam_handle_t *pamh, char* text)
{
  int rv;
  struct pam_conv *conv;
  struct pam_message msg;
  struct pam_message *(msgp[1]) = { &msg };
  struct pam_response *resp;

  /* ask user for the password */
  msg.msg_style = PAM_TEXT_INFO;
  msg.msg = text;

  rv = pam_get_item(pamh, PAM_CONV, (const void**)&conv);
  if (rv != PAM_SUCCESS) return rv;
  if ((conv == NULL) || (conv->conv == NULL)) return PAM_CRED_INSUFFICIENT;
  rv = conv->conv(1, (const struct pam_message **)msgp, &resp, conv->appdata_ptr);
  if (rv != PAM_SUCCESS) return rv;
  if (resp == NULL) return PAM_CRED_INSUFFICIENT;
  free(&resp[0]);

  return PAM_SUCCESS;
}

int readRootCert(X509 **userCert, MSCTokenConnection pConnection, 
		 struct secure_data *sd) {
  EVP_PKEY *pubkey;
  int index, rv;
  X509 *tmpCert, *rootCert;
  STACK *emlst;
  char *userid;  
  char *p;

  rv = getCardCert(pConnection, &tmpCert);

  if (rv == -1) {
    syslog(LOG_ERR, "cannot read certificate from smartcard");
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  emlst = (STACK *) X509_get1_email(tmpCert);

  for (index = 0, p = sk_value(emlst,0); 
       *p && index < MAX_USERNAME_LEN;
       p++,index++)
    if (*p == '@')
      break;

  userid = malloc(sizeof(unsigned char) * index + 1 );
  strncpy(userid, sk_value(emlst, 0), index);
  X509_email_free(emlst);

  if (index >= MAX_USERNAME_LEN || index < 1) {
    syslog(LOG_ERR, "error getting email address from certificate");
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  userid[index] = '\0';

  if (pr.debug)
    printf("UID: %s\tUID2: %s\n", userid, sd->user);


  if (strncmp(userid, sd->user, index) != 0) {
    syslog(LOG_ERR, "e-mail username does not correspond to account (UID: %s, sd UID: %s)", userid, sd->user);
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }  

  free(userid);

  rv = checkCert(tmpCert);

  if (rv == -1) {
    syslog(LOG_ERR, "user certificate expired or revoked");
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  rv = getFileCert(pr.rootcacert, &rootCert);

  if (rv == -1) {
    /* Try to read a PEM cert */
    rv = getFileCertPEM(pr.rootcacert, &rootCert);
  }

  if (rv == -1) {
    syslog(LOG_ERR, "cannot find root certificate in %s", pr.rootcacert);
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  rv = checkCert(rootCert);

  if (rv == -1) {
    syslog(LOG_ERR, "root certificate expired or revoked");
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  rv = getPublicKey(rootCert, &pubkey);

  if (rv == -1) {
    syslog(LOG_ERR, "cannot read public key file from root certificate");
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }
 
   //FIXME: loads a lot more than needed
   OpenSSL_add_all_algorithms();
 
   index = X509_verify(tmpCert, pubkey);
 
   if (pr.debug)
     printf("Verification returns %d\n", index);
 
   if (index <= 0) {
    syslog(LOG_ERR, "user certificate does not have a valid signature");
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  *userCert = tmpCert;

  X509_free(rootCert);

  return 0;
}

int readUserPubKey(EVP_PKEY **userCert, MSCTokenConnection pConnection, 
		 struct secure_data *sd) {
  int rv = 0;
  EVP_PKEY *tmpCert;
  X509 *tmpCert1;
  char homeFile[200];
  /* tries to get a public key information from the user's public key certificate */

  /* Is userpath defined? If so, use it as a way to 'find' someone's
   home directory (ignoring the value in the password file). 
   I'm using this because the existence of the
   specified file overrides the end user's default certificate in their 
   home directory  - so userpath is defined for debugging only */

  if (strlen(pr.userpath)>0) {
	snprintf(homeFile, sizeof(homeFile), "%s%s/.muscle/%s", pr.userpath, sd->user, pr.certname);  
  } else {
	/* It's not defined. We have to determine the user's home directory. */
	struct passwd *pw;
	if ((pw=getpwnam(sd->user)) == NULL) {
	  syslog(LOG_ERR, "su attempt to non-existing user: %s", sd->user);
	  return -1;
	} else
	  snprintf(homeFile, sizeof(homeFile), "%s/.muscle/%s", pw->pw_dir, pr.certname);  
  }
  if (util_CheckFile(homeFile,(char *)(sd->user))) {
	syslog(LOG_ERR, "Unsafe permissions on user certificate, file: %s: user: %s", homeFile, sd->user);
	return -1;
  }
  rv = getFileCert(homeFile, &tmpCert1);

  if (rv == -1) {
    /* Didn't work - now try public key cert */
    /* for UserAuth - this could also be a public key Certificate, not an X509 (signed) */

	/* This makes the assumption the user created the certificate
	   without an outside authority to sign it. */

    rv = getPubKeyFromFile(homeFile, &tmpCert);
	if (pr.debug)
		syslog(LOG_INFO, "user certificate successfully read from %s", homeFile);
  } else {
	rv = checkCert(tmpCert1);

	if (rv == -1) {
	  syslog(LOG_ERR, "user certificate expired or revoked");
	  return -1;
	}
	rv = getPublicKey(tmpCert1, &tmpCert);
  }	
  
  if (rv == -1) {
    syslog(LOG_ERR, "cannot read certificate from %s", homeFile);
	return -1;
  }


  *userCert = tmpCert;

  return 0;
}

PAM_EXTERN int pam_sm_authenticate(pam_handle_t *pamh, int flags, int argc,
                                   const char **argv)
{
  MSCTokenConnection pConnection;
  MSCCryptInit cryptInit;
  MSCULong32 cryptSize;
  //MSCUChar8 pinNumber;
  //MSCKeyInfo keyInfo;

  struct secure_data *sd;
  int i, rv, reader, result;
  int try_first_pass, use_first_pass;
  char error[150];
  //unsigned char nameBuf[100];
  X509 *userCert = 0;
  EVP_PKEY *pubkey;

  reader = 0;
  try_first_pass = 0;
  use_first_pass = 0;

  ERR_load_crypto_strings();
  util_ReadPreferences();

  /* open log */
  openlog(LOGNAME, LOG_CONS | LOG_PID, LOG_AUTHPRIV);

  /* init data */
  sd = malloc(sizeof(struct secure_data));
  if (sd == NULL) {
    syslog(LOG_CRIT, "not enough free memory");
    return PAM_AUTHINFO_UNAVAIL;
  }

  /* init pcsc */
  rv = pcsc_init(&pConnection, reader);
  if (rv != MSC_SUCCESS) {
    syslog(LOG_ERR, "musclecard error during pcsc_init: %s", msc_error(rv));
    pam_release_data(sd);
    /* if we return PAM_AUTHINFO_UNAVAIL, then we cannot use pam "auth
       sufficient" That is, if the token is unavailable, will you
       allow a password to be used. If we return PAM_AUTHINFO_UNAVAIL,
       we cannot fall back on a password. Therefore, return PAM_ERR,
       allowing password to be sufficient.

    return PAM_AUTHINFO_UNAVAIL;
    */
    return PAM_AUTH_ERR; 

  }

  if (pr.debug) printf("Welcome to pam_musclecard.so verification Module\n");

  /* get user name */
  rv = pam_get_user(pamh, &sd->user, NULL);
  if (rv != PAM_SUCCESS) {
    syslog(LOG_ERR, "cannot get username: %s", pam_strerror(pamh, rv));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_USER_UNKNOWN;
  }

  if (pr.debug) printf("User = %s\n", sd->user);
  /* get password */
  if (use_first_pass) {
    rv = pam_get_pin(pamh, sd->pin, NULL, PAM_AUTHTOK, 0);

  } else if (try_first_pass) {
    rv = pam_get_pin(pamh, sd->pin, "Please enter pin: ", PAM_AUTHTOK, 
		     PAM_AUTHTOK);
  } else {
    rv = pam_get_pin(pamh, sd->pin, "Please enter pin: ", 0, PAM_AUTHTOK);
  }
  if (rv != PAM_SUCCESS) {
    syslog(LOG_ERR, "cannot get pin: %s", pam_strerror(pamh, rv));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  /*  if (pr.debug) printf("pin  = %s\n\n", sd->pin); */

  /* get random value */
  rv = getRandom(sd->rand, RAND_SIZE);
  if (rv == -1) {
    syslog(LOG_ERR, "cannot read random number: %s", strerror(errno));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  if (pr.debug) {
    printf("Random value = ");
    for (i=0; i < RAND_SIZE; i++) {
      rv = (unsigned char)sd->rand[i];
      printf("%02x", rv);
    }
    printf("\n\n");
  }

  /* We must get the ACL for key #3 (or whatever) to determine which pin must
     be verified                                                */

  /* verify chv1 */

  rv = MSCVerifyPIN(&pConnection, pr.pinnumber, (unsigned char *)sd->pin,
	PIN_SIZE);
  
  if (rv != MSC_SUCCESS) {
    if (pr.debug) printf("Invalid PIN Entered\n");
    syslog(LOG_ERR, "musclecard error during Verify PIN: %s", msc_error(rv));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  cryptInit.keyNum          = pr.certnumber;
  cryptInit.cipherMode      = MSC_MODE_RSA_NOPAD;
  cryptInit.cipherDirection = MSC_DIR_ENCRYPT;
  cryptInit.optParams       = NULL;
  cryptInit.optParamsSize   = 0;

  cryptSize = RAND_SIZE;

  rv = MSCComputeCrypt(&pConnection, &cryptInit, (MSCPUChar8)sd->rand,
		       RAND_SIZE, (MSCPUChar8)sd->cipher, &cryptSize);

  if (rv == MSC_INCORRECT_P1) {
    syslog(LOG_ERR, "musclecard error during Compute Crypt,  incorrect P1 value, certificate #%d: %s", pr.certnumber, msc_error(rv));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }	else if (rv != MSC_SUCCESS) {
    syslog(LOG_ERR, "musclecard error during Compute Crypt, certificate #%d, %s", pr.certnumber, msc_error(rv));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  if (pr.debug) {
    printf("Cipher value = ");
    for (i=0; i < RAND_SIZE; i++) {
      rv = (unsigned char)sd->cipher[i];
      printf("%02x", rv);
    }
    printf("\n\n");
  }

  /* get the certs for the user and root CA */

  if (pr.authmode == ROOTCERT) {
    rv = readRootCert(&userCert, pConnection, sd);
	if (rv != 0) {
	  pcsc_release(&pConnection);
	  pam_release_data(sd);
        return PAM_AUTHINFO_UNAVAIL;
	}
    rv = getPublicKey(userCert, &pubkey);
  } else {
    rv = readUserPubKey(&pubkey, pConnection, sd);
  }

  if (rv == -1) {
    syslog(LOG_ERR, "cannot read public key file from user certificate");
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  /* This section can be used to verify the user's certificate */
#ifdef DEBUG_SPECIAL1
  if (pr.debug) {
	syslog(LOG_INFO, "Public key file read from user certificate");
	printf("public exponent : %s\n", BN_bn2hex(pubkey->pkey.rsa->e));
	printf("public modulus : %s\n", BN_bn2hex(pubkey->pkey.rsa->n));
  }

  BIO *eout=NULL;
  eout = BIO_new(BIO_s_file());
  BIO_set_fp(eout,stdout,BIO_NOCLOSE);
  RSA_print(eout, pubkey->pkey.rsa, 0);

#endif

  rv = RSA_public_decrypt(128, (unsigned char *)sd->cipher,
	(unsigned char *)sd->plain, pubkey->pkey.rsa, RSA_NO_PADDING);  

  syslog(LOG_ERR, "CLEAR %02X %02X %02X\n", (unsigned char)sd->plain[0], 
	 (unsigned char)sd->plain[1], (unsigned char)sd->plain[0x79]);

  if (rv == -1) {
    syslog(LOG_ERR, "cannot decode random number");
    ERR_error_string(ERR_get_error(), error);
    if (pr.debug) printf("Error from openssl:\t %s\n", error);
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  if (pr.debug) {
    printf("Plain value  = ");
    for (i=0; i < RAND_SIZE; i++) {
      rv = (unsigned char)sd->plain[i];
      printf("%02x", rv);
    }
    printf("\n\n");
  }


  /* Here is another debug section. It asks the card to decrypy what it sent out.
   * This should be the same as the input.
   * Normally this isn't used - Bruce */

#ifdef DEBUG_SPECIAL
  if (pr.debug) {
	/* double check to make sure this works. We should be able to ask the card to decrypt this as well */

/* assume the next certificate is the matching public key - This may be a bas assumption. -Bruce */
	cryptInit.keyNum          = pr.certnumber + 1;  
	cryptInit.cipherMode      = MSC_MODE_RSA_NOPAD;
	cryptInit.cipherDirection = MSC_DIR_DECRYPT;
	cryptInit.optParams       = NULL;
	cryptInit.optParamsSize   = 0;
	cryptSize = RAND_SIZE;

	struct secure_data *sd2;
	sd2 = malloc(sizeof(struct secure_data));
	if (sd2 == NULL) {
	  syslog(LOG_CRIT, "not enough free memory to initialize sd2");
	  pam_release_data(sd);
	  return PAM_AUTHINFO_UNAVAIL;
	}
	memcpy(sd2->rand,sd->cipher,RAND_SIZE);
	rv = MSCComputeCrypt(&pConnection, &cryptInit, (MSCPUChar8)sd2->rand,
						 RAND_SIZE, (MSCPUChar8)sd2->cipher, &cryptSize);

	if (rv != MSC_SUCCESS) {
	  
	  syslog(LOG_ERR, "musclecard error during reverse Compute Crypt, certificate #%d: %s", cryptInit.keyNum, msc_error(rv));
	}
	printf("Undecoded value using key #%d  = ", cryptInit.keyNum);
	  for (i=0; i < RAND_SIZE; i++) {
		rv = (unsigned char)sd2->cipher[i];
		printf("%02X", rv);
	  }	
	  printf("\n\n");
  }
	printf("Compare two values size by side: \n");
	  for (i=0; i < 128; i+=8) {
		printf("\t%02X%02X%02X%02X %02X%02X%02X%02X == %02X%02X%02X%02X %02X%02X%02X%02X?\n",
			   (unsigned char)sd->rand[i],
			   (unsigned char)sd->rand[i+1],
			   (unsigned char)sd->rand[i+2],
			   (unsigned char)sd->rand[i+3],
			   (unsigned char)sd->rand[i+4],
			   (unsigned char)sd->rand[i+5],
			   (unsigned char)sd->rand[i+6],
			   (unsigned char)sd->rand[i+7],
			   (unsigned char)sd->plain[i],
			   (unsigned char)sd->plain[i+1],
			   (unsigned char)sd->plain[i+2],
			   (unsigned char)sd->plain[i+3],
			   (unsigned char)sd->plain[i+4],
			   (unsigned char)sd->plain[i+5],
			   (unsigned char)sd->plain[i+6],
			   (unsigned char)sd->plain[i+7]
			   );
	  }	

#endif

  /* compare original value and result from smartcard */
  if (memcmp(sd->rand, sd->plain, RAND_SIZE) == 0) {
	if (pr.debug)
	  printf("Challenge was Successfully met\n");
	result = PAM_SUCCESS;
  } else {
	if (pr.debug)
	  syslog(LOG_ERR, "musclecard challenge failed for user %s", sd->user);

	result = PAM_AUTH_ERR; 
  }  
  
  /* Release certificates */
  if (userCert)
	X509_free(userCert);

  /* release pcsc */
  pcsc_release(&pConnection);

  /* release data */
  pam_release_data(sd);

  return result;
}

PAM_EXTERN int pam_sm_setcred(pam_handle_t *pamh, int flags, int argc,
                              const char **argv)
{
  /* Actually we should return the same value as pam_sm_authenticate(). */
  /* But pam_[sg]et_data dosen't seem to work, so we return success     */
  return PAM_SUCCESS;
}

PAM_EXTERN int pam_sm_acct_mgmt(pam_handle_t *pamh, int flags, int argc,
                                const char **argv)
{
  openlog(LOGNAME, LOG_CONS | LOG_PID, LOG_AUTHPRIV);
  syslog(LOG_WARNING, "Function pm_sm_acct_mgmt() not implemented in this module");
  closelog();
  return PAM_SERVICE_ERR;
}

PAM_EXTERN int pam_sm_open_session(pam_handle_t *pamh, int flags, int argc,
                                   const char **argv)
{
  openlog(LOGNAME, LOG_CONS | LOG_PID, LOG_AUTHPRIV);
  syslog(LOG_WARNING, "Function pm_sm_open_session() not implemented in this module");
  closelog();
  return PAM_SERVICE_ERR;
}

PAM_EXTERN int pam_sm_close_session(pam_handle_t *pamh, int flags, int argc,
                                    const char **argv)
{
  openlog(LOGNAME, LOG_CONS | LOG_PID, LOG_AUTHPRIV);
  syslog(LOG_WARNING, "Function pm_sm_close_session() not implemented in this module");
  closelog();
  return PAM_SERVICE_ERR;
}

PAM_EXTERN int pam_sm_chauthtok(pam_handle_t *pamh, int flags, int argc,
                                const char **argv)
{

  MSCTokenConnection pConnection;
  struct secure_data *sd;
  int i, rv, reader, ok;
  int use_first_pass, use_authtok = 0;

  /* open log */
  openlog(LOGNAME, LOG_CONS | LOG_PID, LOG_AUTHPRIV);

  /* parse arguments */
  reader = 0;
  use_first_pass = 0;
  for (i = 0; i < argc; i++) {
    /* try to get driver number */
    if (sscanf(argv[i], "reader=%d", &reader) == 1) continue;
    /* test for use_first_pass */
    if (!strcmp(argv[i], "use_first_pass")) { use_first_pass = 1; continue; }
    /* test for use_first_pass */
    if (!strcmp(argv[i], "use_authtok")) { use_authtok = 1; continue; }
    /* if parameter is not recognized, log error message */
    syslog(LOG_ERR, "argument %s is not supported by this module", argv[i]);
  }

  /* do we only test if changing is possible? */
  if (flags & PAM_PRELIM_CHECK) {
    /* try to init pcsc */
    rv = pcsc_init(&pConnection, reader);
    if (rv != SCARD_S_SUCCESS) {
      syslog(LOG_ERR, "pc/sc error: %s", pcsc_stringify_error(rv));
      return PAM_TRY_AGAIN;
    }
    pcsc_release(&pConnection);
    return PAM_SUCCESS;
  }

  /* init data */
  sd = malloc(sizeof(struct secure_data));
  if (sd == NULL) {
    syslog(LOG_CRIT, "not enough free memory");
    return PAM_AUTHINFO_UNAVAIL;
  }

  /* init pcsc */
  rv = pcsc_init(&pConnection, reader);
  if (rv != SCARD_S_SUCCESS) {
    syslog(LOG_ERR, "pc/sc error: %s", pcsc_stringify_error(rv));
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  /* get user name */
  rv = pam_get_user(pamh, &sd->user, NULL);
  if (rv != PAM_SUCCESS) {
    syslog(LOG_ERR, "cannot get username: %s", pam_strerror(pamh, rv));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_USER_UNKNOWN;
  }

  /* get old password */
  if (use_first_pass) {
    rv = pam_get_pin(pamh, sd->pin, NULL, PAM_OLDAUTHTOK, 0);
    /* As some modules don't ask root for the old password, we do it here! */
    if ((rv != PAM_SUCCESS) && (getuid() == 0)) {
      rv = pam_get_pin(pamh, sd->pin, "Please enter old password: ", 0, PAM_OLDAUTHTOK);
    }
  } else {
    rv = pam_get_pin(pamh, sd->pin, "Please enter old password: ", 0, PAM_OLDAUTHTOK);
  }
  if (rv != PAM_SUCCESS) {
    syslog(LOG_ERR, "cannot get password: %s", pam_strerror(pamh, rv));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHINFO_UNAVAIL;
  }

  if (pr.debug)
    printf("old password  = %s\n\n", sd->pin);

  ok = 0;
  while (!ok) {
    /* get new passwords */
    if (use_authtok) {
      rv = pam_get_pin(pamh, sd->new1, NULL, PAM_AUTHTOK, 0);
      rv = pam_get_pin(pamh, sd->new2, NULL, PAM_AUTHTOK, 0);
    } else {
      rv = pam_get_pin(pamh, sd->new1, "Please enter new password: ", 0, 0);
      rv = pam_get_pin(pamh, sd->new2, "Please re-enter new password: ", 0, PAM_AUTHTOK);
    }
    if (rv != PAM_SUCCESS) {
      syslog(LOG_ERR, "cannot get password: %s", pam_strerror(pamh, rv));
      pcsc_release(&pConnection);
      pam_release_data(sd);
      return PAM_AUTHINFO_UNAVAIL;
    }

    if (pr.debug)
      printf("new passwords = %s and %s\n\n", sd->new1, sd->new2);

    /* compare new passwords */
    if (strncmp(sd->new1, sd->new2, PIN_SIZE)) {
      if (!(flags & PAM_SILENT))
        pam_show_message(pamh, "Sorry, passwords do not match.");
    } else {
      ok = 1;
    }
  }

  /* try to change pin */
  rv = MSCChangePIN(&pConnection, 1, (unsigned char *)sd->pin, PIN_SIZE,
	(unsigned char *)sd->new1, PIN_SIZE);

  if (rv != MSC_SUCCESS) {
    syslog(LOG_ERR, "pc/sc error: %s", pcsc_stringify_error(rv));
    pcsc_release(&pConnection);
    pam_release_data(sd);
    return PAM_AUTHTOK_ERR;
  }

  /* release pcsc */
  pcsc_release(&pConnection);

  /* release data */
  pam_release_data(sd);

  /* everything was ok */
  return PAM_SUCCESS;
}

#ifdef PAM_STATIC

/* static module data */
struct pam_module _pam_smartcard_modstruct = {
    "pam_smartcard",
    pam_sm_authenticate,
    pam_sm_setcred,
    pam_sm_acct_mgmt,
    pam_sm_open_session,
    pam_sm_close_session,
    pam_sm_chauthtok
};

#endif


/*** end of file ***********************************************************/
