/********************************************************************
 *                                                                  *
 * File: preferences.c                                              *
 *                                                                  *
 ********************************************************************
 *                                                                  *
 * Authors:                                                         *
 *   Chris Osgod          <oznet@mac.com>                           *
 *   Eirik A. Herskedal   <ehersked@cs.purdue.edu>                  *
 *   Bruce Barnett        <muscle040302@grymoire.com>
 *                                                                  *
 ********************************************************************
 *                                                                  *
 * Parses the configuration file.                                   *
 * This file is a modification of a parser originally written       *
 * by Chris Osgod (oznet@mac.com).                                  *
 *                                                                  *
 ********************************************************************/

#include <stdio.h>
#include <syslog.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include "preferences.h"

/* The third parameter was /home, but this overrides the default value
   of the home directory specified by the passwd file */

struct preferences pr = {
  DEBUGOFF,
  0,
  1,
  "",   /* was "/home/" */
  "user.cert",
  "/etc/musclepam/root.cert",
  "",
  "",
  ROOTCERT
};

/******************************************************************************
** Function: util_ParsePreference
**
** Parse a preference item from a line of text.  The text should be null
** terminated and buf_size keeps overruns from happening.
**
** If a preference item is successfully parsed then it is stored in the
** st.prefs settings.
**
** This whole function is fairly verbose and could be broken into smaller
** pieces to handle things like "get true/false pref" but at least this is
** very straightforward.
**
** Fixme: put this in p11x_prefs.c
**
** Parameters:
**  buf      - Null terminated text buffer
**  buf_size - Size of buffer
**
** Returns:
**  none
******************************************************************************/
void util_ParsePreference(char *buf, int buf_size)
{
  char sep[] = "=\t\r\n ";
  char *token;

  /* We will be using many unsafe string functions so force a NULL at the */
  /* end of the buffer to protect ourselves                               */
  buf[buf_size - 1] = 0;
  token = (char *) strchr(buf, '#');

  if (token)
    *token = 0;

  token = (char *) strtok(buf, sep);
  if (token)
    {
      if (!strcasecmp("Debug", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"Debug\" failed");
	  else
            {
	      if (!strcasecmp("OFF", token))
		pr.debug = DEBUGOFF;
	      else
		pr.debug = DEBUGON;
            }
        }
      else if (!strcasecmp("CertNumber", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"CertNumber\" failed");
	  else
            {
	      pr.certnumber = (int) (*token - '0');
            }
        }
      else if (!strcasecmp("PinNumber", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"PinNumber\" failed");
	  else
            {
	      pr.pinnumber = (int) (*token - '0');
            }
        }
      else if (!strcasecmp("CertName", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"CertName\" failed");
	  else
            {
	      strncpy(pr.certname, token, 200);
            }
        }
      else if (!strcasecmp("UserPath", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"UserPath\" failed");
	  else
            {
	      strncpy(pr.userpath, token, 200);
            }
        }
      else if (!strcasecmp("RootCACert", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"RootCACert\" failed");
	  else
            {
	      strncpy(pr.rootcacert, token, 200);
            }
        }
      else if (!strcasecmp("LDAPHost", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"LDAPHost\" failed");
	  else
            {
	      strncpy(pr.ldaphost, token, 200);
            }
        }
      else if (!strcasecmp("LDAPPath", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"LDAPPath\" failed");
	  else
            {
	      strncpy(pr.ldappath, token, 200);
            }
        }
      else if (!strcasecmp("AuthMode", token))
        {
	  token = (char *) strtok(0, sep);
	  if (!token)
	    syslog(LOG_ERR, "Config option \"AuthMode\" failed");
	  else
            {
	      if (!strcasecmp("UserCert", token))
		pr.authmode = USERCERT;
	      else if (!strcasecmp("RootCert", token))
		pr.authmode = ROOTCERT;
            }
        }
    }
}

/******************************************************************************
** Function: util_ReadPreferences
**
** Gets preferences, if available.  On UNIX, looks for .pkcs11rc
** in the $HOME directory, or root directory if $HOME is not
** defined.  Having a preferences file is optional and it is assumed
** that most of the time users will not have one unless debug/logging
** or other special settings are required.
**
** Parameters:
**  none
**
** Returns:
**  none
******************************************************************************/

int util_ReadPreferences()
{
  int rv = 0;
  FILE *fp;
  char rcfilename[] = "/etc/musclepam/pam-muscle.conf";
  char buf[1024];

  if (pr.debug)
	syslog(LOG_INFO, "Reading the preferences from file '%s'", rcfilename);

  /* unsafe preference file? */
  if (util_CheckFile(rcfilename, "root") != 0)
	return -1;

  fp = fopen(rcfilename, "rb");
  if (fp)
    {
      while (fgets(buf, sizeof(buf), fp))
	util_ParsePreference(buf, sizeof(buf));
      fclose(fp);
    }
  return rv;
}

/******************************************************************************
** Function: util_CheckFile
**
** Checks to make sure the file is not modifyable by someone other
** than the owner This utility is given the complete name for the file
** and recursively walks up the direwctory tree until it reaches the
** root directory.
**
**
** Parameters:
**  file (string)
**   user (string)
**
** Returns:
**  zero if no problem
**  -1 if there is a security risk
**  -2 if the file does not exist
**
******************************************************************************/

int util_CheckFile (char *file, char *user)
{
	char path[1024];
	int i;
	char *l;

	if (strstr(file, "..")) {
	  /* This would probably never happen, and the owner of the
		 /etc/musclepam/pam-muscle.conf file has to put it
		 there. . But it's better to be paranoid and safe than sorry */
	  syslog(LOG_ERR, "File '%s' contains the string '..'  - unsafe place to put configuration file", file);
	  return -1;
	}

	if (file[0] != '/') {
	  syslog(LOG_ERR, "File '%s' is relative. Must use absolute path  - unsafe place to put configuration file", file);
	  return -1;
	}

	/* Now - check the entire file */
	if ((i = util_CheckFileComponent(file,user)) < 0)
		return i;

	/* looks okay - now check all of the paths */
	strncpy(path, file, sizeof(path));

	while ((l = strrchr(path,'/')) != NULL) {
	  /* chop off all characters after the '/' */
	  *l = '\0';
	  if (strlen(path) > 0)
		if ((i = util_CheckFileComponent(path,user)) < 0)
			return i;
	}

	if (pr.debug)
	  syslog(LOG_INFO, "File '%s' and user %s look okay", file, user);

	return 0;
}

/******************************************************************************
** Function: util_CheckFileComponent
**
** Checks to make sure the file component (either a single file or
** directory) is not modifyable by someone other than the owner
**
** Parameters:
**  file (string)
**   user (string)
**
** Returns:
**  zero if no problem
**  -1 if there is a security risk
**  -2 if the file or directory does not exist
**
******************************************************************************/

int util_CheckFileComponent (char *file, char *user)
{
    struct stat buf;
  	struct passwd *pw;
	uid_t uid;

	/* Check if the file is a symbolic link */
	if (lstat(file, &buf)) {
	  /* if (pr.debug) printf("File does not exist %s - done\n", file); */
	  return -2; /* doesn't exist */
	}

	if (S_ISLNK(buf.st_mode)) {
	  syslog(LOG_ERR, "File '%s' is a symbolic link - unsafe place to put configuration file", file);
	  return -1;
	}

	/* Check the real file */
	if (stat(file,&buf)) {
	  /* if (pr.debug) printf("File does not exist (2) %s - done\n", file); */
	  return -2; /* I can't check it, so it's okay */
	}

	/* Find the user's UID number */
	if ((pw = getpwnam(user)) == NULL) {
	  syslog(LOG_ERR, "User '%s' does not exist", user);
	  return -1;
	}

	uid = pw->pw_uid;

	if ((buf.st_uid != uid) && (buf.st_uid != 0)) {
	  syslog(LOG_ERR, "File '%s' is  owned by UID %d, and should be owned by %d (%s) - unsafe operation", file, buf.st_uid, uid, user);
	  return -1;
	}

	if ((buf.st_mode & 0022) != 0) {
	  /* if (pr.debug) printf("File mode is BAD,  %o vs. %o\n", buf.st_mode, (buf.st_mode&022)); */
	  syslog(LOG_ERR, "File '%s' is group or world writable - may be unsafe operation", file);
	  return -1;
	}

	return 0;
}

