# Music Applet
# Copyright (C) 2007 Paul Kuliniewicz <paul@kuliniewicz.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.

import musicapplet.defs
import musicapplet.player

import gobject

#
# HEYME: Use MPRIS to have better control????
#


INFO = {
        "name": "BMPx",
        "version": musicapplet.defs.VERSION,
        "icon-name": "bmpx",
        "author": "Paul Kuliniewicz <paul@kuliniewicz.org>",
        "copyright": "(C) 2007 Paul Kuliniewicz",
        "website": "http://www.kuliniewicz.org/music-applet/",
}


class BMPxPlugin (musicapplet.player.DBusPlugin):
    """
    Music Applet plugin to interface with BMPx.
    """

    def __init__ (self, conf):
        musicapplet.player.DBusPlugin.__init__ (self, conf,
                                                INFO["name"], INFO["icon-name"],
                                                "org.beepmediaplayer.bmp")

        self.min_rating = 0.0
        self.max_rating = 5.0

        self.__handlers = []
        self.__poll_source = None
        self.__core = None


    def _connect (self):
        if self.__core is not None:
            return

        import dbus
        bus = dbus.SessionBus ()

        proxy = bus.get_object ("org.beepmediaplayer.bmp", "/Core")
        self.__core = dbus.Interface (proxy, "org.beepmediaplayer.bmp")

        self.__handlers = [
            self.__core.connect_to_signal ("StartupComplete", self.__startup_complete_cb),
        ]

        self.connected = True

        # FIXME: Just waiting for StartupComplete won't work if BMPx is already running,
        #        but calling a function on it prematurely seems to cause it to crash?


    def _disconnect (self):
        for handler in self.__handlers:
            handler.remove ()
        self.__handlers = []

        self.__core = None

        if self.__poll_source is not None:
            gobject.source_remove (self.__poll_source)
            self.__poll_source = None

        self._set_no_song ()
        self.connected = False


    def toggle_playback (self):
        # Without knowing whether BMPx is paused or not, there's no way to know
        # whether to call Play or Pause.
        print "DEBUG: BMPx: cannot toggle playback"
        pass


    def previous (self):
        self.__core.GoPrev (reply_handler=lambda: None,
                            error_handler=self._report_dbus_error)


    def next (self):
        self.__core.GoNext (reply_handler=lambda: None,
                            error_handler=self._report_dbus_error)


    # BMPx does rating, but provides no way to set them.
    #
    # def rate_song (self, rating):
    #     pass


    def __startup_complete_cb (self):
        if self.__poll_source is None:
            self.__poll ()
            self.__poll_source = gobject.timeout_add (1000, self.__poll)


    def __poll (self):
        self.__core.GetCurrentSource (reply_handler=self.__current_source_cb,
                                      error_handler=self._report_dbus_error)

        # There's apparently no way to tell the current elapsed time, or even
        # whether it's playing or not!

        return True


    def __current_source_cb (self, source):
        if source >= 0:
            self.__core.GetMetadataFromSource (source,
                                               reply_handler=self.__metadata_from_source_cb,
                                               error_handler=self._report_dbus_error)
        else:
            self._set_no_song ()


    def __metadata_from_source_cb (self, metadata):
        self.title = metadata["title"]
        self.artist = metadata["artist"]
        self.album = metadata["album"]
        self.duration = metadata["time"]
        self.rating = metadata["rating"]


def create_instance (conf):
    return BMPxPlugin (conf)
