 /*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System.Xml;
using System.IO;

namespace Nemerle.Xml
{
    /**
     *  class used to generate xml templates using xml files
     */
    public class XmlGenerator : IGenerator
    {

        /* -- CONSTRUCTORS ------------------------------------------------------------------*/
        
        /**
         *  constructs xml template using xml file
         */
        public this(xml_file_name : string)
        {
            doc = XmlDocument();
            def reader = XmlTextReader (xml_file_name);
            doc.Load (reader)
        }

        public this(doc : XmlDocument)
        {
            this.doc = doc;
        }
        
        /* -- PUBLIC METHODS ------------------------------------------------------------------*/
        
        /**
         *  generate xml template
         */
        public Generate () : XmlTemplate
        {
            XmlTemplate(doc);
        }

        /* -- PRIVATE FIELDS ------------------------------------------------------------------*/
        private doc : XmlDocument; 
    }

    /**
     *  Class used to generate xml directory listing 
     */
    public class DirectoryGenerator : IGenerator
    {
        /* -- CONSTRUCTORS --------------------------------------------------------------------- */
        
        /**
         *  generates xml directory listing
         */
        public this (name : string)
        {
            def add_attributes(node : XmlElement, di : FileSystemInfo, size : int) : XmlElement
            {
                def attr1 = doc.CreateAttribute("name");
                def attr2 = doc.CreateAttribute("LastAccessTime");
                def attr3 = doc.CreateAttribute("size");
                
                attr1.Value = di.Name;
                attr2.Value = di.LastAccessTime.ToString();
                attr3.Value = size.ToString();
                
                ignore(node.SetAttributeNode(attr1));
                ignore(node.SetAttributeNode(attr2));
                ignore(node.SetAttributeNode(attr3));

                node
            }
            
            doc = null;
            
            if(Directory.Exists(name))
            {
                def di = DirectoryInfo(name);
                
                doc = XmlDocument();
                doc.LoadXml("<root/>");

                def root = (doc.SelectSingleNode("root") :> XmlElement);

                ignore(add_attributes(root,di,0));
                
                def dirs = di.GetDirectories();
                foreach (di : DirectoryInfo in dirs) 
                {
                    def dir = doc.CreateElement("directory");
                    ignore(add_attributes(dir,di,0));
                    ignore(root.AppendChild(dir));
                }

                def files = di.GetFiles();
                foreach (fi : FileInfo in files) 
                {
                    def file = doc.CreateElement("file");
                    ignore(add_attributes(file,fi,(fi.Length :> int)));
                    ignore(root.AppendChild(file));
                }

                this.doc = doc;
            }
            else when(File.Exists(name))
                      {
                          def fi = FileInfo(name);
                
                          doc = XmlDocument();
                          doc.LoadXml("<file/>");
                          
                          def root = (doc.SelectSingleNode("file") :> XmlElement);
                          ignore(add_attributes(root,fi,(fi.Length :> int)));

                          this.doc = doc;
                      }
                
        }

        /* -- PUBLIC METHODS ------------------------------------------------------------------*/

        /**
         *  generate xml template
         */
        public Generate () : XmlTemplate
        {
            XmlTemplate(doc)
        }

        /* -- PRIVATE FIELDS ------------------------------------------------------------------*/
        private doc : XmlDocument;
    }
}