/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.scripting.php.dbginterface.models;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.ref.WeakReference;
import java.util.*;
import javax.security.auth.RefreshFailedException;
import javax.security.auth.Refreshable;
import javax.swing.JToolTip;
import org.netbeans.api.debugger.Watch;
import org.netbeans.api.debugger.DebuggerManager;
import org.netbeans.api.debugger.DebuggerManagerAdapter;
import org.netbeans.modules.scripting.php.dbginterface.DbgDebuggerImpl;
import org.netbeans.modules.scripting.php.dbginterface.api.VariableNode;
import org.netbeans.spi.debugger.ContextProvider;
import org.netbeans.spi.debugger.ui.Constants;
import org.netbeans.spi.viewmodel.ModelEvent;
import org.netbeans.spi.viewmodel.TreeModel;
import org.netbeans.spi.viewmodel.ModelListener;
import org.netbeans.spi.viewmodel.NodeModel;
import org.netbeans.spi.viewmodel.TableModel;
import org.netbeans.spi.viewmodel.UnknownTypeException;
import org.openide.util.RequestProcessor;


/**
 * @author Jan Jancura
 * @author Peter Williams (originally taken from WatchesModel class in JPDA)
 */
public class WatchesModel implements TreeModel, NodeModel, TableModel {
    private DbgDebuggerImpl debugger;
    private Listener listener;
    private Vector<ModelListener> listeners = new Vector<ModelListener>();
    private ContextProvider lookupProvider;
    private DebugFrame currentFrame;

    // Watch to Expression or Exception
    private Map<Watch, ScriptWatchEvaluating> watchToValue =
            new WeakHashMap<Watch, ScriptWatchEvaluating>();

    public WatchesModel(ContextProvider lookupProvider) {
        this.debugger = (DbgDebuggerImpl)lookupProvider.lookupFirst(null, DbgDebuggerImpl.class);
        this.lookupProvider = lookupProvider;
    }

    // ------------------------------------------------------------------------
    // TreeModel implementation
    // ------------------------------------------------------------------------
    /**
     * Returns the root node of the tree or null, if the tree is empty.
     *
     * @return the root node of the tree or null
     */
    public Object getRoot() {
        return ROOT;
    }

    /**
     *
     * @return watches contained in this group of watches
     */
    public Object [] getChildren(Object parent, int from, int to) throws UnknownTypeException {
        if(parent == ROOT) {
            // 1) get watches
            Watch [] ws = DebuggerManager.getDebuggerManager().getWatches();
            to = Math.min(ws.length, to);
            from = Math.min(ws.length, from);
            Watch [] fws = new Watch [to - from];
            System.arraycopy(ws, from, fws, 0, to - from);

            // 2) create ScriptWatchEvaluating for Watches
            int i, k = fws.length;
            ScriptWatchEvaluating[] jws = new ScriptWatchEvaluating[k];
            for(i = 0; i < k; i++) {
                ScriptWatchEvaluating jw = watchToValue.get(fws[i]);
                if(jw == null) {
                    jw = new ScriptWatchEvaluating(this, fws[i], debugger);
                    watchToValue.put(fws[i], jw);
                }
                jws[i] = jw;

                // The actual expressions are computed on demand in ScriptWatchEvaluating
            }

            if(listener == null) {
                listener = new Listener(this, debugger);
            }
            return jws;
        }
        else if(parent instanceof VariableNode) {
            return getLocalsTreeModel().getChildren(parent, from, to);
        }
        return getLocalsTreeModel().getChildren(parent, from, to);
    }

    /**
     * Returns number of children for given node.
     *
     * @param   node the parent node
     * @throws  UnknownTypeException if this TreeModel implementation is not
     *          able to resolve children for given node type
     *
     * @return  true if node is leaf
     */
    public int getChildrenCount(Object node) throws UnknownTypeException {
        if(node == ROOT) {
            if(listener == null) {
                listener = new Listener(this, debugger);
            }
            // Performance, see issue #59058.
            return Integer.MAX_VALUE; // DebuggerManager.getDebuggerManager().getWatches().length;
        }
        else if(node instanceof VariableNode) {
            return getLocalsTreeModel().getChildrenCount(node);
        }

        return getLocalsTreeModel().getChildrenCount(node);
    }

    public boolean isLeaf(Object node) throws UnknownTypeException {
        if(node == ROOT) {
            return false;
        }
        else if(node instanceof VariableNode) {
            return ((VariableNode) node).isLeaf();
        }

        return getLocalsTreeModel().isLeaf(node);
    }

    // ------------------------------------------------------------------------
    // TableModel implementation
    // ------------------------------------------------------------------------
    public Object getValueAt(Object node, String columnID) throws UnknownTypeException {
        if(Constants.WATCH_TYPE_COLUMN_ID.equals(columnID)) {
            if(node instanceof VariableNode) {
                String className = ((VariableNode)node).getTypeName();
                return (className == null) ? "" : className;
            }
            else if(node instanceof JToolTip) {
                // This is how tooltips are implemented in the debugger views.
                Object row = ((JToolTip)node).getClientProperty("getShortDescription");
                
                if(row instanceof VariableNode) {
                    if(row instanceof Refreshable && !((Refreshable) row).isCurrent()) {
                        return "";
                    }
                    return ((VariableNode)row).getTypeName();
                }
            }
        }
        else if(Constants.WATCH_VALUE_COLUMN_ID.equals(columnID)) {
            if(node instanceof VariableNode) {
                Object value = ((VariableNode)node).getValue();
                return value == null ? "null" : value;
            }
        }

        throw new UnknownTypeException(node);
    }

    public boolean isReadOnly(Object node, String string) throws UnknownTypeException {
        return true;
    }

    public void setValueAt(Object node, String string, Object value) throws UnknownTypeException {
        throw new UnknownTypeException(node);
    }

    // ------------------------------------------------------------------------
    // NodeModel implementation
    // ------------------------------------------------------------------------
    public String getDisplayName(Object node) throws UnknownTypeException {
        if (node == null) {
            return "null";
        }
        else if (node == ROOT) {
            return ROOT;
        }
        else if (node instanceof VariableNode) {
            return ((VariableNode)node).getDisplayName();
        }
        
        throw new UnknownTypeException(node);
    }

    public String getIconBase(Object node) throws UnknownTypeException {
        if (node == null || node == ROOT) {
            return VariableNode.LOCAL_VARIABLE_ICON;
        }
        else if (node instanceof VariableNode) {
            return ((VariableNode)node).getIconBase();
        }
        
        throw new UnknownTypeException(node);
    }

    public String getShortDescription(Object node) throws UnknownTypeException {
        if (node == null || node == ROOT) {
            return null;
        }
        else if (node instanceof VariableNode) {
            return ((VariableNode)node).getShortDescription();
        }
        
        throw new UnknownTypeException(node);
    }


    public void addModelListener(ModelListener l) {
        listeners.add(l);
    }

    public void removeModelListener(ModelListener l) {
        listeners.remove(l);
    }

    private void fireTreeChanged() {
        synchronized(watchToValue) {
            for (ScriptWatchEvaluating swe :  watchToValue.values()) {
                swe.setEvaluated(null);
            }
        }
        
        fireModelChanged(new ModelEvent.TreeChanged(this));
    }

    private void fireWatchesChanged() {
        fireModelChanged(new ModelEvent.NodeChanged(this, ROOT, ModelEvent.NodeChanged.CHILDREN_MASK));
    }

    void fireTableValueChangedChanged(Object node, String propertyName) {
        ((ScriptWatchEvaluating) node).setEvaluated(null);
        fireTableValueChangedComputed(node, propertyName);
    }

    void fireTableValueChangedComputed(Object node, String propertyName) {
        fireModelChanged(new ModelEvent.TableValueChanged(this, node, propertyName));
        
    }

    private void fireModelChanged(ModelEvent event) {
        Vector<ModelListener> v = new Vector<ModelListener>();
        
        synchronized (listeners) {
            v = new Vector<ModelListener>(listeners);
        }
        
        for (ModelListener ml : v) {
            ml.modelChanged(event);
        }       
    }

    // other methods ...........................................................

    DbgDebuggerImpl getDebugger() {
        return debugger;
    }

    private volatile VariablesModel localsModel;

    private VariablesModel getLocalsTreeModel() {
        if(localsModel == null) {
            localsModel = debugger != null ? debugger.getVariablesModel() : null;
        }
        return localsModel;
    }

    /** Current stack frame used to evaluate all non-fixed watches (fixed watches
     *  have a their own stack frame).
     */
    public DebugFrame getCurrentFrame() {
        return currentFrame;
    }

    /** Changes the current stack frame and reevaluates the watch pane.  Passing
     *  in null will leave all watches unevaluated (e.g. Debug | Continue) but
     *  watch expressions remain visible.
     */
    public void setStackFrame(DebugFrame frame) {
        currentFrame = frame;
        fireTreeChanged();
    }


    // innerclasses ............................................................

    static class ScriptWatchEvaluating
            implements VariableNode, Refreshable, PropertyChangeListener {

        private WatchesModel model;
        private Watch w;
        private DbgDebuggerImpl debugger;
        private VariableNode evaluatedWatch;

        private boolean [] evaluating = new boolean [] { false };
        private PropertyChangeSupport propSupp = new PropertyChangeSupport(this);

        public ScriptWatchEvaluating(WatchesModel model, Watch w, DbgDebuggerImpl debugger) {
            this.model = model;
            this.w = w;
            this.debugger = debugger;

            parseExpression(w.getExpression());
        }

        private void parseExpression(String exprStr) {
            setEvaluated(debugger.evaluateExpr(model.getCurrentFrame(), exprStr));
        }

        public void setEvaluated(VariableNode evaluatedWatch) {
            synchronized(this) {
                this.evaluatedWatch = evaluatedWatch;
            }
        }

        synchronized VariableNode getEvaluatedWatch() {
            return evaluatedWatch;
        }

        public void expressionChanged() {
            setEvaluated(null);
            parseExpression(w.getExpression());
        }

        public synchronized String getExceptionDescription() {
            return null;
        }

        public synchronized String getExpression() {
            return w.getExpression();
        }


        public synchronized void remove() {
            w.remove();
        }

        public void setExpression(String expression) {
            w.setExpression(expression);
            expressionChanged();
        }

        public synchronized void setValue(String value) { // throws InvalidExpressionException {
//            if(evaluatedWatch != null) {
//                evaluatedWatch.setValue(value);
//            } else {
//                throw new InvalidExpressionException("Can not set value while evaluating.");
//            }
            // !PW FIXME Cannot set values yet.
        }

        private VariableNode getUpdatedEvaluatedWatch() {
            synchronized(this) {
                VariableNode evaluatedWatch = this.evaluatedWatch;
            }
            if(evaluatedWatch == null) {
                getValue(); // To init the evaluatedWatch
            }
            return evaluatedWatch;
        }

        // --------------------------------------------------------------------
        // VariableNode
        // --------------------------------------------------------------------
        public String getName() {
            return w.getExpression();
        }

        public String getDisplayName() {
            return w.getExpression();
        }

        public String getShortDescription() {
            return w.getExpression();
        }

        public String getIconBase() {
            return WATCH_ICON;
        }

        public int getType() {
            return TYPE_WATCH;
        }

        public String getTypeName() {
            VariableNode evalWatch = getUpdatedEvaluatedWatch();
            return evalWatch != null ? evalWatch.getTypeName() : "Evaluating...";
        }

        public String getValue() {
            synchronized(evaluating) {
                if(evaluating[0]) {
                    try {
                        evaluating.wait();
                    } catch(InterruptedException iex) {
                        return null;
                    }
                }
                synchronized(this) {
                    if(evaluatedWatch != null) {
                        Object value = evaluatedWatch.getValue();
                        return(value != null) ? value.toString() : null;
                    }
                }
                evaluating[0] = true;
            }

            VariableNode vn = null;
            try {
                vn = debugger.evaluateExpr(model.getCurrentFrame(), w.getExpression());
            } finally {
                setEvaluated(vn);
                synchronized(evaluating) {
                    evaluating[0] = false;
                    evaluating.notifyAll();
                }
            }

            Object value = (vn != null) ? vn.getValue() : "no value yet...";
            return (value != null) ? value.toString() : null;
        }

        public String getTooltipValue() {
            Object v = getValue();
            return (v != null) ? v.toString() : null;
        }

        public VariableNode[] getChildren(int from, int to) {
            VariableNode evalWatch = getUpdatedEvaluatedWatch();
            return evalWatch != null ? evalWatch.getChildren(from, to) : new VariableNode[0];
        }

        public boolean isLeaf() {
            VariableNode evalWatch = getUpdatedEvaluatedWatch();
            return evalWatch != null ? evalWatch.isLeaf() : false;
        }

        public int getChildrenCount() {
            VariableNode evalWatch = getUpdatedEvaluatedWatch();
            return evalWatch != null ? evalWatch.getChildrenCount() : 0;
        }

        public boolean isReadOnly() {
            return true;
        }
        
        public void collectUpdates(Object source,
                               Collection<ModelEvent> events,
                               VariableNode newVar) {
            throw new IllegalStateException("This API valid on VariableNodeImpl only");
        }

        // --------------------------------------------------------------------
        // PropertyChange support
        // --------------------------------------------------------------------
        public void addPropertyChangeListener(PropertyChangeListener l) {
            propSupp.addPropertyChangeListener(l);
        }

        public void removePropertyChangeListener(PropertyChangeListener l) {
            propSupp.removePropertyChangeListener(l);
        }


        // --------------------------------------------------------------------
        // PropertyChangeListener
        // --------------------------------------------------------------------
        public void propertyChange(PropertyChangeEvent evt) {
            model.fireTableValueChangedChanged(this, null);
        }


        // --------------------------------------------------------------------
        // Refreshable
        // --------------------------------------------------------------------
        /** Does wait for the value to be evaluated. */
        public void refresh() throws RefreshFailedException {
            synchronized(evaluating) {
                if(evaluating[0]) {
                    try {
                        evaluating.wait();
                    } catch(InterruptedException iex) {
                        throw new RefreshFailedException(iex.getLocalizedMessage());
                    }
                }
            }
        }

        /** Tells whether the variable is fully initialized and getValue()
         *  returns the value immediately. */
        public synchronized boolean isCurrent() {
            return evaluatedWatch != null;
        }
}

    private static class Listener extends DebuggerManagerAdapter
            implements PropertyChangeListener {

        private WeakReference<WatchesModel> model;
        private WeakReference<DbgDebuggerImpl> debugger;

        private Listener(
            WatchesModel tm,
            DbgDebuggerImpl debugger
        ) {
            model = new WeakReference<WatchesModel>(tm);
            this.debugger = new WeakReference<DbgDebuggerImpl>(debugger);
            DebuggerManager.getDebuggerManager().addDebuggerListener(
                DebuggerManager.PROP_WATCHES,
                this
            );
//            debugger.addPropertyChangeListener(this);
            Watch[] ws = DebuggerManager.getDebuggerManager().getWatches();
            int i, k = ws.length;
            for(i = 0; i < k; i++) {
                ws [i].addPropertyChangeListener(this);
            }
        }

        private WatchesModel getModel() {
            WatchesModel m = model.get();
            if(m == null) {
                destroy();
            }
            return m;
        }

        public void watchAdded(Watch watch) {
            WatchesModel m = getModel();
            if(m == null) {
                return;
            }
            watch.addPropertyChangeListener(this);
            m.fireWatchesChanged();
        }

        public void watchRemoved(Watch watch) {
            WatchesModel m = getModel();
            if(m == null) {
                return;
            }
            watch.removePropertyChangeListener(this);
            m.fireWatchesChanged();
        }

        // currently waiting / running refresh task
        // there is at most one
        private RequestProcessor.Task task;

        public void propertyChange(PropertyChangeEvent evt) {
            String propName = evt.getPropertyName();
            // We already have watchAdded & watchRemoved. Ignore PROP_WATCHES:
            if(DebuggerManager.PROP_WATCHES.equals(propName)) {
                return;
            }
            final WatchesModel m = getModel();
            if(m == null) {
                return;
            }
//            if(m.debugger.getState() == JPDADebugger.STATE_DISCONNECTED) {
//                destroy();
//                return;
//            }
//            if(m.debugger.getState() == JPDADebugger.STATE_RUNNING) {
//                return ;
//            }

            if(evt.getSource() instanceof Watch) {
                Object node;
                synchronized(m.watchToValue) {
                    node = m.watchToValue.get(evt.getSource());
                }
                if(node != null) {
                    m.fireTableValueChangedChanged(node, null);
                    return ;
                }
            }

            if(task == null) {
                task = RequestProcessor.getDefault().create(new Runnable() {
                    public void run() {
                        System.err.println("WM do task " + task);
                        m.fireTreeChanged();
                    }
                });
                System.err.println("WM  create task " + task);
            }
            task.schedule(100);
        }

        private void destroy() {
            DebuggerManager.getDebuggerManager().removeDebuggerListener(
                DebuggerManager.PROP_WATCHES,
                this
            );
            DbgDebuggerImpl d = debugger.get();
            if(d != null) {
//                d.removePropertyChangeListener(this);
            }

            Watch[] ws = DebuggerManager.getDebuggerManager().getWatches();
            int i, k = ws.length;
            for(i = 0; i < k; i++) {
                ws [i].removePropertyChangeListener(this);
            }

            if(task != null) {
                // cancel old task
                task.cancel();
                System.err.println("WM cancel old task " + task);
                task = null;
            }
        }
    }
}
