/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <gtk/gtk.h>

#include <interface.h>
#include <support.h>

#include <netdude/nd.h>
#include <netdude/nd_timestamp.h>
#include <netdude/nd_clipboard.h>
#include <netdude/nd_globals.h>
#include <netdude/nd_dialog.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_main.h>
#include <netdude/nd_prefs.h>
#include <netdude/nd_recent.h>
#include <netdude/nd_tcpdump.h>
#include <netdude/nd_macros.h>
#include <netdude/nd_plugin.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_protocol_plugin.h>
#include <netdude/nd_trace_registry.h>
#include <netdude/nd_raw_protocol.h>
#include <netdude/nd_crc_32.h>

#ifdef RELEASE
#  define WELCOMESTRING   "Welcome to Netdude, Version " VERSION "."
#else
#  define WELCOMESTRING   "Welcome to Netdude, DEVELOPMENT Version " VERSION ", " LAST_UPDATE "."
#endif


GtkWidget            *toplevel;
char                  pcap_errbuf[PCAP_ERRBUF_SIZE];

static GList         *filenames = NULL;
ND_CmdOptions         cmd_options;

static void
main_help_exit(char *progname)
{
  printf(_("Netdude -- The Network Dump data Displayer and Editor.\n"
	   "USAGE: %s [OPTIONS] [tracefiles]\n"
	   "\n"
	   "  --help        This message.\n"
	   "  --debug       Print debugging output.\n"
	   "  --includedir  Prefix directory of header files.\n"
	   "  --plugindir   Plugin installation directory.\n"
	   "  --protodir    Protocol installation directory.\n"
	   "  --protos      Names of installed protocols.\n"
	   "  --version     Prints out version info.\n"
	   "\n"),
	 progname);
  exit(0);
}


static void
main_dump_protocol(ND_Protocol *proto, void *user_data)
{
  if (proto->id != 1)
    printf("%-32s%s\n", proto->name, (proto->plugin ? proto->plugin->version() : _("Unknown")));

  return;
  TOUCH(user_data);
}


static void
main_cmd_options_read(int argc, char** argv)
{
  int i;

  memset(&cmd_options, 0, sizeof(ND_CmdOptions));

  for (i = 1; i < argc; i++)
    {
      if (!strcmp(argv[i], "-h")     ||
	  !strcmp(argv[i], "--help") ||
	  !strcmp(argv[i], "-?"))
	{
	  main_help_exit(argv[0]);
	}
      else if (!strcmp(argv[i], "--debug"))
	{
	  cmd_options.debug = TRUE;
	}
      else if (!strcmp(argv[i], "--plugindir"))
	{
	  printf("%s/plugins\n", PACKAGE_DATA_DIR);
	  exit(0);
	}
      else if (!strcmp(argv[i], "--includedir"))
	{
	  printf("%s\n", PACKAGE_INCLUDE_DIR);
	  exit(0);
	}
      else if (!strcmp(argv[i], "--protodir"))
	{
	  printf("%s/protocols\n", PACKAGE_DATA_DIR);
	  exit(0);
	}
      else if (!strcmp(argv[i], "--protos"))
	{
	  cmd_options.show_protos = TRUE;
	}
      else if (!strcmp(argv[i], "--version"))
	{
	  printf("%s\n", VERSION);
	  exit(0);
	}
      else
	filenames = g_list_append(filenames, argv[i]);
    }
}


static void
main_cmd_options_apply(void)
{
  if (cmd_options.show_protos)
    {
      printf(_("Name\t\t\t\tVersion\n"));
      printf("--------------------------------------------------\n");
      nd_proto_registry_foreach_proto(main_dump_protocol, NULL);
      exit(0);
    }
}


static gint
main_startup_idle_handler(gpointer data)
{
  gint tcpdump_init_state = GPOINTER_TO_INT(data);

  if (tcpdump_init_state == FALSE)
    {
      nd_dialog_message(_("Tcpdump problem."),
			_("Cannot find a tcpdump executable as\n"
			  "specified in the preference settings.\n"
			  "Please specify the full path to your\n"
			  "tcpdump instance in the preferences dialog."),
			TRUE);
      return 0;
    }
  
  if (filenames)
    {
      ND_Trace *trace;
      GList    *l;
      char      canonical[MAXPATHLEN];
      char     *final;

      for (l = filenames; l; l = g_list_next(l))
	{
	  char *filename = (char *) l->data;

	  if (filename)
	    {
	      if (filename[0] != '/')
		{
		  char *file;
		  int   len;

		  if (!getcwd(canonical, MAXPATHLEN))
		    continue;
		  
		  len = strlen(canonical);
		  
		  file = canonical + len;
		  snprintf(canonical + len, MAXPATHLEN - len, "/%s",
			   filename);

		  final = canonical;
		}
	      else
		{
		  final = filename;
		}

	      trace = nd_trace_new(final);
	      nd_trace_registry_add(trace);
	    }
	}
    }
  
  return 0;
}


static void
main_close_tcpdumps_cb(ND_Trace *trace, void *user_data)
{
  nd_tcpdump_close(trace);

  return;
  TOUCH(user_data);
}


void
nd_main_safe_exit(void)
{
  ND_Trace *trace;

  trace = nd_trace_registry_get_current();

  if (trace && trace->dirty)
    {
      nd_dialog_exit();
    }
  else
    {
      nd_prefs_save();
      nd_trace_registry_foreach(main_close_tcpdumps_cb, NULL);
      
      exit(0);
    }
}


void
nd_main_unsafe_exit(void)
{
  nd_prefs_save_general();
  exit(0);
}


/* Paints a string with Netdude version number etc onto the
 * splash screen image.
 */
static void
main_setup_splash(GtkWidget *splash)
{
  char          label[MAXPATHLEN];
  GdkFont      *font;
  GtkWidget    *pixmap;
  GdkColor      fg;
  GdkGC        *gc;
  GdkColormap  *cmap;
  int           label_w, pix_w, pix_h;
  
  pixmap = gtk_object_get_data(GTK_OBJECT(splash), "splash_pixmap");
  font = gdk_font_load("-*-helvetica-medium-r-normal-*-8-*-*-*-p-*-*");
  cmap = gtk_widget_get_colormap(splash);
  memset(&fg, 0, sizeof(GdkColor));
  fg.red = 16384;
  fg.green = 16384;
  fg.blue = 16384;
  gdk_color_alloc(cmap, &fg);
  gc = gdk_gc_new(GTK_PIXMAP(pixmap)->pixmap);
  gdk_gc_set_foreground(gc, &fg);
  
  g_snprintf(label, MAXPATHLEN,
	     _("The NETwork DUmp data Displayer and Editor, %s %s"),
	     _("Version"), VERSION);
  
  label_w = gdk_string_width(font, label);
  gdk_window_get_size(GTK_PIXMAP(pixmap)->pixmap, &pix_w, &pix_h);

  gdk_draw_string(GTK_PIXMAP(pixmap)->pixmap, font, gc,
		  (pix_w - label_w) / 2, pix_h - 1, label);
}


int
main (int argc, char *argv[])
{
  int show_splash = TRUE;
  gint tcpdump_init_state;

#ifdef ENABLE_NLS
  bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
  textdomain (PACKAGE);
#endif

  gtk_set_locale ();
  gtk_init (&argc, &argv);

  add_pixmap_directory (PACKAGE_DATA_DIR "/pixmaps");
  add_pixmap_directory (PACKAGE_SOURCE_DIR "/pixmaps");


  /*
   * The following code was added by Glade to create one of each component
   * (except popup menus), just so that you see something after building
   * the project. Delete any components that you don't want shown initially.
   */
  toplevel = create_toplevel();
  

  /*
   * Init stuff here ...
   */
  signal(SIGPIPE, SIG_IGN);

  main_cmd_options_read(argc, argv);

  nd_gui_init();
  nd_clipboard_init();
  nd_prefs_init();

  nd_raw_proto_init();
  nd_proto_registry_init();
  nd_proto_plugin_init();

  nd_prefs_load();
  nd_recent_init();
  nd_plugin_init();
  nd_trace_registry_init();
  nd_crc_32_init();
  tcpdump_init_state = nd_tcpdump_init();
  nd_gui_statusbar_set(WELCOMESTRING);

  main_cmd_options_apply();

  gtk_widget_show (toplevel);

  nd_prefs_get_int_item(ND_DOM_NETDUDE, "show_splash", &show_splash);
  
  if (show_splash)
    {
      GtkWidget *splash;
      splash = create_splash_window();
      
      main_setup_splash(splash);
      gtk_widget_show (splash);
      gtk_timeout_add(4000, (GtkFunction)gtk_widget_hide, splash);
    }

  gtk_idle_add(main_startup_idle_handler, GINT_TO_POINTER(tcpdump_init_state));

  gtk_main ();

  return 0;
}

