/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_packet_iterator_h
#define __nd_packet_iterator_h

#include <gtk/gtk.h>

#include <netdude/nd_trace.h>

typedef struct nd_packet_iterator
{
  ND_Trace      *trace;
  ND_Packet     *current;
  int            current_index;
  gboolean       selected;

} ND_PacketIterator;


/* An iterator interface for iterating over a trace's packets,
   either all of them or only the selected ones. If apply-to-all
   is activated, iterating over all selected packets automatically
   equals iterating over all packets. */

/**
 * nd_pit_init - packet iterator initializer
 * @pit: iterator to initialize.
 * @trace: trace whose packets are to be iterated.
 * @selected: iterate all packets or only selected?
 *
 * This is the initializer for an iterator. Set @selected to %TRUE
 * when you want to iterate over the selected packets only, otherwise all
 * packets in the trace are iterated. Also initializes the
 * progress bar accordingly. By keeping iterators on the local
 * stack and not dynamically allocating/freeing each one, we
 * can simplify the usage a good deal. See nd_filer_activate() as
 * an example.
 */
void               nd_pit_init(ND_PacketIterator *pit,
			       ND_Trace *trace,
			       gboolean selected);

/**
 * nd_pit_get - accessor function.
 * @pit: iterator to retrieve current packet from.
 *
 * Accessor function, returns the packet the iterator currently
 * points to.
 *
 * Returns: currently iterated packet.
 */
ND_Packet         *nd_pit_get(ND_PacketIterator *pit);


/**
 * nd_pit_next - advances the iterator.
 * @pit: iterator to advance
 *
 * This function advances the iterator, and returns that packet. If there are
 * no more packets, %NULL is returned. Upon advancing the iterator,
 * the progressbar and the packet's tcpdump line automatically
 * get updated.
 *
 * Returns: next packet.
 */
ND_Packet         *nd_pit_next(ND_PacketIterator *pit);


/**
 * nd_pit_get_index - packet index retrieval.
 * @pit: iterator to get current packet index from.
 *
 * Assuming @pit is an iterator over all packets (i.e. either
 * we're in apply-to-all mode or selected was not set when the
 * iterater was initialized), it is possible to get the index
 * of the currently iterated packet using this function. Otherwise,
 * -1 is returned.
 *
 * Returns: index of current packet.
 */
int                nd_pit_get_index(ND_PacketIterator *pit);

#endif
