
/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <dirent.h>
#include <glib.h>

#include <netdude/nd_globals.h>
#include <netdude/nd_macros.h>
#include <netdude/nd_debug.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_plugin.h>
#include <netdude/nd_prefs.h>
#include <netdude/nd_trace_registry.h>
#include <netdude/nd_packet_iterator.h>
#include <callbacks.h>
#include <interface.h>
#include <support.h>

static GList *plugins = NULL;  /* list<ND_Plugin*> */

static ND_Plugin *plugin_new(char *filename);
static void       plugin_scan(const char *dirname);
static void       plugin_hook_up(ND_Plugin *p);
static void       plugin_load(void);
static void       plugin_setup_gui(void);
static int        plugin_cmp(ND_Plugin *p1, char *filename);

static const char      *
plugin_dummy_name(void)
{
  return _("Unnamed plugin.");
}


static const char      *
plugin_dummy_description(void)
{
  return _("No description given.");
}


static const char      *
plugin_dummy_author(void)
{
  return _("No author(s) specified.");
}

static const char      *
plugin_dummy_version(void)
{
  return _("No version specified.");
}


static int
plugin_dummy_run(ND_Trace *trace)
{
  return 0;
  TOUCH(trace);
}


static ND_Plugin *
plugin_new(char *filename)
{
  ND_Plugin *p;

  p = (ND_Plugin*) g_new0(ND_Plugin, 1);
  D_ASSERT_PTR(p);
  if (!p)
    return NULL;

  if (filename)
    p->filename = g_strdup(filename);

  p->name        = plugin_dummy_name;
  p->description = plugin_dummy_description;
  p->author      = plugin_dummy_author;
  p->version     = plugin_dummy_version;
  p->run         = plugin_dummy_run;

  return (p);
}


static int    
plugin_cmp(ND_Plugin *p1, char *filename)
{
  if (!p1 || !filename)
    return (-1);

  return (strcmp(p1->filename, filename));
}


static void
plugin_scan(const char *dir)
{
  DIR             *dirp;
  struct dirent   *de;
  char            *dot;

  D(("Scanning '%s' for plugins\n", dir));

  if ((dirp = opendir(dir)) != NULL)
    {
      for (de = readdir(dirp); de; de = readdir(dirp))
	{
	  if (de->d_name[0] != '.')
	    {
	      dot = strrchr(de->d_name, '.');

	      if (dot)
		*dot = '\0';

	      if (!g_list_find_custom(plugins, de->d_name, (GCompareFunc)plugin_cmp))
		plugins = g_list_append(plugins, plugin_new(de->d_name));
	    }
	}      
      closedir(dirp);
    }
}


static void       
plugin_hook_up(ND_Plugin *plugin)
{
  lt_ptr_t     sym;

  if ((sym = lt_dlsym(plugin->lt, "name")) != NULL)
    plugin->name = sym;
  if ((sym = lt_dlsym(plugin->lt, "description")) != NULL)
    plugin->description = sym;
  if ((sym = lt_dlsym(plugin->lt, "author")) != NULL)
    plugin->author = sym;
  if ((sym = lt_dlsym(plugin->lt, "version")) != NULL)
    plugin->version = sym;

  if ((sym = lt_dlsym(plugin->lt, "run")) != NULL)
    plugin->run = sym;
}


static gint
plugin_strcmp(gconstpointer a, gconstpointer b)
{
  ND_Plugin *p1 = (ND_Plugin *) a;
  ND_Plugin *p2 = (ND_Plugin *) b;

  return strcmp(p1->name(), p2->name());
}

static void       
plugin_load(void)
{
  GList       *l;
  ND_Plugin   *plugin;
  char         s[MAXPATHLEN];

  for (l = g_list_first(plugins); l; l = g_list_next(l))
    {
      plugin = (ND_Plugin*)l->data;

      g_snprintf(s, MAXPATHLEN, "%s/%s", nd_prefs_get_plugin_dir_user(), plugin->filename);

      if ((plugin->lt = lt_dlopenext(s)))
	{
	  plugin_hook_up(plugin);
	}
      else
	{
	  g_snprintf(s, MAXPATHLEN, "%s/%s", nd_prefs_get_plugin_dir_global(), plugin->filename);

	  if ((plugin->lt = lt_dlopenext(s)))
	    {
	      plugin_hook_up(plugin);
	    }
	}
    }

  /* Now sort the list of plugins alphanumerically,
     so that we automatically get sorted menus etc.
  */
  plugins = g_list_sort(plugins, plugin_strcmp);
}


static void       
plugin_setup_gui(void)
{
  GtkWidget   *parent_menu, *menu, *item;
  GList       *l;
  ND_Plugin   *plugin;
  int          i;

  /* Hook plugin names into "Plugins" menu ... */

  menu = gtk_object_get_data(GTK_OBJECT(toplevel), "plugins_menu");
  D_ASSERT_PTR(menu);

  for (i = 0, l = g_list_first(plugins); l; i++, l = g_list_next(l))
    {
      plugin = (ND_Plugin*)l->data;

      item = gtk_menu_item_new_with_label(plugin->name());
      gtk_widget_ref(item);
      gtk_widget_show(item);
      gtk_object_set_data(GTK_OBJECT(item), "plugin", plugin);
      gtk_menu_insert(GTK_MENU(menu), item, i);
      /*     gtk_container_add(GTK_CONTAINER(menu), item); */
      gtk_signal_connect (GTK_OBJECT (item), "activate",
			  GTK_SIGNAL_FUNC (on_plugin_activate),
			  NULL);
    }

  /* Hook plugin names into "About Plugins" menu ... */

  parent_menu = gtk_object_get_data(GTK_OBJECT(toplevel), "about_plugins");
  D_ASSERT_PTR(parent_menu);

  menu = gtk_menu_new();
  gtk_widget_ref (menu);
  gtk_object_set_data_full (GTK_OBJECT (toplevel), "about_plugins_menu", menu,
                            (GtkDestroyNotify) gtk_widget_unref);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (parent_menu), menu);

  for (l = g_list_first(plugins); l; l = g_list_next(l))
    {
      plugin = (ND_Plugin*)l->data;

      item = gtk_menu_item_new_with_label(plugin->name());
      gtk_widget_ref(item);
      gtk_object_set_data_full (GTK_OBJECT (menu), plugin->name(), item,
				(GtkDestroyNotify) gtk_widget_unref);
      gtk_widget_show(item);
      gtk_object_set_data(GTK_OBJECT(item), "plugin", plugin);
      gtk_container_add(GTK_CONTAINER(menu), item);
      gtk_signal_connect (GTK_OBJECT (item), "activate",
			  GTK_SIGNAL_FUNC (on_about_plugin_activate),
			  NULL);
    }
}


void 
nd_plugin_init(void)
{
  if (lt_dlinit() != 0)
    {
      fprintf(stderr, "Could not initialize libltdl -- exiting.\n");
      exit(-1);
    }

  plugin_scan(nd_prefs_get_plugin_dir_user());
  plugin_scan(nd_prefs_get_plugin_dir_global());
  plugin_load();
  plugin_setup_gui();
}


void    
nd_plugin_run(ND_Plugin *plugin)
{
  ND_Trace          *trace;

  trace = nd_trace_registry_get_current();

  plugin->run(trace);
}


const char   *
nd_plugin_get_description(ND_Plugin *plugin)
{
  if (!plugin)
    return (NULL);

  return (plugin->description());
}


const char   *
nd_plugin_get_author(ND_Plugin *plugin)
{
  if (!plugin)
    return (NULL);

  return (plugin->author());
}


void    
nd_plugin_show_about(ND_Plugin *plugin)
{
  char       s[MAXPATHLEN];
  GtkWidget *w;
  GtkWidget *label;

  D_ASSERT_PTR(plugin);
  if (!plugin)
    return;

  w = create_plugin_about_dialog();

  g_snprintf(s, MAXPATHLEN, "%s, %s.", plugin->name(), plugin->version());
  label = gtk_object_get_data(GTK_OBJECT(w), "plugin_name_label");
  D_ASSERT_PTR(label);
  gtk_label_set_text(GTK_LABEL(label), s);
  
  g_snprintf(s, MAXPATHLEN, _("Author: %s"), plugin->author());
  label = gtk_object_get_data(GTK_OBJECT(w), "plugin_author_label");
  D_ASSERT_PTR(label);
  gtk_label_set_text(GTK_LABEL(label), s);

  label = gtk_object_get_data(GTK_OBJECT(w), "plugin_description_label");
  D_ASSERT_PTR(label);
  gtk_label_set_text(GTK_LABEL(label), plugin->description());
  
  gtk_widget_show(w);  
}

