/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifndef __nd_plugin_h
#define __nd_plugin_h

#include <ltdl.h>
#include <netdude/nd_trace.h>

/* Netdude plugins are dynamically loaded modules that can act as filters
 * on packets of a tracefile, or can basically do anything you want them
 * to. They can potentially access any public functions in Netdude and
 * can act as if they were part of the main application.
 *
 * Possible applications are anonymizers, checksummers etc. 
 */
typedef struct nd_plugin
{
  const char *(*name) (void);
  const char *(*description) (void);
  const char *(*author) (void);
  const char *(*version) (void);

  int         (*run) (ND_Trace *trace);

  char        *filename;
  lt_dlhandle  lt;

} ND_Plugin;


/**
 * nd_plugin_init - initializes feature plugins.
 *
 * The function scans the plugin directories, reads in the
 * plugins it finds and hooks them into the main window.
 */
void          nd_plugin_init(void);

/**
 * nd_plugin_run - runs a plugin.
 * @plugin: plugin to run.
 * 
 * The function runs the given @plugin on the  current
 * trace. If there is no current trace, it just returns.
 */
void          nd_plugin_run(ND_Plugin *plugin);

/**
 * nd_plugin_get_description - returns plugin description string.
 * @plugin: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string describing the plugin.
 *
 * Returns: description string.
 */
const char   *nd_plugin_get_description(ND_Plugin *plugin);

/**
 * nd_plugin_get_author - returns plugin author.
 * @plugin: plugin to query.
 *
 * The function returns a pointer to statically allocated memory
 * containing a string with the author's name.
 *
 * Returns: author name string.
 */
const char   *nd_plugin_get_author(ND_Plugin *plugin);

/**
 * nd_plugin_show_about - shows About dialog for plugin.
 * @plugin: plugin whose About info is shown.
 *
 * The function shows the About dialog for @plugin, containing
 * version and author info, a description etc.
 */
void          nd_plugin_show_about(ND_Plugin *plugin);

#endif
