/*

Copyright (C) 2000, 2001, 2002 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#ifdef LINUX
#define __FAVOR_BSD
#endif

#include <gtk/gtk.h>
#include <netdude/nd.h>
#include <netdude/nd_globals.h>
#include <netdude/nd_dialog.h>
#include <netdude/nd_gui.h>
#include <netdude/nd_misc.h>
#include <netdude/nd_tcpdump.h>
#include <netdude/nd_packet.h>
#include <netdude/nd_trace.h>
#include <netdude/nd_prefs.h>
#include <netdude/nd_recent.h>
#include <netdude/nd_null_protocol.h>
#include <netdude/nd_packet_iterator.h>
#include <netdude/nd_protocol_registry.h>
#include <netdude/nd_protocol_inst.h>
#include <netdude/nd_trace_registry.h>
#include <callbacks.h>
#include <support.h>


typedef struct nd_trace_save_data
{
  ND_Trace           *trace;
  char               *filename;
  ND_DialogCallback   callback_finished;
  void               *user_data;

} ND_TraceSaveData;


static void
trace_save_no_cb                    (void *user_data)
{
  ND_Trace *trace = (ND_Trace *) user_data;

  nd_trace_registry_remove(trace);
  nd_trace_free(trace);
}

static void
trace_save_yes_cb                   (void *user_data)
{
  ND_Trace *trace = (ND_Trace *) user_data;

  nd_trace_save_as_dialog(trace, trace_save_no_cb, trace);
}


void 
nd_trace_close(ND_Trace *trace)
{
  if (!trace)
    return;

  if (trace->dirty)
    {
      nd_dialog_ync(_("Trace modified"),
		    _("The trace file has changed, do yo want\n"
		      "to save changes before closing it?"),
		    trace_save_yes_cb,
		    trace_save_no_cb,
		    NULL,
		    trace);
    }
  else
    {
      nd_trace_registry_remove(trace);
      nd_trace_free(trace);
    }
}


static void
trace_cleanup_state_cb(ND_Protocol *proto, void *user_data)
{
  if (proto->is_stateful)
    proto->free_state((ND_Trace *) user_data);
}


void    
nd_trace_free(ND_Trace *trace)
{
  ND_Packet *p, *p2;

  if (!trace)
    return;

  D_ENTER;

  nd_proto_registry_foreach_proto(trace_cleanup_state_cb, trace);

  if (trace->pcap)
    pcap_close(trace->pcap);

  nd_tcpdump_close(trace);

  nd_gui_del_monowidth_widget(trace->list);

  /* Clean up the gui memory -- needs to be
     done before cleaning up packets */
  gtk_widget_unref(GTK_WIDGET(trace->tab));

  g_hash_table_destroy(trace->registry);

  g_free(trace->filename);
  g_free(trace->unnamed);

  p = trace->pl;

  while (p)
    {
      p2 = p;
      p = p->next;
      nd_packet_free(p2);
    }
  
  g_free(trace->cur_pi);
  g_free(trace->cur_pi_sel);
  g_free(trace);
  D_RETURN;
}


void
nd_trace_add_proto_tab(ND_Trace *trace,
		       ND_Protocol *proto,
		       guint nesting)
{
  GtkNotebook   *notebook;
  GtkWidget     *scrolledwin, *viewport;
  const char    *pi_string;
  ND_ProtoInfo  *pinf;

  if (!trace || !proto)
    return;

  D(("Initializing protocol %s, nesting %i, for trace %s\n",
     proto->name, nesting, trace->filename));
  
  notebook = nd_trace_get_notebook(trace);
  D_ASSERT_PTR(notebook);

  pi_string = nd_proto_inst_to_string(proto, nesting);

  if (gtk_object_get_data(GTK_OBJECT(notebook), pi_string))
    return;

  /* Create a scrolled window -- everything else is placed
     into it: */

  scrolledwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_ref(scrolledwin);

  /* From the protocol notebook of this trace, each tab can
     be retrieved by a protocol instance: */

  gtk_object_set_data_full (GTK_OBJECT (notebook), pi_string, scrolledwin,
			    (GtkDestroyNotify) gtk_widget_unref);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwin),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  /* Create the protocol's content viewport: */
  
  viewport = gtk_viewport_new (NULL, NULL);
  gtk_widget_ref (viewport);
  gtk_object_set_data_full (GTK_OBJECT (scrolledwin), "viewport", viewport,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show (viewport);
  gtk_viewport_set_shadow_type (GTK_VIEWPORT (viewport), GTK_SHADOW_NONE);
  gtk_container_add(GTK_CONTAINER(scrolledwin), viewport);

  /* And now create and hook in the protocol's GUI, as it desires: */

  pinf = nd_proto_info_new(proto, nesting);
  pinf->proto_tab = scrolledwin;
  pinf->proto_gui = proto->create_gui(trace, pinf);
  pinf->proto_label = gtk_label_new(proto->name);

  gtk_container_add(GTK_CONTAINER(viewport), pinf->proto_gui);
  gtk_widget_show(pinf->proto_gui);
  gtk_widget_ref(pinf->proto_gui);
  gtk_object_set_data_full (GTK_OBJECT (scrolledwin), "content", pinf->proto_gui,
			    (GtkDestroyNotify) gtk_widget_unref);
  
  gtk_notebook_append_page(notebook, scrolledwin,
			   pinf->proto_label);

  /* Via the string version of the proto info, every protocol info
     becomes available from the notebook widget */
  gtk_object_set_data(GTK_OBJECT(notebook), pi_string, pinf);
  
  /* From each protocol's tab in the GUI, information in a ND_ProtoInfo
   * structure is made available: */

  gtk_object_set_data(GTK_OBJECT(scrolledwin), "pinf", pinf);

  if (proto->is_stateful)
    proto->init_state(trace);
}


/* Callback that gets called for each registered protocol
 * when a new trace is created. It hooks each protocol's gui
 * into the trace and initializes any protocol's state information,
 * given that a protocol is stateful.
 */
static void
trace_proto_cb(ND_Protocol *proto, void *user_data)
{
  ND_Trace    *trace = (ND_Trace *) user_data;

  nd_trace_add_proto_tab(trace, proto, 0);
}


ND_Trace *
nd_trace_new(const char *filename)
{
  char                 message[MAXPATHLEN];
  ND_Trace            *trace;

  if (filename)
    {
      if (!nd_misc_can_read(filename))
	{
	  g_snprintf(message, MAXPATHLEN, _("Cannot access '%s'."), filename);
	  nd_gui_statusbar_set(message);
	  return NULL;
	}

      if (!nd_misc_is_tcpdump_file(filename))
	{
	  g_snprintf(message, MAXPATHLEN, _("File '%s' doesn't seem to be a tcpdump trace."), filename);
	  nd_gui_statusbar_set(message);
	  return NULL;
	}
    }

  trace = g_new0(ND_Trace, 1);
  D_ASSERT_PTR(trace);
  if (!trace)
    return NULL;

  trace->sel.handler = -1;

  /* Create the trace data registry: */
  trace->registry = g_hash_table_new(g_str_hash, g_str_equal);

  if (filename)
    {
      /* Hook in the canonical filename: */
      trace->filename = g_strdup(filename);
    }
  else
    {
      trace->unnamed = g_strdup(nd_misc_get_unnamed_string());
    }

  /* Fork tcpdump child and initialize comms channel: */
  if (!nd_tcpdump_open(trace))
    goto error;

  /* Create this trace's GUI content: */
  nd_gui_trace_new_tab(trace);
  nd_proto_registry_foreach_proto(trace_proto_cb, trace);
  gtk_notebook_set_page(nd_trace_get_notebook(trace), 0);
  nd_gui_list_incomplete_column_visible(trace->list, FALSE);

  if (filename)
    {
      if ( (trace->pcap = pcap_open_offline(trace->filename, pcap_errbuf)) == NULL)
	goto error;
      
      nd_gui_statusbar_set(_("Loading file..."));

      /* Do it! */
      nd_gui_pbar_start_activity();  
      pcap_loop(trace->pcap, 0, nd_packet_pcap_read_handler, (u_char*) trace);
      nd_gui_pbar_stop_activity();

      nd_trace_init_packets(trace);

      /* Sync GUI list to packets: */
      nd_gui_list_update(trace, FALSE);

      nd_gui_statusbar_set(_("File loaded."));
  
      /* Update "recently-used" menu entries: */
      nd_recent_add_file(filename);
    }

  return trace;
  
 error:

  if (trace->pcap)
    pcap_close(trace->pcap);

  if (trace->tcpdump.pid)
    nd_tcpdump_close(trace);

  g_free(trace->filename);
  g_free(trace);

  return NULL;
}


void
nd_trace_init_header(ND_Trace *trace, guint32 snap_len, guint32 datalink_type)
{
  if (!trace)
    return;

  if (nd_trace_initialized(trace))
    return;

  memset(&trace->tcpdump.pfh, 0, sizeof(struct pcap_file_header));
  trace->tcpdump.pfh.magic = 0xa1b2c3d4;
  trace->tcpdump.pfh.version_major = PCAP_VERSION_MAJOR;
  trace->tcpdump.pfh.version_minor = PCAP_VERSION_MINOR;
  trace->tcpdump.pfh.snaplen = snap_len;
  trace->tcpdump.pfh.linktype = datalink_type;

  nd_tcpdump_send_header(trace); 
}


gboolean       
nd_trace_initialized(const ND_Trace *trace)
{
  if (!trace)
    return FALSE;

  return (trace->tcpdump.pfh.magic   != 0   &&
	  trace->tcpdump.pfh.snaplen != 0);
}


void           
nd_trace_init_packets(ND_Trace *trace)
{
  int i;
  ND_Packet *p;
  
  if (!trace || !trace->pl)
    return;
  
  nd_gui_statusbar_set(_("Initializing packets..."));
  nd_gui_pbar_reset(trace->num_packets);
  
  /* Initialize all ND_Packets internally: */
  for (i = 0, p = trace->pl; p; p = p->next, i++)
    {
      nd_packet_init(p);
      nd_packet_update_proto_state(p, i);
      
      if (!nd_packet_is_complete(p))
	{
	  nd_gui_list_incomplete_column_visible(trace->list, TRUE);
	  trace->incomplete = TRUE;
	}
      
      nd_gui_pbar_inc();
    }
  
  nd_gui_pbar_clear();

  if (trace == nd_trace_registry_get_current())
    nd_gui_num_packets_set();
}


const char    *
nd_trace_get_name(const ND_Trace *trace)
{
  if (!trace)
    return _("netdude.trace");

  return (trace->filename ? trace->filename : trace->unnamed);
}


ND_ProtoInst *
nd_trace_get_current_proto_inst(const ND_Trace *trace)
{
  if (!trace)
    return NULL;

  return trace->cur_pi;
}


guchar *
nd_trace_get_current_proto_header(const ND_Trace *trace)
{
  if (!trace)
    return NULL;
  
  return trace->cur_header;
}


void
nd_trace_set_current_proto_selection(ND_Trace *trace,
				     const ND_ProtoInst *pi)
{
  if (!trace || !pi || !pi->proto)
    return;

  if (!trace->cur_pi_sel)
    {
      trace->cur_pi_sel = g_new0(ND_ProtoInst, 1);

      if (!trace->cur_pi_sel)
	return;
    }

  D(("Setting current proto selection to %s/%i\n",
     pi->proto->name, pi->nesting));

  *trace->cur_pi_sel = *pi;
}


void
nd_trace_set_current_proto(ND_Trace *trace,
			   const ND_ProtoInst *pi,
			   gboolean gui_update)
{
  GtkNotebook *notebook;
  int num;

  if (!trace || !trace->cur_packet || !pi)
    return;

  D(("Setting current proto to %s/%i\n",
     (pi ? pi->proto->name : "none"),
     (pi ? pi->nesting : 0)));

  if (!trace->cur_pi)
    {
      trace->cur_pi = g_new0(ND_ProtoInst, 1);

      if (!trace->cur_pi)
	return;
    }
  
  *trace->cur_pi = *pi;

  if (trace->cur_packet)
    {
      trace->cur_header = nd_packet_get_data(trace->cur_packet,
					     pi->proto,
					     pi->nesting);
      D(("NEW current header: %p\n", trace->cur_header));
    }

  if (gui_update)
    {
      ND_ProtoInfo *pinf;

      notebook = nd_trace_get_notebook(trace);
      pinf = nd_trace_get_proto_info(trace, pi->proto, pi->nesting);
      D_ASSERT_PTR(pinf);
      if (!pinf)
	return;

      num = gtk_notebook_page_num(notebook, pinf->proto_tab);
      
      gtk_signal_handler_block_by_data(GTK_OBJECT(notebook), notebook);
      if (num != gtk_notebook_get_current_page(notebook))
	gtk_notebook_set_page(notebook, num);
      gtk_signal_handler_unblock_by_data(GTK_OBJECT(notebook), notebook);
    }
}


void           
nd_trace_set_current_packet(ND_Trace *trace, ND_Packet *packet)
{    
  if (!trace)
    return;

  if (packet)
    {
      if (trace != packet->trace)
	return;
    }
  
  D(("Current packet now %p\n", packet));
  trace->cur_packet = packet;

  nd_trace_set_current_proto(trace, trace->cur_pi, FALSE);
  nd_packet_set_gui(packet);
}


static void
trace_load_dialog_cb(const char *filename,
		     void *user_data)
{
  if (nd_misc_is_tcpdump_file(filename))
    {
      ND_Trace *trace;
      char *dir;
      nd_dialog_filesel_close();
      
      dir = g_dirname(filename);      
      nd_prefs_set_str_item(ND_DOM_NETDUDE, "load_dir", dir);
      g_free(dir);
      trace = nd_trace_new(filename);
      D_ASSERT_PTR(trace);
      nd_trace_registry_add(trace);
    }
  else
    {
      nd_dialog_message(_("Error loading file."),
			_("This file does not seem\n"
			  "to be a tcpdump tracefile."),
			TRUE);
    }

  return;
  TOUCH(user_data);
}


void           
nd_trace_load_dialog(void)
{
  char *load_dir = NULL;

  if (nd_prefs_get_str_item(ND_DOM_NETDUDE, "load_dir", &load_dir))
    {
      load_dir = nd_misc_add_slash(g_strdup(load_dir));
    }

  nd_dialog_filesel(_("Load Tcpdump Tracefile"),
		    load_dir,
		    trace_load_dialog_cb, NULL);

  g_free(load_dir);		    
}


static void
trace_save_overwrite_yes_cb(ND_TraceSaveData *data)
{
  char *dirname;

  nd_dialog_filesel_close();

  dirname = g_dirname(data->filename);
  nd_prefs_set_str_item(ND_DOM_NETDUDE, "save_dir", dirname);
  nd_trace_save_as(data->trace, data->filename);

  if (data->callback_finished)
    data->callback_finished(data->user_data);

  g_free(dirname);
  g_free(data->filename);
  g_free(data);
}


static void
trace_save_as_dialog_cb(const char *filename,
			void *user_data)
{
  ND_TraceSaveData *data;

  data = (ND_TraceSaveData *) user_data;
  data->filename = g_strdup(filename);

  if (nd_misc_exists(filename))
    {
      nd_dialog_generic(ND_DIALOG_QUESTION,
			_("File exists."),
			_("The file already exists.\n"
			  "Do you want to overwrite it?"),
			TRUE,
			NULL,
			data,
			2,
			_("Yes"), (ND_DialogCallback) trace_save_overwrite_yes_cb,
			_("No"), NULL);
	  
      return;
    }

  trace_save_overwrite_yes_cb(data);
}


void           
nd_trace_save_as_dialog(ND_Trace *trace,
			ND_DialogCallback callback_finished,
			void *user_data)
{
  ND_TraceSaveData *data;
  char filename[MAXPATHLEN];
  char *save_dir = NULL;
  const char *trace_filename_only;

  if (!trace)
    return;

  data = g_new0(ND_TraceSaveData, 1);
  data->trace = trace;
  data->callback_finished = callback_finished;
  data->user_data = user_data;

  filename[0] = 0;

  if (trace->filename)
    trace_filename_only = g_basename(trace->filename);
  else
    trace_filename_only = trace->unnamed;
  
  if (nd_prefs_get_str_item(ND_DOM_NETDUDE, "save_dir", &save_dir))
    g_snprintf(filename, MAXPATHLEN, "%s/%s", save_dir, trace_filename_only);
  else
    g_snprintf(filename, MAXPATHLEN, "%s", trace_filename_only);

  nd_dialog_filesel(_("Save Tcpdump Tracefile"),
		    filename,
		    trace_save_as_dialog_cb,
		    data); 
}


gboolean
nd_trace_save(ND_Trace *trace)
{
  pcap_dumper_t  *pdt;
  ND_Packet      *p;

  if (!trace)
    return FALSE;

  /* Live captures don't yet have a filename ... */
  if (!trace->filename)
    {
      nd_trace_save_as_dialog(trace, NULL, NULL);
      return FALSE;
    }

  D(("Dump open: %p %p\n", trace->pcap, trace->filename));
  if ( (pdt = pcap_dump_open(trace->pcap, trace->filename)) == NULL)
    return 0;

  for (p = trace->pl; p; p = p->next)
    {
      if (!p->is_hidden)
	pcap_dump((u_char*)pdt, &(p->ph), p->data);
    }
  
  pcap_dump_close(pdt);
  nd_gui_statusbar_set(_("File saved."));

  /* Update "recently-used" menu entries: */
  nd_recent_add_file(trace->filename);

  nd_trace_set_dirty(trace, FALSE);
  nd_gui_modified_set(trace);
  return TRUE;
}


int
nd_trace_save_as(ND_Trace *trace, const char *filename)
{
  if (!trace || !filename)
    return FALSE;

  g_free(trace->filename);
  trace->filename = g_strdup(filename);

  if (nd_trace_save(trace))
    {
      nd_gui_windowtitle_set(nd_trace_get_name(trace));
      nd_gui_trace_name_set(trace);

      return TRUE;
    }

  return FALSE;
}


void    
nd_trace_move_packet(ND_Trace *trace, int from_index, int to_index)
{
  ND_Packet *p, *p2;

  if (!trace)
    return;

  if (to_index > from_index)
    to_index++;

  D(("%i --> %i\n", from_index, to_index));
  
  p  = nd_trace_packet_get_nth(trace, from_index);
  p2 = nd_trace_packet_get_nth(trace, to_index);

  if (p)
    {
      if (p->prev)
	{
	  if (p->next)
	    {
	      p->prev->next = p->next;
	      p->next->prev = p->prev;
	    }
	  else
	    {
	      p->prev->next = NULL;
	      trace->pl_end = p->prev;
	    }

	}
      else if (p->next)
	{
	  trace->pl = p->next;
	  p->next->prev = NULL;
	}
      else
	{
	  /* No prev and no next -- moving doesn't make
	     much sense, so we return. */
	  return;
	}

      if (p2)
	{
	  p->next = p2;
	  p->prev = p2->prev;

	  if (p2->prev)
	    p2->prev->next = p;
	  else
	    trace->pl = p;

	  p2->prev = p;
	}
      else
	{
	  trace->pl_end->next = p;
	  p->prev = trace->pl_end;
	  trace->pl_end = p;
	  p->next = NULL;
	}

      nd_trace_set_dirty(trace, TRUE);
      nd_packet_update_proto_state(p, to_index);
      nd_gui_list_update_packet(p);
      trace->sel.last_valid = FALSE;
    }
}


/* Gtk idle handler for setting current packet */
static int
trace_set_current_packet_ih(gpointer data)
{
  ND_Trace *trace = (ND_Trace *) data;

  if (!trace)
    return FALSE;

  trace->sel.handler = -1;
  nd_trace_set_current_packet(trace, trace->sel.last);

  return FALSE;
}


void    
nd_trace_select_packet(ND_Trace *trace, int index)
{
  ND_Packet   *p, *s = NULL, *s_last = NULL;
  int          i;

  if (!trace)
    return;

  p = trace->pl;
  s = nd_trace_sel_get(trace, TRUE);
  i = 0;

  if (!s)
    {
      /* No selected packet yet -- we can speed things up */

      p = nd_trace_packet_get_nth(trace, index);
      if (p)
	{
	  trace->sel.sel = p;
	  p->sel_next = NULL;
	  p->sel_prev = NULL;
	  trace->sel.size++;

	  trace->sel.last = p;
	  trace->sel.last_index = index;
	  trace->sel.last_valid = TRUE;

	  if (trace->sel.handler < 0)
	    trace->sel.handler = gtk_idle_add(trace_set_current_packet_ih, trace);
	}

      return;
    }

  /* We already have something selected */

  if (trace->sel.last_valid && index >= trace->sel.last_index)
    {
      if (index == trace->sel.last_index)
	/* already selected -- do nothing */
	return;
      
      s_last = p = trace->sel.last;
      s = s_last->sel_next;
      i = trace->sel.last_index;

      D(("Speeding up, from index %i\n", i));
    }
  
  while (p)
    {
      if (i == index)
	{
	  if (s_last)
	    {
	      p->sel_prev = s_last;
	      p->sel_next = s_last->sel_next;
	      if (p->sel_next)
		p->sel_next->sel_prev = p;
	      s_last->sel_next = p;
	    }
	  else /* It's before the first selected packet */
	    {
	      p->sel_next = nd_trace_sel_get(trace, TRUE);
	      trace->sel.sel->sel_prev = p;
	      p->sel_prev = NULL;
	      trace->sel.sel = p;
	    }

	  trace->sel.size++;

	  trace->sel.last = p;
	  trace->sel.last_index = index;
	  trace->sel.last_valid = TRUE;

	  if (trace->sel.handler < 0)
	    trace->sel.handler = gtk_idle_add(trace_set_current_packet_ih, trace);
	  return;
	}

      if (p == s)
	{
	  s_last = p;
	  s = p->sel_next;
	}
      
      i++;
      p = p->next;
    }
}


void    
nd_trace_unselect_packet(ND_Trace *trace, int index)
{
  ND_Packet *p = NULL;
  
  if (!trace)
    return;

  trace->sel.last_valid = FALSE;

  if ((p = nd_trace_packet_get_nth(trace, index)))
    {      
      if (p->sel_next)
	{
	  if (p->sel_prev)
	    {
	      p->sel_next->sel_prev = p->sel_prev;
	      p->sel_prev->sel_next = p->sel_next;
	    }
	  else
	    {
	      trace->sel.sel = p->sel_next;
	      p->sel_next->sel_prev = NULL;
	    }
	}
      else /* Last packet in selection */
	{
	  if (p->sel_prev) /* Not single selected packet */
	    p->sel_prev->sel_next = NULL;
	  else
	    trace->sel.sel = NULL;
	}
      
      p->sel_next = p->sel_prev = NULL;
      trace->sel.size--;
    }
}


void    
nd_trace_clear_selection(ND_Trace *trace)
{
  ND_Packet *p, *p_last = NULL;
 
  if (!trace)
    return;

  /* Unselect everything in the packet structures: */

  p_last = trace->sel.sel;

  if (p_last)
    {
      for(p = p_last->sel_next; p; p = p->sel_next)
	{
	  p_last->sel_next = p_last->sel_prev = NULL;
	  p_last = p;
	}

      p_last->sel_next = p_last->sel_prev = NULL;
    }


  trace->sel.sel = NULL;
  trace->sel.size = 0;
  trace->sel.last_valid = FALSE;
}


void    
nd_trace_full_selection(ND_Trace *trace)
{
  ND_Packet *p, *p_last = NULL;
  
  if (!trace)
    return;

  /* Select everything in the packet structures: */

  if ((p = trace->pl))
    {
      p_last = p;
      trace->sel.sel = p_last;
      p_last->sel_prev = NULL;
      p_last->sel_next = NULL;
  
      for(p = p->next; p; p = p->next)
	{
	  p_last->sel_next = p;
	  p->sel_prev = p_last;
	  p->sel_next = NULL;
	  p_last = p;
	}

      trace->sel.size = trace->num_packets;
      trace->sel.last_valid = FALSE;
    }
}


void
nd_trace_packet_insert_at_index(ND_Trace *trace, ND_Packet *p, int index)
{
  ND_Packet *t;
  char       line[MAXPATHLEN];
  char      *s[2];

  if (!trace || !p || p->trace == trace || index < 0)
    return;

  D_ASSERT_PTR(trace->list);

  nd_tcpdump_get_packet_line(p, line, TRUE);
  p->trace = trace;

  s[0] = line;
  s[1] = "";

  gtk_clist_insert(GTK_CLIST(trace->list), index, s);
  
  t = nd_trace_packet_get_nth(trace, index);
  
  if (t)
    {
      /* We're not inserting at the end of the list */

      if (t->prev)
	{
	  /* We're not inserting at the beginning of the list */

	  t->prev->next = p;
	  p->prev = t->prev;
	  p->next = t;
	  t->prev = p;

	  /* trace->pl_end remains correct, because we insert
	     before the old packet at that index */
	}
      else
	{
	  /* We insert at the beginning of the list */

	  trace->pl = p;
	  p->prev = NULL;
	  p->next = trace->pl;
	  t->prev = p;
	}
    }
  else
    {
      /* We insert at the end of the list */

      trace->pl_end->next = p;
      p->prev = trace->pl_end;
      trace->pl_end = p;
    }

  trace->num_packets++;
  trace->sel.last_valid = FALSE;
  nd_trace_set_dirty(trace, TRUE);
  
  if (trace == nd_trace_registry_get_current())
    nd_gui_num_packets_set();
}


ND_Packet  *
nd_trace_packet_get_nth(const ND_Trace *trace, int n)
{
  ND_Packet *p;
  int        i = 0;
  
  if (!trace)
    return NULL;

  p = trace->pl;
  while (p)
    {
      if (i == n)
	return p;

      p = p->next;
      i++;
    }

  return NULL;
}


ND_Packet  *
nd_trace_sel_get(ND_Trace *trace, gboolean selected_only)
{
  if (trace->apply_to_all && !selected_only)
    return trace->pl;

  return trace->sel.sel;
}


ND_Packet  *
nd_trace_sel_next(ND_Trace *trace, ND_Packet *p)
{
  if (!p || !trace)
    return NULL;

  if (trace->apply_to_all)
    return p->next;

  return p->sel_next;
}


void        
nd_trace_sel_delete(ND_Trace *trace)
{
  ND_Packet        *p, *p2;
  int               index;

  if (!trace)
    return;

  if (trace->sel.size == 0)
    return;

  index = trace->num_packets - 1;

  p = trace->sel.sel;

  /* Remove lines in the gtkclist, quickly ... */
  nd_gui_list_remove_selected_rows(trace->list);

  while (p)
    {
      p2 = p;
      p  = p->sel_next;

      /* ... and clean up the packet, without the GUI updates. */
      nd_packet_delete(p2, FALSE);
    }

  if (trace == nd_trace_registry_get_current())
    nd_gui_num_packets_set();

  trace->sel.size = 0;
  trace->sel.sel  = NULL;
  trace->sel.last_valid = FALSE;

  nd_trace_set_current_packet(trace, NULL);
}


void        
nd_trace_sel_hide(ND_Trace *trace)
{
  ND_Packet        *p;
  GtkStyle         *gs;
  int               index;

  if (!trace)
    return;

  D_ASSERT_PTR(trace->list);

  gs = gtk_widget_get_style(trace->list);

  for (p = nd_trace_sel_get(trace, TRUE); p; p = p->sel_next)
    {
      index = nd_packet_get_index(p);
      p->is_hidden = 1;
      gtk_clist_set_foreground(GTK_CLIST(trace->list), index, gs->mid);
    }
}


void        
nd_trace_sel_show(ND_Trace *trace)
{
  ND_Packet        *p;
  GtkStyle         *gs;
  int               index;


  if (!trace)
    return;

  gs = gtk_widget_get_style(trace->list);
  
  for (p = nd_trace_sel_get(trace, TRUE); p; p = p->sel_next)
    {
      index = nd_packet_get_index(p);
      p->is_hidden = 0;
      gtk_clist_set_foreground(GTK_CLIST(trace->list), index, gs->text);
    }
}

guint          
nd_trace_sel_size(ND_Trace *trace)
{
  if (!trace)
    return 0;

  return trace->sel.size;
}


guint          
nd_trace_size(ND_Trace *trace)
{
  if (!trace)
    return 0;

  return trace->num_packets;
}

void        
nd_trace_set_dirty(ND_Trace *trace, gboolean dirty)
{
  if (!trace)
    return;

  if (dirty && !trace->dirty)
    {      
      trace->dirty = TRUE;

      if (trace == nd_trace_registry_get_current())
	nd_gui_modified_set(trace);
    }
  else if (!dirty && trace->dirty)
    {
      trace->dirty = FALSE;
      
      if (trace == nd_trace_registry_get_current())
	nd_gui_modified_set(trace);
    }
}


GtkNotebook   *
nd_trace_get_notebook(const ND_Trace *trace)
{
  if (!trace || !trace->tab)
    return NULL;

  return gtk_object_get_data(GTK_OBJECT(trace->tab), "notebook");
}


ND_ProtoInfo  *
nd_trace_get_proto_info(const ND_Trace *trace,
			const ND_Protocol *proto,
			guint nesting)
{
  GtkNotebook *notebook;
  const char  *key;
  
  if (!trace || !proto)
    return NULL;

  notebook = nd_trace_get_notebook(trace);
  D_ASSERT_PTR(notebook);

  if (!notebook)
    return NULL;

  key = nd_proto_inst_to_string(proto, nesting);

  return gtk_object_get_data(GTK_OBJECT(notebook), key);
}


ND_Packet     *
nd_trace_get_current_packet(const ND_Trace *trace)
{
  if (!trace)
    return NULL;

  return trace->cur_packet;
}


void
nd_trace_goto_top(const ND_Trace *trace)
{
  GtkScrolledWindow *scr;
  GtkAdjustment *adj;

  if (!trace)
    return;

  scr = gtk_object_get_data(GTK_OBJECT(trace->tab), "scrolledwin");
  D_ASSERT_PTR(scr);
  adj = gtk_scrolled_window_get_vadjustment(scr);
  
  if (adj)
    {
      adj->value = adj->lower;
      gtk_scrolled_window_set_vadjustment(scr, adj);
    }
}


void
nd_trace_goto_bottom(const ND_Trace *trace)
{
  GtkScrolledWindow *scr;
  GtkAdjustment *adj;

  if (!trace)
    return;

  scr = gtk_object_get_data(GTK_OBJECT(trace->tab), "scrolledwin");
  D_ASSERT_PTR(scr);
  adj = gtk_scrolled_window_get_vadjustment(scr);
  
  if (adj)
    {
      adj->value = adj->upper;
      gtk_scrolled_window_set_vadjustment(scr, adj);
    }
}


void         
nd_trace_set_data(ND_Trace *trace, const char *key, void *data)
{
  char *key_dup;

  if (!trace || !key)
    return;

  key_dup = g_strdup(key);

  g_hash_table_insert(trace->registry, key_dup, data);
}


void   *
nd_trace_get_data(const ND_Trace *trace, const char *key)
{
  void       *result;

  if (!trace)
    return NULL;

  result = g_hash_table_lookup(trace->registry, key);

  if (!result)
    D(("Trace table lookup for key %s failed.\n", key));

  return result;
}


void *    
nd_trace_remove_data(ND_Trace *trace, const char *key)
{
  void       *result;

  result = g_hash_table_lookup(trace->registry, key);
  g_hash_table_remove(trace->registry, key);

  return result;
}
