/*  Neutrino:  neutrino-window.c
 *
 *  Copyright (C) 2002 David A Knight <david@ritter.demon.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gtk/gtktreemodel.h>
#include <gtk/gtktreemodelsort.h>

#include <gdk/gdk.h>

#include <glib/gmarkup.h>

#include <string.h>

#include "neutrino-jukebox-view.h"
#include "nomad-jukebox-track.h"

static void neutrino_jukebox_view_add_initial_tracks( gpointer key,
						      gpointer value,
						      gpointer data );


static void neutrino_jukebox_view_add_track( NomadJukebox *jukebox,
		guint jobid, guint total_tracks,
		const NomadTrack *track, NeutrinoJukeboxView *view );
static void neutrino_jukebox_view_remove_track( NomadJukebox *jukebox,
		guint jobid, const NomadTrack *track,
		NeutrinoJukeboxView *view );
static void neutrino_jukebox_view_change_track( NomadJukebox *jukebox,
		guint jobid, const NomadTrack *orig,
		const NomadTrack *track, NeutrinoJukeboxView *view );

static void neutrino_jukebox_view_add_playlist( NomadJukebox *jukebox,
		guint jobid, guint total_tracks, const gchar *name, 
		NeutrinoJukeboxView *view );
static void neutrino_jukebox_view_remove_playlist( NomadJukebox *jukebox,
		guint jobid, const gchar *name,
		NeutrinoJukeboxView *view );

static void neutrino_jukebox_view_add_artist( NeutrinoJukeboxView *view, 
					 const NomadTrack *track );
static void neutrino_jukebox_view_remove_artist( NeutrinoJukeboxView *view, 
					    const NomadTrack *track );

static void neutrino_jukebox_view_album_add( gpointer key,
					     gpointer value,
					     gpointer data );

static void neutrino_jukebox_view_track_add( gpointer key,
					     gpointer value,
					     gpointer data );

static void neutrino_jukebox_view_do_search( gpointer key,
					     gpointer value,
					     gpointer data );

static gint compare_func( GtkTreeModel *model,
			  GtkTreeIter *a,
			  GtkTreeIter *b,
			  gpointer data );

NeutrinoJukeboxView *neutrino_jukebox_view_new( void )
{
	NeutrinoJukeboxView *view;

	view = g_new0( NeutrinoJukeboxView, 1 );

	view->artists = gtk_list_store_new( ARTIST_MAX_COL,
					    G_TYPE_STRING );

	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( view->artists ),
					      ARTIST_NAME_COL, GTK_SORT_ASCENDING );
	gtk_tree_sortable_set_sort_func( GTK_TREE_SORTABLE( view->artists ),
			ARTIST_NAME_COL, compare_func, view, NULL );
	
	view->albums = gtk_list_store_new( ALBUM_MAX_COL,
					   G_TYPE_STRING );
	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( view->albums ),
					      ALBUM_NAME_COL, GTK_SORT_ASCENDING );
	gtk_tree_sortable_set_sort_func( GTK_TREE_SORTABLE( view->albums ),
			ALBUM_NAME_COL, compare_func, view, NULL );

	view->tracks = gtk_list_store_new( TRACK_MAX_COL,
					   G_TYPE_UINT,
					   G_TYPE_STRING,
					   G_TYPE_UINT,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_STRING,
					   G_TYPE_BOOLEAN );

	gtk_tree_sortable_set_sort_column_id( GTK_TREE_SORTABLE( view->tracks ),
					      TRACK_TRACK_NUM_COL, GTK_SORT_ASCENDING );
	return view;
}

void neutrino_jukebox_view_free( NeutrinoJukeboxView *view )
{		
	/* disconnect signals */
	if( view->jukebox ) {
		g_signal_handlers_disconnect_matched( G_OBJECT( view->jukebox ),
						      G_SIGNAL_MATCH_DATA,
						      0, 0, NULL,
						      NULL,
						      view );
	}
	gtk_list_store_clear( GTK_LIST_STORE( view->artists ) );
	gtk_list_store_clear( GTK_LIST_STORE( view->albums ) );
	gtk_list_store_clear( GTK_LIST_STORE( view->tracks ) );

	/* clear hash tables */
	g_hash_table_destroy( view->artist_hash );

	g_object_unref( G_OBJECT( view->artists ) );
	g_object_unref( G_OBJECT( view->albums ) );
	g_object_unref( G_OBJECT( view->tracks ) );
	g_free( view->current_artist );
	g_free( view->current_album );
}

void neutrino_jukebox_view_set_jukebox( NeutrinoJukeboxView *view, 
					NomadJukebox *jukebox )
{
	GHashTable *table;


	if( view->jukebox ) {
		nomad_jukebox_lock( view->jukebox );

		gtk_list_store_clear( GTK_LIST_STORE( view->artists ) );
		gtk_list_store_clear( GTK_LIST_STORE( view->albums ) );
		gtk_list_store_clear( GTK_LIST_STORE( view->tracks ) );
		
		/* disconnect signals */
		g_signal_handlers_disconnect_matched( G_OBJECT( view->jukebox ),
						      G_SIGNAL_MATCH_DATA,
						      0, 0, NULL,
						      NULL,
						      view );
		/* clear hash tables */
		g_hash_table_destroy( view->artist_hash );
		view->artist_hash = NULL;

		nomad_jukebox_unlock( view->jukebox );	
	}

	/* 
	   key = artist name, value = hash table of albums 
	   
	   hash table of albums:  key = album name, value = hash table of tracks
	*/
	view->artist_hash = g_hash_table_new_full( g_str_hash, 
						g_str_equal,
						(GDestroyNotify)g_free,
						(GDestroyNotify)g_hash_table_destroy );

	view->jukebox = jukebox;

	if( jukebox ) {
		nomad_jukebox_lock( jukebox );
	
		/* connect signals */
		g_signal_connect( G_OBJECT( jukebox ), "track_add",
				  G_CALLBACK( neutrino_jukebox_view_add_track ),
				  view );
		g_signal_connect( G_OBJECT( jukebox ), "track_remove",
				  G_CALLBACK( neutrino_jukebox_view_remove_track ),
				  view );
		
		g_signal_connect( G_OBJECT( jukebox ), "track_changed",
				  G_CALLBACK( neutrino_jukebox_view_change_track ), view );

		g_signal_connect( G_OBJECT( jukebox ), "playlist_add",
				  G_CALLBACK( neutrino_jukebox_view_add_playlist ),
				  view );
		g_signal_connect( G_OBJECT( jukebox ), "playlist_remove",
				  G_CALLBACK( neutrino_jukebox_view_remove_playlist ),
				  view );

		/* request track list */
		table = nomad_jukebox_request_tracks( jukebox );
		gdk_threads_enter();
		g_hash_table_foreach( table,
				      (GHFunc)neutrino_jukebox_view_add_initial_tracks,
				      view );
		gdk_threads_leave();
		nomad_jukebox_unlock( jukebox );
	}
}

void neutrino_jukebox_view_select_artist( NeutrinoJukeboxView *view,
					  const gchar *artist )
{
	if( view->current_artist ) {
		g_free( view->current_artist );
	}

	/* build album view for selected artist */
	gtk_list_store_clear( GTK_LIST_STORE( view->albums ) );
	gtk_list_store_clear( GTK_LIST_STORE( view->tracks ) );

	if( artist ) {
		GHashTable *albums;

		view->current_artist = g_strdup( artist );

		albums = g_hash_table_lookup( view->artist_hash,
					      artist );
		g_assert( albums );

		g_hash_table_foreach( albums,
				      (GHFunc)neutrino_jukebox_view_album_add,
				      view->albums );
	} else {
		view->current_artist = NULL;
	}
}

void neutrino_jukebox_view_select_album( NeutrinoJukeboxView *view,
					 const gchar *album )
{
	/* build track view for selected album */
	GHashTable *albums;
	GHashTable *tracks;

	if( view->current_album ) {
		g_free( view->current_album );
	}

	gtk_list_store_clear( GTK_LIST_STORE( view->tracks ) );

	view->num = 0;

	if( album ) {
		view->current_album = g_strdup( album );

		albums = g_hash_table_lookup( view->artist_hash,
					      view->current_artist );
		g_assert( albums );
		
		tracks = g_hash_table_lookup( albums, album );
		g_assert( tracks );
		
		g_hash_table_foreach( tracks,
				      (GHFunc)neutrino_jukebox_view_track_add,
				      view );
	} else {
		view->current_album = NULL;
	}
}

void neutrino_jukebox_view_select_playlist( NeutrinoJukeboxView *view,
					    const gchar *playlist )
{
	GHashTable *lists;
	GSList *list;

	gtk_list_store_clear( GTK_LIST_STORE( view->tracks ) );
	
	if( playlist ) {
		lists = nomad_jukebox_request_playlists( view->jukebox );
		list = g_hash_table_lookup( lists, playlist );
		g_assert( list );
	} else {
		list = NULL;
		return;
	}
	
	view->num = 0;

	for( list = list->next; list; list = list->next ) {
		NomadTrack *track;
		guint tid;

		track = (NomadTrack *)list->data;
		tid = nomad_track_get_id( track );

		neutrino_jukebox_view_track_add( GUINT_TO_POINTER( tid ),
						 NULL,
						 view );
	}
}

gboolean neutrino_jukebox_view_search( NeutrinoJukeboxView *view,
				   NeutrinoJukeboxSearchMode mode,
				   const gchar *search_string )
{
	GHashTable *table;
	gboolean ret;
	
	table = nomad_jukebox_request_tracks( view->jukebox );

	view->mode = mode;

	view->num = 0;

	view->search = g_new0( regex_t, 1 );
	ret = FALSE;
	if( ! regcomp( view->search, search_string, REG_EXTENDED | REG_NOSUB ) ) {
		ret = TRUE;
		gtk_list_store_clear( GTK_LIST_STORE( view->tracks ) );

		g_hash_table_foreach( table,
				     (GHFunc)neutrino_jukebox_view_do_search,
				     view );
	}
	g_free( view->search );

	return ret;
}

/* static funcs */

static void neutrino_jukebox_view_add_track( NomadJukebox *jukebox,
		guint jobid, guint total_tracks,
		const NomadTrack *track, NeutrinoJukeboxView *view )
{
	if( view->jukebox != jukebox ) {
		return;
	}

	gdk_threads_enter();
	neutrino_jukebox_view_add_artist( view, track );
	gdk_threads_leave();
}

static void neutrino_jukebox_view_remove_track( NomadJukebox *jukebox,
		guint jobid, const NomadTrack *track,
		NeutrinoJukeboxView *view )
{
	if( view->jukebox != jukebox ) {
		return;
	}

	gdk_threads_enter();
	neutrino_jukebox_view_remove_artist( view, track );
	gdk_threads_leave();
}

static void neutrino_jukebox_view_change_track( NomadJukebox *jukebox,
		guint jobid, const NomadTrack *orig,
		const NomadTrack *track, NeutrinoJukeboxView *view )
{
	gchar *artist;
	gchar *album;

	gchar *tartist;
	gchar *talbum;
	gchar *title;
	gchar *genre;
	gchar *length;
	gchar *year;
	gchar *trck;

	guint id;

	GtkTreeModel *model;
	GtkTreeIter it;

	if( view->jukebox != jukebox ) {
		return;
	}

	gdk_threads_enter();

	nomad_track_get( orig, &artist, NULL, &album,
			    NULL, NULL, NULL, NULL, NULL, NULL, NULL,
			    &id );
	nomad_track_get( track, &tartist, &title, &talbum,
			    &genre, &length, &year, NULL, NULL, &trck, NULL,
			    NULL );

	if( strcmp( album, talbum ) || strcmp( artist, tartist ) ) {
		/* album, and/or artist has been changed */
		neutrino_jukebox_view_add_artist( view, track );
		neutrino_jukebox_view_remove_artist( view, orig );
	} 

	/* check if id is in the currently displayed tracks */
	/* remove the album */
	model = GTK_TREE_MODEL( view->tracks );
	
	if( gtk_tree_model_get_iter_root( model, &it ) ) {
		GValue value = { 0 };
		guint tid;
		do  {
			gtk_tree_model_get_value( model, &it,
						  TRACK_TRACK_ID_COL,
						  &value );
			tid = g_value_get_uint( &value );
			if( tid == id ) {
				gtk_list_store_set( GTK_LIST_STORE( model ),
						    &it,
						    TRACK_TITLE_COL, title,
						    TRACK_TRACK_NUM_COL,
						    atoi( trck ),
						    TRACK_GENRE_COL, genre,
						    TRACK_LENGTH_COL, length,
						    TRACK_YEAR_COL, year,
						    TRACK_PLAY_ONLY_COL,
						    nomad_track_get_play_only( track ),
						    -1 );

			}
			g_value_unset( &value );
			
		} while( gtk_tree_model_iter_next( model, &it ) );
	}

	g_free( tartist );
	g_free( talbum );
	g_free( title );
	g_free( genre );
	g_free( length );
	g_free( year );
	g_free( trck );

	g_free( artist );
	g_free( album );

	gdk_threads_leave();
}


static void neutrino_jukebox_view_add_playlist( NomadJukebox *jukebox,
		guint jobid, guint total_tracks, const gchar *name,
		NeutrinoJukeboxView *view )
{
	if( view->jukebox != jukebox ) {
		return;
	}
}

static void neutrino_jukebox_view_remove_playlist( NomadJukebox *jukebox,
		guint jobid, const gchar *name,
		NeutrinoJukeboxView *view )
{
	if( view->jukebox != jukebox ) {
		return;
	}
}

static void neutrino_jukebox_view_add_initial_tracks( gpointer key,
						      gpointer value,
						      gpointer data )
{
	NeutrinoJukeboxView *view;

	view = (NeutrinoJukeboxView*)data;

	neutrino_jukebox_view_add_artist( view, (const NomadTrack *)value );
}

static void neutrino_jukebox_view_add_artist( NeutrinoJukeboxView *view, 
					 const NomadTrack *track )
{
	GtkTreeModel *model;
	GtkTreeIter root;	
	gboolean present;
	gchar *tartist;
	gchar *talbum;
	guint id;
	GHashTable *album_hash;
	GHashTable *track_hash;
	gboolean artist_match;

	gchar *tmp;
	
	g_return_if_fail( view != NULL );
	g_return_if_fail( track != NULL );
	
	model = GTK_TREE_MODEL( view->artists );

	nomad_track_get( track, &tartist, NULL, &talbum,
			    NULL, NULL, NULL, NULL, NULL, NULL,
			    NULL, &id );

	tmp = g_markup_escape_text( tartist, strlen( tartist ) );
	g_free( tartist );
	tartist = tmp;

	tmp = g_markup_escape_text( talbum, strlen( talbum ) );
	g_free( talbum );
	talbum = tmp;

	present = FALSE;
	if( gtk_tree_model_get_iter_root( model, &root ) ) {
		/* find artist */
		GValue value = { 0 };
		const gchar *artist;

		do  {
			gtk_tree_model_get_value( model, &root,
						  ARTIST_NAME_COL,
						  &value );
			artist = g_value_get_string( &value );
			present = ( ! strcmp( tartist, artist ) );
			g_value_unset( &value );
		} while( ! present && gtk_tree_model_iter_next(model, &root) );
	}
	if( ! present ) {
		GtkTreeIter it;
		
		gtk_list_store_append( GTK_LIST_STORE( model ),
				       &it );
		gtk_list_store_set( GTK_LIST_STORE( model ), &it,
				    ARTIST_NAME_COL, tartist, -1 );

		album_hash = g_hash_table_new_full( g_str_hash, 
					g_str_equal,
					(GDestroyNotify)g_free,
					(GDestroyNotify)g_hash_table_destroy );
		g_hash_table_insert( view->artist_hash,
				     g_strdup( tartist ),
				     album_hash );
	} else {
		album_hash = g_hash_table_lookup( view->artist_hash, tartist );
	}

	/* get track hash */
	artist_match = ( view->current_artist && 
			 ! strcmp( tartist, view->current_artist ) );

	track_hash = g_hash_table_lookup( album_hash, talbum );
	if( ! track_hash ) {
		if( artist_match ) {
			GtkTreeIter it;
			gtk_list_store_append( GTK_LIST_STORE( view->albums ),
					       &it );
			gtk_list_store_set( GTK_LIST_STORE( view->albums ),
					    &it,
					    ALBUM_NAME_COL, talbum, -1 );
		}
	
		track_hash = g_hash_table_new( NULL, NULL );
		g_hash_table_insert( album_hash, g_strdup( talbum ),
				     track_hash );
	}

	if( ! g_hash_table_lookup( track_hash, GUINT_TO_POINTER( id ) ) ) {
		g_hash_table_insert( track_hash, 
				     GUINT_TO_POINTER( id ), 
				     GUINT_TO_POINTER( id ) );
	}

	if( artist_match && view->current_album &&
	    ! strcmp( talbum, view->current_album ) ) {
		/* display the track in tracks model */
		neutrino_jukebox_view_track_add( GUINT_TO_POINTER( id ),
						 NULL,
						 view );
	}

	g_free( tartist );
	g_free( talbum );
}

static void neutrino_jukebox_view_remove_artist( NeutrinoJukeboxView *view, 
					    const NomadTrack *track )
{

	GtkTreeModel *model;
	GtkTreeIter root;	
	gboolean present;
	guint tid;
	gchar *tartist;
	gchar *talbum;

	GHashTable *album_hash;
	GHashTable *track_hash;

	gchar *tmp;
	
	g_return_if_fail( view != NULL );
	g_return_if_fail( track != NULL );
	
	nomad_track_get( track, &tartist, NULL, &talbum,
			    NULL, NULL, NULL, NULL, NULL, NULL,
			    NULL, &tid );

	tmp = g_markup_escape_text( tartist, strlen( tartist ) );
	g_free( tartist );
	tartist = tmp;

	tmp = g_markup_escape_text( talbum, strlen( talbum ) );
	g_free( talbum );
	talbum = tmp;
	
	album_hash = g_hash_table_lookup( view->artist_hash, tartist );
	g_assert( album_hash );
	track_hash = g_hash_table_lookup( album_hash, talbum );
	g_assert( track_hash );
	
	/* only remove if tid is found */
	if( g_hash_table_lookup( track_hash, GUINT_TO_POINTER( tid ) ) ) {
		g_hash_table_remove( track_hash, GUINT_TO_POINTER( tid ) );
	}
	if( g_hash_table_size( track_hash ) == 0 ) {
		g_hash_table_remove( album_hash, talbum );
		track_hash = NULL;
	}
	if( g_hash_table_size( album_hash ) == 0 ) {
		g_hash_table_remove( view->artist_hash, tartist );
		album_hash = NULL;
	}

	if( view->current_artist && ! strcmp( tartist, view->current_artist ) ) {
		/* remove from tracks model, and remove album if needed */

		model = GTK_TREE_MODEL( view->tracks );

		if( gtk_tree_model_get_iter_root( model, &root ) ) {
			GValue value = { 0 };
			GSList *remove;
			GtkTreeIter *it;
			guint rid;

			remove = NULL;

			do  {
				gtk_tree_model_get_value( model, &root,
							  TRACK_TRACK_ID_COL,
							  &value );
				rid = g_value_get_uint( &value );
				if( tid == rid ) {
					it = gtk_tree_iter_copy( &root );
					remove = g_slist_prepend( remove,
								  it );

				}
				g_value_unset( &value );
				
			} while( gtk_tree_model_iter_next(model, &root) );
			while( remove ) {
				it = remove->data;
				gtk_list_store_remove( GTK_LIST_STORE( model ),
						       it );
				remove = g_slist_remove( remove, it );
				gtk_tree_iter_free( it );
			}
		}
	}
	if( ! track_hash ) {
		/* remove the album */
		model = GTK_TREE_MODEL( view->albums );

		if( gtk_tree_model_get_iter_root( model, &root ) ) {
			GValue value = { 0 };
			const gchar *album;
			do  {
				gtk_tree_model_get_value( model, &root,
							  ALBUM_NAME_COL,
							  &value );
				album = g_value_get_string( &value );
				if( ! strcmp( talbum, album ) ) {
					g_value_unset( &value );
					gtk_list_store_remove( GTK_LIST_STORE( model ),
							       &root );
					break;
				}
				g_value_unset( &value );
				
			} while( gtk_tree_model_iter_next(model, &root) );
		}
	}

	model = GTK_TREE_MODEL( view->artists );
	present = FALSE;
	if( gtk_tree_model_get_iter_root( model, &root ) ) {
		/* find artist */
		GValue value = { 0 };
		const gchar *artist;

		do  {
			gtk_tree_model_get_value( model, &root,
						  ARTIST_NAME_COL,
						  &value );
			artist = g_value_get_string( &value );
			present = ( ! strcmp( tartist, artist ) );
			g_value_unset( &value );
		} while( ! present && gtk_tree_model_iter_next(model, &root) );
	}
      	if( present && ! album_hash ) {
		gtk_list_store_remove( GTK_LIST_STORE( model ), &root );
		g_hash_table_remove( view->artist_hash, tartist );
	}

	g_free( tartist );
	g_free( talbum );
}


static void neutrino_jukebox_view_album_add( gpointer key,
					     gpointer value,
					     gpointer data )
{
	GtkTreeModel *model;
	GtkTreeIter it;

	model = GTK_TREE_MODEL( data );

	gtk_list_store_append( GTK_LIST_STORE( model ), &it );
	gtk_list_store_set( GTK_LIST_STORE( model ), &it,
			    ALBUM_NAME_COL,
			    key, -1 );
}

static void neutrino_jukebox_view_track_add( gpointer key,
					     gpointer value,
					     gpointer data )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	NomadTrack *track;
	NeutrinoJukeboxView *view;

	gchar *title;
	gchar *trck;
	gchar *genre;
	gchar *length;
	gchar *year;
	gchar *size;
	gchar *codec;

	view = (NeutrinoJukeboxView*)data;

	model = GTK_TREE_MODEL( view->tracks );

	view->num ++;

	gtk_list_store_append( GTK_LIST_STORE( model ), &it );

	track = nomad_jukebox_request_track( view->jukebox,
						GPOINTER_TO_UINT( key ) );

	nomad_track_get( track, NULL, &title, NULL,
			    &genre, &length, &year, &size, &codec,
			    &trck, NULL, NULL );

	gtk_list_store_set( GTK_LIST_STORE( model ), &it,
			    TRACK_TRACK_ID_COL, GPOINTER_TO_UINT( key ),
			    TRACK_TITLE_COL, title,
			    TRACK_TRACK_NUM_COL, atoi( trck ),
			    TRACK_GENRE_COL, genre,
			    TRACK_LENGTH_COL, length,
			    TRACK_YEAR_COL, year,
			    TRACK_SIZE_COL, size,
			    TRACK_CODEC_COL, codec,
			    TRACK_PLAY_ONLY_COL,
			    nomad_track_get_play_only( track ),
			    -1 );

	nomad_track_free( track );
	
	g_free( title );
	g_free( trck );
	g_free( genre );
	g_free( length );
	g_free( year );
	g_free( size );
	g_free( codec );
}

static void neutrino_jukebox_view_do_search( gpointer key,
					     gpointer value,
					     gpointer data )
{
	NeutrinoJukeboxView *view;
	NomadTrack *track;
	gchar *field;
	guint tid;

	view = (NeutrinoJukeboxView*)data;

	track = (NomadTrack*)value;

	tid = nomad_track_get_id( track );

	switch( view->mode ) {
	case SEARCH_ARTIST:
		field = nomad_track_get_artist( track );
		break;
	case SEARCH_ALBUM:
		field = nomad_track_get_album( track );
		break;
	case SEARCH_TITLE:
		field = nomad_track_get_title( track );
		break;
	case SEARCH_GENRE:
		field = nomad_track_get_genre( track );
		break;
	case SEARCH_YEAR:
		field = nomad_track_get_year( track );
		break;
	default:
		field = NULL;
		g_assert( FALSE );
		break;
	}

	if( ! regexec( view->search, field, 0, NULL, 0 ) ) {
		neutrino_jukebox_view_track_add( GUINT_TO_POINTER( tid ),
						 NULL,
						 view );
	}

	g_free( field );
}

static gint compare_func( GtkTreeModel *model,
			  GtkTreeIter *a,
			  GtkTreeIter *b,
			  gpointer data )
{
	NeutrinoJukeboxView *view;
	gchar *aname;
	gchar *bname;
	gint ret;
	
	view = (NeutrinoJukeboxView*)data;
	ret = 0;


	if( ret == 0 ) {
		gtk_tree_model_get( model, a,
				    0, &aname,
				    -1 );
		gtk_tree_model_get( model, b,
				    0, &bname,
				    -1 );
		ret = g_strcasecmp( aname, bname );
	}

	g_free( aname );
	g_free( bname );
	
	return ret;
}

