/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "AsynchMsgs_ASN1.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_SEQUENCE(WORKING_REQUEST) = {
	ASN1_SIMPLE(WORKING_REQUEST, SenderName, ASN1_UTF8STRING), 
	ASN1_SIMPLE(WORKING_REQUEST, sender, X509_PUBKEY), 
	ASN1_SIMPLE(WORKING_REQUEST, transactionID, ASN1_OCTET_STRING), 
	ASN1_SIMPLE(WORKING_REQUEST, request, NEWPKI_REQUEST), 
}ASN1_SEQUENCE_END(WORKING_REQUEST)
ASN1_SEQUENCE(STACK_REQUEST) = {
	ASN1_SIMPLE(STACK_REQUEST, id, ASN1_INTEGER),
	ASN1_SIMPLE(STACK_REQUEST, req, CRYPTED_NEWPKI_REQUEST),
}ASN1_SEQUENCE_END(STACK_REQUEST)
WorkingRequest WorkingRequest::EmptyInstance;
bool WorkingRequest::set_sendername(const mString & c_sendername)
{
	m_sendername = c_sendername;
	return true;
}

const mString & WorkingRequest::get_sendername() const
{
	return m_sendername;
}

mString & WorkingRequest::get_sendername()
{
	return m_sendername;
}

bool WorkingRequest::set_request(const NewpkiRequest & c_request)
{
	m_request = c_request;
	return true;
}

const NewpkiRequest & WorkingRequest::get_request() const
{
	return m_request;
}

NewpkiRequest & WorkingRequest::get_request()
{
	return m_request;
}

bool WorkingRequest::set_sender(const X509_PUBKEY * c_sender)
{
	if(m_sender)
		ASN1_item_free((ASN1_VALUE*)m_sender, ASN1_ITEM_rptr(X509_PUBKEY));
	m_sender = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)c_sender);
	if(!m_sender)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

const X509_PUBKEY * WorkingRequest::get_sender() const
{
	if(!m_sender)
		((WorkingRequest*)this)->m_sender = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_sender;
}

X509_PUBKEY * WorkingRequest::get_sender()
{
	if(!m_sender)
		m_sender = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
	return m_sender;
}

bool WorkingRequest::set_transactionid(const Asn1OctetString & c_transactionid)
{
	m_transactionid = c_transactionid;
	return true;
}

const Asn1OctetString & WorkingRequest::get_transactionid() const
{
	return m_transactionid;
}

Asn1OctetString & WorkingRequest::get_transactionid()
{
	return m_transactionid;
}

bool WorkingRequest::to_PEM(mString & PemDatas) const
{
	WORKING_REQUEST * c_localvar = NULL;
	if(!give_Datas(&c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Private_toPEM("WORKING NEWPKI REQUEST", get_ASN1_ITEM(), (ASN1_VALUE*)c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

bool WorkingRequest::from_PEM(const mString & PemDatas)
{
	WORKING_REQUEST * c_localvar = NULL;
	if(!Private_fromPEM("WORKING NEWPKI REQUEST", get_ASN1_ITEM(), (ASN1_VALUE**)&c_localvar, PemDatas))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!load_Datas(c_localvar))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)c_localvar, get_ASN1_ITEM());
	return true;
}

WorkingRequest::WorkingRequest():NewPKIObject()
{
	resetAll();
}

WorkingRequest::WorkingRequest(const WorkingRequest & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

WorkingRequest::~WorkingRequest()
{
	Clear();
}

void WorkingRequest::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void WorkingRequest::freeAll()
{
	if(m_sender)
	{
		ASN1_item_free((ASN1_VALUE*)m_sender, ASN1_ITEM_rptr(X509_PUBKEY));
	}
}

void WorkingRequest::resetAll()
{
	m_sendername = "";
	m_request.Clear();
	m_sender = NULL;
	m_transactionid.Clear();
}

bool WorkingRequest::load_Datas(const WORKING_REQUEST * Datas)
{
	Clear();
	if(Datas->SenderName)
	{
		m_sendername = Datas->SenderName;
	}
	if(Datas->request)
	{
		if(!m_request.load_Datas(Datas->request))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->sender)
	{
		if(m_sender)
			ASN1_item_free((ASN1_VALUE*)m_sender, ASN1_ITEM_rptr(X509_PUBKEY));
		m_sender = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), Datas->sender);
		if(!m_sender)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	if(Datas->transactionID)
	{
		if(!m_transactionid.load_Datas(Datas->transactionID))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool WorkingRequest::give_Datas(WORKING_REQUEST ** Datas) const
{
	if(!(*Datas) && !(*Datas = (WORKING_REQUEST*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->SenderName && !((*Datas)->SenderName = (ASN1_UTF8STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_UTF8STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_sendername.c_ASN1_UTF8STRING(&(*Datas)->SenderName))
	{
		ASN1_UTF8STRING_free((*Datas)->SenderName);
		(*Datas)->SenderName = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->request && !((*Datas)->request = (NEWPKI_REQUEST*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_REQUEST))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_request.give_Datas(&(*Datas)->request))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->request, ASN1_ITEM_rptr(NEWPKI_REQUEST));
		(*Datas)->request = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(m_sender)
	{
		if((*Datas)->sender)
			ASN1_item_free((ASN1_VALUE*)(*Datas)->sender, ASN1_ITEM_rptr(X509_PUBKEY));
		if(!((*Datas)->sender = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (ASN1_VALUE*)m_sender)))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
			return false;
		}
	}
	else
	{
		if(!(*Datas)->sender)
		{
			(*Datas)->sender = (X509_PUBKEY*)ASN1_item_new(ASN1_ITEM_rptr(X509_PUBKEY));
			if(!(*Datas)->sender)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
		}
	}
	if(!(*Datas)->transactionID && !((*Datas)->transactionID = (ASN1_OCTET_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_OCTET_STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_transactionid.give_Datas(&(*Datas)->transactionID))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->transactionID, ASN1_ITEM_rptr(ASN1_OCTET_STRING));
		(*Datas)->transactionID = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool WorkingRequest::operator=(const WorkingRequest & other)
{
	Clear();
	m_sendername = other.m_sendername;
	m_request = other.m_request;
	if(other.m_sender)
	{
		if(m_sender)
			ASN1_item_free((ASN1_VALUE*)m_sender, ASN1_ITEM_rptr(X509_PUBKEY));
		m_sender = (X509_PUBKEY*)ASN1_item_dup(ASN1_ITEM_rptr(X509_PUBKEY), (void*)other.m_sender);
		if(!m_sender)
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_transactionid = other.m_transactionid;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * WorkingRequest::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(WORKING_REQUEST);
}
StackRequest StackRequest::EmptyInstance;
bool StackRequest::set_id(unsigned long c_id)
{
	m_id = c_id;
	return true;
}

unsigned long StackRequest::get_id() const
{
	return m_id;
}

bool StackRequest::set_req(const CryptedNewpkiRequest & c_req)
{
	m_req = c_req;
	return true;
}

const CryptedNewpkiRequest & StackRequest::get_req() const
{
	return m_req;
}

CryptedNewpkiRequest & StackRequest::get_req()
{
	return m_req;
}

StackRequest::StackRequest():NewPKIObject()
{
	resetAll();
}

StackRequest::StackRequest(const StackRequest & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

StackRequest::~StackRequest()
{
	Clear();
}

void StackRequest::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void StackRequest::freeAll()
{
}

void StackRequest::resetAll()
{
	m_id = 0;
	m_req.Clear();
}

bool StackRequest::load_Datas(const STACK_REQUEST * Datas)
{
	Clear();
	if(Datas->id)
	{
		m_id = ASN1_INTEGER_GET(Datas->id);
	}
	if(Datas->req)
	{
		if(!m_req.load_Datas(Datas->req))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool StackRequest::give_Datas(STACK_REQUEST ** Datas) const
{
	if(!(*Datas) && !(*Datas = (STACK_REQUEST*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->id && !((*Datas)->id = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->id, m_id) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->id);
		(*Datas)->id = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->req && !((*Datas)->req = (CRYPTED_NEWPKI_REQUEST*)ASN1_item_new(ASN1_ITEM_rptr(CRYPTED_NEWPKI_REQUEST))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_req.give_Datas(&(*Datas)->req))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->req, ASN1_ITEM_rptr(CRYPTED_NEWPKI_REQUEST));
		(*Datas)->req = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool StackRequest::operator=(const StackRequest & other)
{
	Clear();
	m_id = other.m_id;
	m_req = other.m_req;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * StackRequest::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(STACK_REQUEST);
}
