/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// RepStore.h: interface for the RepStore class.
//
//////////////////////////////////////////////////////////////////////

#ifndef REPSTORE_H
#define REPSTORE_H

#include "NewPKIStore.h"
#include <ReadersWriter.h>
#include <PEM_DER.h>
#include <HashCorrelation.h>
#include "RepStore_ASN1.h"

typedef enum REP_OBJECT_STATE
{
	REP_OBJECT_STATE_ACTIVE=1,
	REP_OBJECT_STATE_DELETED=2
};


/*!
	This class is the store for a Repository
*/
class RepStore : public NewPKIStore
{
public:
	/*! \brief This is the constructor.
	 *  \param EntityName [IN] The name of the entity.
	 *  \param e [IN] The ENGINE, can be NULL.
	 */
	RepStore(const mString & EntityName, ENGINE * e);

	/*! \brief This is the destructor.
	 */
	virtual ~RepStore();

	bool CreateTables(const SQL_Connection * DbConn);

	/*! \brief This function creates the intial datas.
	 *  \return true on success, false on failure.
	 */
	bool Create();

	/*! \brief This function loads the intial datas.
	 *  \return true on success, false on failure.
	 */
	bool Load();

	/*! \brief This function returns all the known responses.
	 *  \param tids [OUT] All the known responses.
	 *  \param index [IN] The index.
	 *  \param max [IN] The max number of objects.
	 *  \return true on success, false on failure.
	 */
	bool GetKnownResponses(TransactionIds & tids, unsigned long index, unsigned long max);
	
	/*! \brief This function returns all the known requests.
	 *  \param tids [OUT] All the known requests.
	 *  \param index [IN] The index.
	 *  \param max [IN] The max number of objects.
	 *  \return true on success, false on failure.
	 */
	bool GetKnownRequests(TransactionIds & tids, unsigned long index, unsigned long max);
	
	/*! \brief This function function stores a request received from another repository.
	 *  \param request [IN] The new request.
	 *  \param Status [OUT] The status.
	 *  \return true on success, false on failure.
	 */
	bool InsertRequest(const CryptedNewpkiRequest & request, int & Status);
	
	/*! \brief This function function stores a response received from another entity.
	 *  \param response [IN] The new response.
	 *  \param Status [OUT] The status.
	 *  \return true on success, false on failure.
	 */
	bool InsertResponse(const CryptedNewpkiResponse & response, int & Status);
	
	/*! \brief This function function returns a list of requests.
	 *  \param Requests [OUT] The list of requests.
	 *  \param pubkey [IN] The recipient public key, can be NULL when synchronizing with another repository.
	 *  \param ListIds [IN] A list of TID, which the user wants or doesn't want, depending on inList.
	 *  \param inList [IN] Is ListIds a list of requests the user want, or doesn't want.
	 *  \return true on success, false on failure.
	 */
	bool GetRequests(CryptedNewpkiRequests & Requests, const EVP_PKEY *pubkey, const TransactionIds & ListIds, bool inList);
	
	/*! \brief This function function returns a list of responses.
	 *  \param Responses [OUT] The list of responses.
	 *  \param pubkey [IN] The recipient public key, can be NULL when synchronizing with another repository.
	 *  \param ListIds [IN] A list of TID, which the user wants or doesn't want, depending on inList.
	 *  \param inList [IN] Is ListIds a list of responses the user want, or doesn't want.
	 *  \return true on success, false on failure.
	 */
	bool GetResponses(CryptedNewpkiResponses & Responses, const EVP_PKEY *pubkey, const TransactionIds & ListIds, bool inList);
	
	/*! \brief This function checks if a request is already known.
	 *  \param transactionID [IN] The request's transaction ID.
	 *  \param exists [OUT] true when exists, false when doesn't exist.
	 *  \return true on success, false on failure.
	 */
	bool RequestExists(const Asn1OctetString & transactionID, bool & exists);
	
	/*! \brief This function checks if a response is already known.
	 *  \param transactionID [IN] The response's transaction ID.
	 *  \param exists [OUT] true when exists, false when doesn't exist.
	 *  \param Status [OUT] The status (only valid when exists is true).
	 *  \return true on success, false on failure.
	 */
	bool ResponseExists(const Asn1OctetString & transactionID, bool & exists, int & Status);

	/*! \brief This function deletes a response.
	 *  \param ClientCert [IN] The certificate of the requesting entity.
	 *  \param transactionID [IN] The response's transaction ID.
	 *  \param VerifyRcpt [IN] Should we verify that the ClientCert is the recipient of the response.
	 *  \param Status [OUT] The status.
	 *  \return true on success, false on failure.
	 */
	bool DeleteResponse(const PKI_CERT & ClientCert, const Asn1OctetString & transactionID, bool VerifyRcpt, int & Status);

private:
	#define REPSTORE_REQUESTS_TABLE			"requests"
	#define REPSTORE_RESPONSES_TABLE		"responses"

	#define REPSTORE_CREATE_1				"create table "REPSTORE_REQUESTS_TABLE" (id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, transactionID VARCHAR(41) NOT NULL UNIQUE, recipient VARCHAR(41) NOT NULL, request_PEM LONGBLOB NOT NULL, signature LONGBLOB NOT NULL, state INT UNSIGNED NOT NULL, INDEX (state), INDEX(recipient));"
	#define REPSTORE_CREATE_2				"create table "REPSTORE_RESPONSES_TABLE" (id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, transactionID VARCHAR(41) NOT NULL UNIQUE, recipient VARCHAR(41) NOT NULL, response_PEM LONGBLOB NOT NULL, signature LONGBLOB NOT NULL, state INT UNSIGNED NOT NULL, INDEX (state), INDEX(recipient));"
	
	#define REPSTORE_INSERT_REQ				"INSERT INTO "REPSTORE_REQUESTS_TABLE" (transactionID, recipient, request_PEM, signature, state) VALUES ('%s', '%s', '%s', '%s', %d);"
	#define REPSTORE_GET_REQS				"SELECT request_PEM, state FROM "REPSTORE_REQUESTS_TABLE" %s;"
	#define REPSTORE_SEARCH_REQ				"SELECT transactionID, state FROM "REPSTORE_REQUESTS_TABLE" WHERE transactionID='%s';"
	#define REPSTORE_REQ_EXISTS				"SELECT id FROM "REPSTORE_REQUESTS_TABLE" WHERE transactionID='%s';"
	#define REPSTORE_GET_KNOWN_REQS			"SELECT transactionID FROM "REPSTORE_REQUESTS_TABLE" WHERE state=%d LIMIT %ld,%ld;"
	#define REPSTORE_SET_REQ_STATE			"UPDATE "REPSTORE_REQUESTS_TABLE" SET request_PEM='', signature='', state=%d WHERE transactionID='%s';"
	#define REPSTORE_GET_REQS_LIMIT			"SELECT transactionID, signature FROM "REPSTORE_REQUESTS_TABLE" LIMIT %d,%d;"


	#define REPSTORE_INSERT_RESP			"INSERT INTO "REPSTORE_RESPONSES_TABLE" (transactionID, recipient, response_PEM, signature, state) VALUES ('%s', '%s', '%s', '%s', %d);"
	#define REPSTORE_GET_RESPS				"SELECT response_PEM, state FROM "REPSTORE_RESPONSES_TABLE" %s;"
	#define REPSTORE_GET_RESP_STATE			"SELECT state, response_PEM FROM "REPSTORE_RESPONSES_TABLE" WHERE transactionID='%s';"
	#define REPSTORE_SEARCH_RESP			"SELECT id, state FROM "REPSTORE_RESPONSES_TABLE" WHERE transactionID='%s';"
	#define REPSTORE_GET_KNOWN_RESPS		"SELECT transactionID FROM "REPSTORE_RESPONSES_TABLE" WHERE state=%d LIMIT %ld,%ld;"
	#define REPSTORE_SET_RESP_STATE			"UPDATE "REPSTORE_RESPONSES_TABLE" SET response_PEM='', signature='', state=%d WHERE transactionID='%s';"
	#define REPSTORE_GET_RESPS_LIMIT		"SELECT transactionID, signature FROM "REPSTORE_RESPONSES_TABLE" LIMIT %d,%d;"


	#define CRYPTED_NEWPKI_REQUEST_sign(x,sig,pkey) \
		ASN1_item_sign(CryptedNewpkiRequest::get_ASN1_ITEM(), sig.get_sigAlg(), NULL, sig.get_signature(), (char *)x, pkey, EVP_sha1())

	#define CRYPTED_NEWPKI_REQUEST_verify(x,sig,pkey) \
		ASN1_item_verify(CryptedNewpkiRequest::get_ASN1_ITEM(), sig.get_sigAlg(), sig.get_signature(),(char *)x, pkey)


	#define CRYPTED_NEWPKI_RESPONSE_sign(x,sig,pkey) \
		ASN1_item_sign(CryptedNewpkiResponse::get_ASN1_ITEM(), sig.get_sigAlg(), NULL, sig.get_signature(), (char *)x, pkey, EVP_sha1())

	#define CRYPTED_NEWPKI_RESPONSE_verify(x,sig,pkey) \
		ASN1_item_verify(CryptedNewpkiResponse::get_ASN1_ITEM(), sig.get_sigAlg(), sig.get_signature(),(char *)x, pkey)


	bool SqlToRequest(long i, CryptedNewpkiRequest & Request, SQL & sql);
	bool SqlToResponse(long i, CryptedNewpkiResponse & Response, SQL & sql);
};

#endif
