/* $Id: nws_memory.c,v 1.195 2005/08/04 00:18:32 graziano Exp $ */

#include "config_nws.h"

#include <errno.h>           /* errno */
#include <stdio.h>           /* sprintf() */
#include <unistd.h>          /* getopt() stat() */
#include <stdlib.h>
#include <string.h>          /* strchr() strstr() */

#include "protocol.h"
#include "nws_state.h"
#include "host_protocol.h"
#include "diagnostic.h"
#include "strutil.h"
#include "dnsutil.h"
#include "osutil.h"
#include "messages.h"
#include "nws_memory.h"
#include "skills.h"
#include "nws_db.h"
#include "nws_daemon.h"

/*
 * This program implements the NWS memory host.  See nws_memory.h for supported
 * messages.
 */

/* we keep a list of series we know about, so that we can register them
 * with the nameserver. We have 2: one for the orphaned series and one
 * for the registered series: the orphaned will disappears when the new
 * series kicks in. */
static registrations *mySeries;		/* where we keep series registrations */
static registrations *orphanedSeries;	/* where we keep series registrations */

/* performance data */
static double fetchTime;		/* time spent serving FETCH */
static double oldStoreTime;		/* time spent serving STORE */
static double storeTime;		/* time spent serving the new STORE */
static double autoFetchTime;		/* time spent serving autofetching */
static unsigned long fetchQuery;
static unsigned long oldStoreQuery;
static unsigned long storeQuery;

/*
 * Information about a registered auto-fetch request. The list of series
 * the client is interested in and the socket is kept here.
 */
typedef struct {
	registrations *reg;
	Socket clientSock;
} AutoFetchInfo;


/*
 * Module globals.  #autoFetches# is the list of registered auto-fetch requests
 * that we're still servicing; #autoFetchCount# is the length of this list.
 * #fileSize# contains the maximum number of records allowed in files.
 */
AutoFetchInfo *autoFetches = NULL;
static size_t autoFetchCount = 0;
static size_t fileSize;

/* select which backend to use to store data: so far localdir is the most
 * stable and tested one. */
typedef enum {LOCALDIR, NETLOGGER, POSTGRES} memType;
static memType backType = LOCALDIR;

/*
 * Called when an operation fails.  Cleans up any bad connections and, if we've
 * run out, frees up a connection so that others may connect.
 */
static void
CheckConnections(void) {
	if(CloseDisconnections() == 0) {
		CloseConnections(0, 1, 1);
	}
}

/*
 * this function goes through a string which contains states separated
 * by '\n' and check if they are valid registrations. It starts from the
 * end of the string, check if the name is in #good# (the good list of
 * registrations) otherwise it adds to #orphan#. Checks only #howMany#
 * entries at a time. Returns 1 if more entries need to be processed.
 */
static int
CheckOldNames(	char *names,
		registrations *good,
		registrations *orphan,
		int howMany) {
	char *where, *tmp;
	Object obj;
	int nskill, i, k, done;
	MeasuredResources m;
	KnownSkills t;

	/* sanity check */
	if (orphan == NULL || good == NULL) {
		ERROR("CheckOldNames: NULL parameter\n");
		return 0;
	}
	if (howMany < 0) {
		howMany = 10;
	}
	if (names == NULL) {
		/* we are done */
		return 0;
	}

	for (where = names + strlen(names); where > names; *(--where) = '\0') {
		/* look for the beginning of the filename */
		while (where >= names &&  *where != '\n') {
			where--;
		}
		where++;

		/* let's check if we have just a newline */
		if (strlen(where) <= 0) {
			continue;
		}

		/* let's check the filename/table */
		switch (backType) {
		case LOCALDIR:
			if (!CheckFileName(where)) {
				INFO1("CheckOldNames: %s is not a valid NWS filename\n", where);
				continue;
			}
			break;

		case POSTGRES:
			if (!CheckTable(where)) {
				INFO1("CheckOldNames: %s is not a valid NWS table\n", where);
				continue;
			}
			break;
		
		case NETLOGGER:
			INFO("CheckOldNames: not implemented for netlogger\n");
			return 0;
			break;
		}

		/* if the series is already registered in the good lists,
		 * we skip it */
		if (SearchForName(good, where, 1, &i)) {
			continue;
		}

		/* if we already have it in the orphaned series we skip: 
		 * we need it anyway to find where to insert the object */
		if (SearchForName(orphan, where, 1, &i)) {
			continue;
		}

		/* we need to insert it: let's create the object first
		 * then add it */
		obj = NewObject();
		AddNwsAttribute(&obj, "name", where);
		AddNwsAttribute(&obj, "memory", EstablishedRegistration());
		AddNwsAttribute(&obj, "objectclass", "nwsSeries");

		/* let's find which resource we are dealing with */
		for (m = 0; m < RESOURCE_COUNT; m++) {
			if (strstr(where, ResourceName(m)) != NULL) {
				break;
			}
		}
		if (m >= RESOURCE_COUNT) {
			INFO1("CheckOldNames: unknown resource in %s\n", where);
			continue;
		}

		/* now, we are checking if the resource as a target
		 * options: if so the target is at the end of the name,
		 * after the options. This is from Ye  code. Thanks Ye! */ 
		/* to do that let's find the skill responsible for the
		 * resource */
		done = 0;
		for (t = 0; t < SKILL_COUNT; t++) {
			const MeasuredResources *resources;

			if (!SkillResources(t, &resources, &nskill)) {
				INFO("CheckOldNames: failed to get info on skill\n");
				continue;
			}
			for (k = 0; k < nskill; k++) {
				if (resources[k] == m) {
					/* we found the right skill:
					 * let's get out of here */
					done = 1;
					break;
				}
			}
			if (done) {
				break;
			}
		}

		/* if we didn't find the skill, it's a very bad news. */
		if (t >= SKILL_COUNT) {
			WARN1("CheckOldNames: unable to find skill for resource %s\n", ResourceName(m));
			continue;
		}

		/* now let's look if we have a target option: clique
		 * control adds it. */
		if (SkillAvailableForControl(t, "", CLIQUE_CONTROL)) {
			const char *options;
			options = SkillSupportedOptions(t);
			if (options == NULL) {
				/* I guess we don't have the target */
				continue;
			}
			/* let's count how many options we have (the are
			 * comma separated so there are 1 + # of commas) */
			for (done = 1; *options != '\0'; options++) {
				if (*options == ',') {
					done++;
				}
			}

			/* now let's get the target: it is done commas
			 * away from the resource name */
			tmp = strstr(where, ResourceName(m));
			/* we have an extra '.' (the one after
			 * the resource name */
			for (done++; *tmp != '\0'; tmp++) {
				if (*tmp == '.') {
					done--;
				}
				if (done == 0) {
					/* we got it: let's pass the dot */
					tmp++;
					AddNwsAttribute(&obj, "target", tmp);
					break;
				}
			}
		}
		
		/* now let's add the resource and the host */
		AddNwsAttribute(&obj, "resource", ResourceName(m));

		/* if we have the resource we can take an educate
		 * guess on what the host is */
		tmp = strstr(where, ResourceName(m));
		if (tmp != NULL) {
			tmp--;
			if (*tmp == '.') {
				*tmp = '\0';
				AddNwsAttribute(&obj, "host", where);
			}
		}

		InsertRegistration(orphan, obj, 10, i);
		FreeObject(&obj);

		/* done with this name */
		*where = '\0';

		if (--howMany <= 0) {
			/* done for this round */
			break;
		}
	}

	return (howMany <= 0);
}


/* 
 * this function looks for series that needs to be registered (expiration
 * will tell) with the nameserver. We register only #hoeMany# of them at a
 * time. Returns 1 if more registration work needs to be done.
 */
static int
RegisterSeries(	registrations *r, 
		int howMany) {
	int i, j;
	unsigned long now;
	
	/* sanity check */
	if (r->howMany <= 0) {
		return 0;
	}

	now = (unsigned long)CurrentTime();

	for (i = 0, j = 0; j < r->howMany; j++) {
		if (r->expirations[j] < now) {
			/* need to be registered */
			if (RegisterWithNameServer(r->vals[j], DEFAULT_HOST_BEAT * 10)) {
				r->expirations[j] = now + DEFAULT_HOST_BEAT * 5;
				/* we register only howMany series at a time */
				if (i++ >= howMany) 
					break;
			}
		}
	}

	return  (j < r->howMany);
}

/* todo list when we exit */
static int
MyExit(void) {
	char filter[255];

	LOG("MyExit: unregistering series.\n");
	snprintf(filter, 255, "(memory=%s)", EstablishedRegistration());
	UnregisterFilter(filter);

	return 1;
}


/*
 * this functions gets a #registration# of a Series in input and looks for
 * it in the lists of Series names. If not found it adds it. 
 */
static int
StoreSeriesName(char *registration) {
	int ind;
	Object obj;
	char *name;

	/* sanity check */
	if (registration == NULL) {
		ERROR("StoreSeriesName: NULL registration\n");
		return 0;
	}

	/* we'll need the name later */
	name = NwsAttributeValue_r(FindNwsAttribute(registration, "name"));
	if (name == NULL) {
		ERROR1("StoreSeriesName: object without a name (%s)!\n", registration);
		return 0;
	}

	/* before doing anything else, let's set straight who is the
	 * memory here */
	obj = strdup(registration);
	DeleteNwsAttribute(&obj, "memory");
	AddNwsAttribute(&obj, "memory", EstablishedRegistration());

	/* we need to see if there is similar registration in the
	 * orphaned list: if so remove it and add it to the good list */
	if (SearchForName(orphanedSeries, name, 1, &ind)) {
		/* we found it: remove it from orphaned ... */
		DeleteRegistration(orphanedSeries, ind);
		/* .. and from the nameserver */
		UnregisterObject(name);
	}
	FREE(name);

	/* let's look for the registration in the good list */
	if (!SearchForObject(mySeries, obj, &ind)) {
		InsertRegistration(mySeries, obj, 10, ind);
	}
	FREE(obj);

	return 1;
}

/* do the real transfer of a series. The string we get a tab-separated
 * list of series, preceded by the memory (in host:port format). 
 */
static int
DoTransfer(	const char *contents) {
	char series[255];
	struct host_cookie mem;
	int i;
	size_t j;
	NWSAPI_Measurement *meas;

	/* sanoty check */
	if (contents == NULL || strlen(contents) < 5) {
		ERROR("DoTransfer: invalid parameter\n");
		return 0;
	}

	/* the first element is the memory */
	if (!GETWORD(series, contents, &contents)) {
		ERROR("DoTransfer: error reading remote memory info\n");
		return 0;
	}
	Host2Cookie(series, DefaultHostPort(MEMORY_HOST), &mem);
	if (!ConnectToHost(&mem, NULL)) {
		ERROR("DoTransfer: failed to connect to remote memory\n");
		return 0;
	}

	/* let's allocate the space for the measurements */
	meas = (NWSAPI_Measurement *)MALLOC(fileSize * sizeof(NWSAPI_Measurement), 1);

	/* now let's go though all the series we might have */
	while(GETWORD(series, contents, &contents)) {
		/* let's see if we already have this series */
		if (SearchForName(mySeries, series, 1, &i)) {
			WARN1("DoTransfer: series already present (%s)\n", series);
			continue;
		}

		/* get the experiments from the memory */
		if (!LoadExperiments(&mem, series, meas, fileSize, 1, &j, NULL, -1)) {
			WARN1("DoTransfer: couldn't get old data for %s\n", series);
			continue;
		}

		/* let's go thorugh the measurments */
		for (i = 0; i < j; i++) {

		}
	}
	FREE(meas);

	return 1;
}

/*
 * A "local" function of ProcessRequest().  Implements the MEMORY_CLEAN service
 * by deleting all files in the memory directory that have not been accessed
 * within the past #idle# seconds.  Returns 1 if successful, else 0.
 */
static int
DoClean(	const char *series,
		unsigned long idle) {
	int ret, i;
	char tmp[255], *s, *t;
	const char *ptr;

	/* if series is empty, we have to operate on all the series we
	 * are responsible */
	if (series == NULL || strlen(series) <= 2) {
		s = GenerateList(mySeries);
		if (s) {
			i = strlen(s);
		} else {
			i = 0;
		}
		t = GenerateList(orphanedSeries);
		if (t) {
			REALLOC(s, i + strlen(t) + 2, 1);
			if (i > 0) { 
				strcat(s, "\t");
				memmove(s + i + 1, t, strlen(t) + 1);
			}
			FREE(t);
		}
	} else {
		s = strdup(series);
		if (s == NULL) {
			ERROR("DoClean: out of memory\n");
			return 0;
		}
	}

	/* let's go through all the series and clean them up */
	for (ret = 0, ptr = s; GETWORD(tmp, ptr, &ptr);) {
		if (strlen(tmp) <= 1) {
			continue;
		}

		/* now let's check that we are indeed responsible of this
		 * series: we don't want to delete random files */
		if (!SearchForName(orphanedSeries, tmp, 1, &i) &&
				!SearchForName(mySeries, tmp, 1, &i)) {
			WARN("DoClean: unknown series, not cleaning\n");
			continue;
		}

		i = 0;
		switch (backType) {
		case LOCALDIR:
			i = CleanLocalStates(tmp, idle);
			break;

		case POSTGRES:
			INFO("DoClean: no cleaning implemented when using database\n");
			break;
	
		case NETLOGGER:
			INFO("DoClean: no cleaning implemented when using netlooger\n");
			break;
		}

		/* if we deleted a series, we need to remove the
		 * corresponding registration too */
		if (i) {
			LOG1("DoClean: removed series %s\n", tmp);
			if (SearchForName(orphanedSeries, tmp, 1, &i)) {
				DeleteRegistration(orphanedSeries, i);
				UnregisterObject(tmp);
			}
			if (SearchForName(mySeries, tmp, 1, &i)) {
				DeleteRegistration(mySeries, i);
				UnregisterObject(tmp);
			}
			ret = 1;
		}

	}
	FREE(s);

	return ret;
}


/*
 * Removes from the #autoFetchInfo# module variable all information for the
 * client connected to #sock#.
 */
static void
EndAutoFetch(Socket sock) {
	int i;

	/* sanity check */
	if (sock == NO_SOCKET) {
		return;
	}

	for(i = 0; i < autoFetchCount; i++) {
		if(autoFetches[i].clientSock == sock) {
			/* feedback */
			INFO1("EndAutoFetch: cleaning after socket %d\n", sock);

			/* remove all the registrations */
			while(autoFetches[i].reg->howMany > 0) {
				DeleteRegistration(autoFetches[i].reg, 0);
			}
			autoFetches[i].clientSock = NO_SOCKET;
		}
	}
}

/* Used in ProcessRequest to store the data.
 * Stores #data# in #directory# with the attributes indicated by #s#.
 * Fails if the record size and count in #s# yield more than #len#
 * total bytes.  Returns 1 if successful, else 0.
 */
static int
KeepState(	const struct nws_memory_state *s,
		const char *data,
		size_t len) {
	const char *curr;
	int i, ret;
	size_t recordSize;

	/* sanity check */
	if (s == NULL || data == NULL) {
		ERROR("KeepState: NULL parameter\n");
		return 0;
	}
	if(s->rec_count > fileSize) {
		FAIL("KeepState: rec count too big\n");
	}
	if(s->rec_size > MAX_RECORD_SIZE) {
		WARN("KeepState: state record too big.\n");
		recordSize = MAX_RECORD_SIZE;
	} else {
		recordSize = (size_t)s->rec_size;
	}
	if(s->rec_count * recordSize > len) {
		FAIL1("KeepState: too much data %d\n", s->rec_count * recordSize);
	}

	/* all right, we are clear to store the data */
	ret = 1;
	for(curr = data, i = 0; i < s->rec_count; curr += recordSize, i++) {
		switch (backType) {
		case LOCALDIR:
			ret = WriteState(s->id,
					fileSize,
					s->time_out,
					s->seq_no,
					curr,
					recordSize);
			if (!ret) {
				/* as of 2.8.2 we disabled the journal */
				/*|| !EnterInJournal(s->id, s->seq_no)) */
				if(errno == EMFILE) {
					CheckConnections();
				}
			}
			break;

		case POSTGRES:
			/* let's go to the database */
			ret = WriteNwsDB(s->id, 
					s->time_out, 
					s->seq_no, 
					curr, 
					recordSize);
			break;

		case NETLOGGER:
#ifdef WITH_NETLOGGER
			ret = WriteStateNL(logLoc->path,
					s->id,
					fileSize,
					s->time_out,
					s->seq_no,
					curr,
					recordSize);
#endif
			break;
		}
		/* let's see how did it go */
		if (ret == 0) {
			ERROR("KeepState: write failed\n");
			break;
		}
	}

	return ret;
}


static void
NotifyFetchers(	const struct nws_memory_state *s,
		const char *data,
		size_t len) {
	DataDescriptor contentsDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	int i, j;
	Socket sock;

	/* set the size of the experiments string to be sent */
	contentsDescriptor.repetitions = len;

	/* forward to the autofetch clients */
	for (i = 0; i < autoFetchCount; i++) {
		if (autoFetches[i].clientSock == NO_SOCKET) {
			/* empty slot */
			continue;
		}

		if (SearchForName(autoFetches[i].reg, s->id, 1, &j)) {
			if (!SendMessageAndDatas(autoFetches[i].clientSock, STATE_FETCHED, s, stateDescriptor, stateDescriptorLength, data, &contentsDescriptor, 1, -1)) {
				/* some feedback */
				LOG1("NotifyFetchers: failed for socket %d\n", autoFetches[i].clientSock);
	
				/* now let's remove traces for this guy */
				sock = autoFetches[i].clientSock;
				EndAutoFetch(autoFetches[i].clientSock);
				DROP_SOCKET(&sock);
			}
		}
	}
}

static int
AddAutoFetcher(	Socket *sd,
		char *stateNames) {
	int i, ret, j;
	const char *word;
	char name[255 + 1];
	AutoFetchInfo *expandedAutoFetches;
	Object obj;

	/* sanity check */
	if (*sd == NO_SOCKET || stateNames == NULL) {
		WARN("AddAutoFetcher: NULL parameter(s)\n");
		return 0;
	}

	/* we have a new list of series to autoFetch: is
	 * this an old client? */
	for(j = -1, i = 0; i < autoFetchCount; i++) {
		if (autoFetches[i].clientSock == NO_SOCKET && j == -1) {
			/* remember empty slot */
			j = i;
		}
		if (autoFetches[i].clientSock == *sd) {
			/* old client: remove the old list */
			EndAutoFetch(*sd);
		}
	}

	/* let's make room for the new autofetcher */
	if (j == -1) {
		expandedAutoFetches = REALLOC(autoFetches, (i+1)*sizeof(AutoFetchInfo), 0);
		if (expandedAutoFetches == NULL) {
			ERROR("AddAutoFetcher: out of memory\n");
			return 0;
		}
		autoFetchCount++;
		autoFetches = expandedAutoFetches;
		j = i;

		/* let's set the new registrations */
		if (!InitRegistrations(&autoFetches[j].reg)) {
			ERROR("AddAutoFetcher: failed to init structure\n");
			EndAutoFetch(*sd);
			return 0;
		}
	}
	autoFetches[j].clientSock = *sd;

	/* let's get the registrations one by one */
	for (word = stateNames; GETWORD(name, word, &word); ) {
		/* let's get the right spot top add it */
		if (SearchForName(autoFetches[j].reg, name, 1, &ret)) {
			/* we already have it */
			LOG1("AddAutoFetcher: duplicate series (%s)\n", name);
			continue;
		}
					
		/* set the object */
		obj = NewObject();
		AddNwsAttribute(&obj, "name", name);

		if (!InsertRegistration(autoFetches[j].reg,
				obj,
				0,
				ret)) {
			ABORT("AddAutoFetcher: failed to insert series!\n");
		}
		FreeObject(&obj);
	}

	INFO1("AddAutoFetcher: we have %d autofetchers\n", autoFetchCount);

	return 1;
}


/* wrap around the reading functions */
static int
ReadRecord(	char *where,
		int maxSize,
		struct nws_memory_state *stateDesc) {
	int ret = 0;

	/* let's see which backend are we using */
	switch (backType) {
	case LOCALDIR:
		ret = ReadState(	stateDesc->id,
					where,
					stateDesc->rec_count,
					stateDesc->rec_count * maxSize,
					stateDesc->seq_no,
					&stateDesc->time_out,
					&stateDesc->seq_no,
					&stateDesc->rec_count,
					&stateDesc->rec_size);
		break;

	case POSTGRES:
		ret = ReadNwsDB(	stateDesc->id,
					where,
					stateDesc->rec_count,
					stateDesc->rec_count * maxSize,
					stateDesc->seq_no,
					&stateDesc->time_out,
					&stateDesc->seq_no,
					&stateDesc->rec_count,
					&stateDesc->rec_size);
		break;

	case NETLOGGER:
		ERROR("HELP!\n");
		break;
	}

	return ret;
}


/*
 * A "local" function of main().  Handles a #header#.message message arrived on
 * #sd# accompanied by #header#.dataSize bytes of data.
 */
static void
ProcessRequest(	Socket *sd,
		MessageHeader header) {
	char *contents;
	unsigned long expiration;
	DataDescriptor contentsDes = SIMPLE_DATA(CHAR_TYPE, 0);
	DataDescriptor expDescriptor = SIMPLE_DATA(UNSIGNED_LONG_TYPE, 1);
	DataDescriptor stateNamesDescriptor = SIMPLE_DATA(CHAR_TYPE, 0);
	int i, ret;
	struct nws_memory_state stateDesc;
	struct nws_memory_new_state newStateDesc;
	char *stateNames, *tmp;

	/* makes the compiler happy */
	contents = NULL;

	switch(header.message) {
	case FETCH_STATE:
		/* got an extra query */
		fetchQuery++;
		fetchTime -= MicroTime();

		if(!RecvData(*sd, &stateDesc, stateDescriptor, stateDescriptorLength, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: state receive failed\n");
		} else {
			/* we are cool */
			contents = (char *)MALLOC(stateDesc.rec_count * MAX_RECORD_SIZE, 0);
			if(contents == NULL) {
				(void)SendMessage(*sd, MEMORY_FAILED, -1);
				ERROR("ProcessRequest: out of memory\n");
			} else {
				if(ReadRecord( contents,
						MAX_RECORD_SIZE,
						&stateDesc)) {
					if(stateDesc.rec_count > 0) {
						contentsDes.repetitions = stateDesc.rec_size * stateDesc.rec_count;
						(void)SendMessageAndDatas(*sd,
							STATE_FETCHED,
							&stateDesc,
							stateDescriptor,
							stateDescriptorLength,
							contents,
							&contentsDes,
							1,
							-1);
					} else {
						(void)SendMessageAndData(*sd,
							STATE_FETCHED,
							&stateDesc,
							stateDescriptor,
							stateDescriptorLength,
							-1);
					}
				} else {
					INFO1("ProcessRequest: couldn't read state %s\n", stateDesc.id);
					(void)SendMessage(*sd, MEMORY_FAILED, -1);
					if(errno == EMFILE) {
						CheckConnections();
					}
				}
				free(contents);
			}
		}

		fetchTime += MicroTime();	/* stop the timer */
		break;

	case STORE_STATE:
		/* we got an old store request */
		oldStoreQuery++;
		oldStoreTime -= MicroTime();

		ret = 0;

		if(!RecvData(*sd,
				&stateDesc,
				stateDescriptor,
				stateDescriptorLength,
				-1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: state receive failed\n");
		} else {
			/* we are cool */
			contentsDes.repetitions = stateDesc.rec_size * stateDesc.rec_count;
			contents = (char *)MALLOC(contentsDes.repetitions + 1, 0);
			if(contents == NULL) {
				(void)SendMessage(*sd, MEMORY_FAILED, -1);
				ERROR("ProcessRequest: out of memory\n");
				break;
			}
			if(!RecvData(*sd, contents, &contentsDes, 1, -1)) {
				DROP_SOCKET(sd);
				ERROR("ProcessRequest: data receive failed\n");
				break;
			}
			contents[contentsDes.repetitions] = '\0';
	
			/* let's try to save the received experiment */
			ret = KeepState(&stateDesc, contents, contentsDes.repetitions);
			if (ret) {
				(void)SendMessage(*sd, STATE_STORED, -1);
			} else {
				(void)SendMessage(*sd, MEMORY_FAILED, -1);
			}
		}
		oldStoreTime += MicroTime();	/* stop the timer */

		/* notify the auto fetchters */
		autoFetchTime -= MicroTime();
		if (ret) {
			NotifyFetchers(	&stateDesc, 
					contents, 
					contentsDes.repetitions);
		}
		autoFetchTime += MicroTime();

		FREE(contents);
		break;

	/* new message since 2.9: we receive the full registration piggy
	 * backed into the packed experiment */
	case STORE_AND_REGISTER:
		/* we got a new query */
		storeQuery++;
		storeTime -= MicroTime();

		ret = 0;

		if(!RecvData(		*sd, 
					&newStateDesc,
					newStateDescriptor,
					newStateDescriptorLength,
					-1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: new state receive failed\n");
			return;
		} else {
			/* we are cool */
			contentsDes.repetitions = newStateDesc.rec_size * newStateDesc.rec_count + newStateDesc.id_len;
			contents = (char *)MALLOC(contentsDes.repetitions + 1, 0);
			tmp = (char *)MALLOC(newStateDesc.id_len + 1, 0);
			if(contents == NULL || tmp == NULL) {
				(void)SendMessage(*sd, MEMORY_FAILED, -1);
				ERROR("ProcessRequest: out of memory\n");
				break;
			}
			if(!RecvData(*sd, contents, &contentsDes, 1, -1)) {
				FREE(contents);
				FREE(tmp);
				DROP_SOCKET(sd);
				ERROR("ProcessRequest: data receive failed\n");
				break;
			}
			contents[contentsDes.repetitions] = '\0';

			/* let's try to save the received experiment */
			/* now is a trick: KeepState expects a stateDesc not a
			 * newStateDesc, so we just do it */
			stateDesc.rec_size = newStateDesc.rec_size;
			stateDesc.rec_count = newStateDesc.rec_count;
			stateDesc.seq_no = newStateDesc.seq_no;
			stateDesc.time_out = newStateDesc.time_out;

			/* let's extract the registration now: it's at the
			 * beginning of the content */
			memcpy(tmp, contents, newStateDesc.id_len);
			tmp[newStateDesc.id_len] = '\0';
	
			/* here only the name is needed */
			stateNames = NwsAttributeValue_r(FindNwsAttribute(tmp, "name"));
			if (stateNames == NULL || stateNames[0] == '\0') {
				ERROR("ProcessRequest: series with no name\n");
				DROP_SOCKET(sd);
				FREE(tmp);
				FREE(contents);
				break;
			}
			SAFESTRCPY(stateDesc.id, stateNames);
			FREE(stateNames);

			/* let's see how big is the record to be registered */
			i = strlen(contents + newStateDesc.id_len);
			ret = KeepState(&stateDesc, contents + newStateDesc.id_len, i);
			if (ret) {
				(void)SendMessage(*sd, STATE_STORED, -1);
			} else {
				(void)SendMessage(*sd, MEMORY_FAILED, -1);
			}

			/* last to be done for this is the REGISTER part: let's
			 * keep track of this registration */
			StoreSeriesName(tmp);
		}
		storeTime += MicroTime();	/* stop the timer */

		/* notify the auto fetchters */
		autoFetchTime -= MicroTime();
		if (ret) {
			NotifyFetchers(	&stateDesc, 
					contents + newStateDesc.id_len, 
					i);
		}
		autoFetchTime += MicroTime();

		FREE(contents);
		FREE(tmp);
		break;

	case AUTOFETCH_BEGIN:
		autoFetchTime -= MicroTime();

		stateNamesDescriptor.repetitions = header.dataSize;
		stateNames = (char *)MALLOC(header.dataSize, 0);
		if(stateNames == NULL) {
			(void)SendMessage(*sd, MEMORY_FAILED, -1);
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: out of memory\n");
		} else if(!RecvData(*sd,
					stateNames,
					&stateNamesDescriptor,
					1,
					-1)) {
			(void)SendMessage(*sd, MEMORY_FAILED, -1);
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: data receive failed\n");
		} else if(*stateNames == '\0') {
			/* empty list: remove the client from the list of
			 * our autofetchers */
			EndAutoFetch(*sd);

			(void)SendMessage(*sd, AUTOFETCH_ACK, -1);
		} else {
			/* let's add the new autofetcher */
			if (!AddAutoFetcher(sd, stateNames)) {
				SendMessage(*sd, MEMORY_FAILED, -1);
				DROP_SOCKET(sd);
				ERROR("ProcessRequest: failed to add autoFetcher\n");
			} else {
				(void)SendMessage(*sd, AUTOFETCH_ACK, -1);
			}
		}
		FREE(stateNames);
		autoFetchTime += MicroTime();
		break;

	case MEMORY_CLEAN:
		/* let's get the idle timeout first */
		if (!RecvData(*sd, &expiration, &expDescriptor, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: data (idle) receive failed\n");
			break;
		}

		/* then the series */
		contentsDes.repetitions = header.dataSize - HomogenousDataSize(UNSIGNED_LONG_TYPE, 1, NETWORK_FORMAT);
		contents = MALLOC(contentsDes.repetitions + 1, 1);
		if (!RecvData(*sd, contents, &contentsDes, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: data (series) receive failed\n");
		} else {
			contents[contentsDes.repetitions] = '\0';
			(void)SendMessage(*sd, MEMORY_CLEANED, -1);
			(void)DoClean(contents, expiration);
		}
		FREE(contents);

		break;

	case TRANSFER_SERIES:
		/*  we are taking over this series from another memory */
		contentsDes.repetitions = header.dataSize;
		contents = MALLOC(contentsDes.repetitions + 1, 1);
		if (!RecvData(*sd, contents, &contentsDes, 1, -1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: data (series) receive failed\n");
		} else {
			contents[contentsDes.repetitions] = '\0';
			DoTransfer(contents);
		}
		FREE(contents);
		break;

#ifdef WITH_NETLOGGER	  
static struct loglocation memLogLocation;
	case MEMORY_LOGDEST: /* config message contains log location */
		if (!RecvData(*sd,
				&memLogLocation,
				loglocationDescriptor,
				loglocationDescriptorLength,
				-1)) {
			DROP_SOCKET(sd);
			ERROR("ProcessRequest: loglocation receive failed\n");
			return;
		}else {
			(void)SendMessage(*sd, MEMORY_LOGDEST_ACK, -1);
		}
		LOG2("ProcessRequest: loglocation %d .%s.\n", memLogLocation.loc_type, memLogLocation.path);
		break;
#endif /* WITH_NETLOGGER */	  	

	default:
		DROP_SOCKET(sd);
		ERROR1("ProcessRequest: unknown message %d\n", header.message);
	}

	/* socket is now available */
	SocketIsAvailable(*sd);
}

#define DEFAULT_MEMORY_DIR "."
#define DEFAULT_MEMORY_SIZE "2000"
#define DEFAULT_JOURNAL_SIZE "2000"

static void usage() {
	DaemonUsage("nws_memory", "Memory");
	printf("\t-s num              keep #num# measurement per experiment\n");
	printf("\t-d dir              save series files into directory dir\n");
	printf("\t-B db               use database #db# to store data\n");
	printf("\t-I idle             remove series unmodified for #idle# seconds\n");
	printf("\t-C #                # of caches to keep (0 to disable cache)\n");
}

int
main(		int argc,
		char *argv[]) {
	double nextBeatTime, wakeup, nextClean;
	double now;
	int opt, 
		cacheEntries,
		autoIdle,
		ind;
	extern char *optarg;
	char memoryDir[255 + 1];
	char *toProcess, *dbname;

	/* Set up default values */
	toProcess = NULL;
	cacheEntries = 256;
	autoIdle = 0;

	dbname = NWSAPI_EnvironmentValue_r("MEMORY_SIZE", DEFAULT_MEMORY_SIZE);
	if (dbname == NULL) {
		ABORT("out of memory\n");
	} 
	fileSize = strtol(dbname, NULL, 10);
	FREE(dbname);

	dbname = NWSAPI_EnvironmentValue_r("MEMORY_DIR", DEFAULT_MEMORY_DIR);
	if (dbname == NULL) {
		ABORT("out of memory\n");
	}
	SAFESTRCPY(memoryDir, dbname);
	FREE(dbname);
	dbname = NULL;

	/* no error from getopt */
	opterr = 0;
	ind = InitializeDaemonStructure(MEMORY_HOST, "I:C:d:s:B:", MyExit);

	while((int)(opt = getopt(argc, argv, DaemonAllSwitches(ind))) != EOF) {
		if (DaemonSwitch(opt, optarg, ind)) {
			continue;
		}

		switch(opt) {
		case 'd':
			SAFESTRCPY(memoryDir, optarg);
			break;

		case 'I':
			autoIdle = (unsigned int)strtol(optarg, NULL, 10);
			if (autoIdle < 300) {
				INFO("main: too short idle: setting to 300\n");
				autoIdle = 300;
			}
			break;

		case 's':
			fileSize = strtol(optarg, NULL, 10);
			break;

		case 'B':
			dbname = strdup(optarg);
			if (dbname == NULL) {
				fprintf(stderr, "out of memory\n");
			}
#ifdef NWS_WITH_DB
			backType = POSTGRES;
			cacheEntries = 0;
#else
			fprintf(stderr, "database is not enabled: ignoring -B\n");
#endif
			break;

		case 'C':
			cacheEntries = strtol(optarg, NULL, 10);
			break;

		default:
			usage();
			exit(1);
			break;

		}
	}

	/* initialize the performance counters */
	fetchTime = storeTime = oldStoreTime = autoFetchTime = 0;
	fetchQuery = storeQuery = oldStoreQuery = 0;

	/* now initializing the backend: if we use the directory, we need
	 * to initialize the caching schema */
	/* let's see if we are trying to use a database */
	switch (backType) {
	case LOCALDIR:
		/* WARNING: filesize s very important to have it right or
		 * cache and backing store could be inconsistent */
		InitStateModule(cacheEntries, fileSize, memoryDir);

		/* let's get the list of filename of possibly old states */
		toProcess = ReadOldStates();
		break;

	case POSTGRES:
		if (dbname == NULL || strlen(dbname) == 0) {
			ABORT("You need to specify a database name (-B)\n");
		}
		if (!ConnectToNwsDB(dbname)) {
			ABORT("Fail to use database try to remove -B\n");
		}

		/* let's get the list of tables of possibly old states */
		toProcess = GetTables();
		break;
	
	case NETLOGGER:
		WARN("we should initialize to use netlooger\n");
		break;
	}

	/* initialize the mySeries/orphanedSeries structures */
	if (!InitRegistrations(&orphanedSeries) 
			|| !InitRegistrations(&mySeries)) {
		ABORT("main: out of memory\n");
	}

	/* let's get the port and start serving */
	if (!InitializeDaemon(0, 1, 0, ind)) {
		exit(-1);
	}

	RegisterListener(STORE_STATE, "STORE_STATE", &ProcessRequest);
	RegisterListener(STORE_AND_REGISTER, "STORE_AND_REGISTER", &ProcessRequest);
	RegisterListener(FETCH_STATE, "FETCH_STATE", &ProcessRequest);
	RegisterListener(AUTOFETCH_BEGIN, "AUTOFETCH_BEGIN", &ProcessRequest);
	RegisterListener(MEMORY_CLEAN, "MEMORY_CLEAN", &ProcessRequest);
#ifdef WITH_NETLOGGER
	RegisterListener(MEMORY_LOGDEST, "MEMORY_LOGDEST", &ProcessRequest);
#endif
	if (!NotifyOnDisconnection(&EndAutoFetch)) {
		WARN("main: failed to register for disconnections\n");
	}

	nextClean = nextBeatTime = 0;

	/* main service loop */
	while(1) {
		now = CurrentTime();
		if(now >= nextBeatTime) {
			RegisterHost(DEFAULT_HOST_BEAT * 2);
			nextBeatTime = now + (HostHealthy() ? DEFAULT_HOST_BEAT : SHORT_HOST_BEAT);

			/* print performance timing */
			LOG1("main: spent %.0fms in fetch\n", fetchTime/1000);
			LOG1("main: spent %.0fms in store\n", storeTime/1000);
			LOG1("main: spent %.0fms in old store\n", oldStoreTime/1000);
			LOG1("main: spent %.0fms for autofetching\n", autoFetchTime/1000);
			LOG3("main: got %d fetch, %d store and %d old store requests\n", fetchQuery, storeQuery, oldStoreQuery);
			/* resetting */
			fetchTime = storeTime = oldStoreTime = autoFetchTime = 0;
			fetchQuery = storeQuery = oldStoreQuery = 0;


			LOG2("main: we have %d series and %d orphaned series\n", mySeries->howMany, orphanedSeries->howMany);
		}

		/* we want to wake up at least once a minute */
		if ((now + 60) < nextBeatTime) {
			wakeup = 60;
		} else {
			wakeup = nextBeatTime - now;
		}

		/* let's convert filenames to orphaned series: if the
		 * directory is big it may take time to read it. */
		if (toProcess != NULL) {
			if (CheckOldNames(	toProcess,
						mySeries,
						orphanedSeries,
						1)) {
				wakeup = -1;
			} else {
				FREE(toProcess);
			}
		}

		/* let's register the series we are in charge of */
		if (RegisterSeries(orphanedSeries, 10)) {
			wakeup = -1;
		}
		if (RegisterSeries(mySeries, 10)) {
			wakeup = -1;
		}

		/* let's see if we need to clean older series */
		if (autoIdle > 0 && nextClean < now) {
			nextClean = now + autoIdle;
			DoClean("", autoIdle);
			LOG("main: autocleaning\n");
		}

		/* we want the memory to be repsonsive, so we flush all
		 * the waiting messages */
		while(ListenForMessages(-1)) {
			;
		}
		ListenForMessages((wakeup > 0) ? wakeup : -1);
	}

	/* return(0); Never reached */
}
