## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_surf(@var{x},@var{y},@var{z},@var{c},[@var{prop},@var{val}])
## Displays a surface plot defined by @var{x},@var{y},@var{z} matrices,
## where @var{x} and @var{y} are typically formed by meshgrid.
## @var{c} is an optional matrix to give a color scale.
##
## Optional @var{prop},@var{val} arguments can be used to change
## properties of the plot.  Currently, valid properties are
## ";Opacity;Color;".
## 
## Example: Spherical harmonics
## @example
## [phi,theta] = meshgrid(0:pi/250:pi,0:pi/250:2*pi); 
## m0 = 4; m1 = 3; m2 = 2; m3 = 3; m4 = 6; m5 = 2; m6 = 6; m7 = 4;
## r = sin(m0*phi).^m1 + cos(m2*phi).^m3 + sin(m4*theta).^m5 + cos(m6*theta).^m7;
## x = r .* sin(phi) .* cos(theta);
## y = r .* cos(phi);
## z = r .* sin(phi) .* sin(theta);
## vtk_surf(x,y,z);
## @end example
## 
## @end deftypefn
## @seealso{vtk_trisurf, vtk_mesh}

## Author: Dragan Tubic

function vtk_surf( varargin )
  
  valid_props = ";Opacity;Color;";
  
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, varargin{:});
  
  if ( no_numerical_params < 1 | no_numerical_params > 4 )
    error("Syntax is vtk_surf( x, (y), (z), (c) )");
  end
  
  if ( no_numerical_params == 1 )
    z = nth (varargin,1);
    [nr nc] = size(z);
    [x y] = meshgrid(1:nr,1:nc);
    c = z;
  elseif ( no_numerical_params == 2 )
    z = nth (varargin,1);
    [nr nc] = size(z);
    [x y] = meshgrid(1:nr,1:nc);
    c = nth (varargin,1);
  elseif ( no_numerical_params == 3 )
    x = nth (varargin,1);
    y = nth (varargin,2);
    z = nth (varargin,3);
    c = z;
  elseif ( no_numerical_params == 4 )
    x = nth (varargin,1);
    y = nth (varargin,2);
    z = nth (varargin,3);
    c = nth (varargin,4);
  end
  
  [nr nc] = size(x);
  
  if ( nr < 2 | nc < 2 )
    error("input has to be at least 2xN or Nx2 matrix");
  end	
  
  m=nc; n=nr;
  i=0:m-2;
  j=0:n-2;
  [i,j]=meshgrid(i,j);
  i=i(:); j=j(:);
  ## t=[i*n+j,i*n+(j+1),(i+1)*n+(j+1) ; (i+1)*n+(j+1),(i+1)*n+j,i*n+j];  %1; one way to construct the triangles
  t=[i*n+j,(i+1)*n+j,(i+1)*n+(j+1) ; (i+1)*n+(j+1),i*n+(j+1),i*n+j];  %2; another way, seems a bit better
  if ( first_prop_index > 0 )
    f=vtk_trisurf(t+1,x(:),y(:),z(:),c(:),varargin{first_prop_index:length(varargin)});
  else
    f=vtk_trisurf(t+1,x(:),y(:),z(:),c(:));
  end
  
endfunction
