/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


/* XML.cpp - XML parsing functionality for runtime

   Scott Cantor
   2/21/02

   $History:$
*/

#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/framework/LocalFileInputSource.hpp>
#include <xercesc/framework/Wrapper4InputSource.hpp>

#include "internal.h"
using namespace saml;
using namespace std;
using namespace log4cpp;

XMLCh* XML::assign(const XMLCh* src, bool trim)
{
    if (src) {
        if (trim) {
            XMLCh* ret=XMLString::replicate(src);
            XMLString::trim(ret);
            return ret;
        }
        else
            return XMLString::replicate(src);
    }
    return NULL;
}

bool XML::isElementNamed(const DOMElement* e, const XMLCh* ns, const XMLCh* localName)
{
    return (e && !XMLString::compareString(ns,e->getNamespaceURI()) && !XMLString::compareString(localName,e->getLocalName()));
}

DOMElement* XML::getFirstChildElement(const DOMNode* n)
{
    DOMNode* child = n->getFirstChild();
    while (child && child->getNodeType() != DOMNode::ELEMENT_NODE)
        child = child->getNextSibling();
    if (child)
        return static_cast<DOMElement*>(child);
    return NULL;
}    

DOMElement* XML::getLastChildElement(const DOMNode* n)
{
    DOMNode* child = n->getLastChild();
    while (child && child->getNodeType() != DOMNode::ELEMENT_NODE)
        child = child->getPreviousSibling();
    if (child)
        return static_cast<DOMElement*>(child);
    return NULL;
}    

DOMElement* XML::getFirstChildElement(const DOMNode* n, const XMLCh* ns, const XMLCh* localName)
{
    DOMElement* e = getFirstChildElement(n);
    while (e && !isElementNamed(e, ns, localName))
        e = getNextSiblingElement(e);
    return e;
}

DOMElement* XML::getLastChildElement(const DOMNode* n, const XMLCh* ns, const XMLCh* localName)
{
    DOMElement* e = getLastChildElement(n);
    while (e && !isElementNamed(e, ns, localName))
        e = getPreviousSiblingElement(e);
    return e;
}

DOMElement* XML::getNextSiblingElement(const DOMNode* n)
{
    DOMNode* sib = n->getNextSibling();
    while (sib && sib->getNodeType() != DOMNode::ELEMENT_NODE)
        sib = sib->getNextSibling();
    if (sib)
        return static_cast<DOMElement*>(sib);
    return NULL;
}

DOMElement* XML::getPreviousSiblingElement(const DOMNode* n)
{
    DOMNode* sib = n->getPreviousSibling();
    while (sib && sib->getNodeType() != DOMNode::ELEMENT_NODE)
        sib = sib->getPreviousSibling();
    if (sib)
        return static_cast<DOMElement*>(sib);
    return NULL;
}

DOMElement* XML::getNextSiblingElement(const DOMNode* n, const XMLCh* ns, const XMLCh* localName)
{
    DOMElement* e = getNextSiblingElement(n);
    while (e && !isElementNamed(e, ns, localName))
        e = getNextSiblingElement(e);
    return e;
}

DOMElement* XML::getPreviousSiblingElement(const DOMNode* n, const XMLCh* ns, const XMLCh* localName)
{
    DOMElement* e = getPreviousSiblingElement(n);
    while (e && !isElementNamed(e, ns, localName))
        e = getPreviousSiblingElement(e);
    return e;
}


void XML::registerSchema(const XMLCh* xmlns, const XMLCh* systemId, DOMEntityResolver* resolver, bool compatModePool)
{
    static_cast<SAMLInternalConfig&>(SAMLConfig::getConfig()).m_pool->registerSchema(xmlns,systemId,resolver);
    if (compatModePool)
        static_cast<SAMLInternalConfig&>(SAMLConfig::getConfig()).m_compat_pool->registerSchema(xmlns,systemId,resolver);
}

XML::Parser::Parser()
{
    SAMLInternalConfig& conf=static_cast<SAMLInternalConfig&>(SAMLConfig::getConfig());
    if (conf.compatibility_mode) {
        m_parser=conf.m_compat_pool->get();
        m_minor=0;
    }
    else {
        m_parser=conf.m_pool->get();
        m_minor=1;
    }
}

XML::Parser::Parser(int minor) : m_minor(minor)
{
    SAMLInternalConfig& conf=static_cast<SAMLInternalConfig&>(SAMLConfig::getConfig());
    if (minor==1)
        m_parser=conf.m_pool->get();
    else
        m_parser=conf.m_compat_pool->get();
}

XML::Parser::~Parser()
{
    SAMLInternalConfig& conf=static_cast<SAMLInternalConfig&>(SAMLConfig::getConfig());
    if (m_minor==1)
        conf.m_pool->put(m_parser);
    else
        conf.m_compat_pool->put(m_parser);
}

XML::ParserPool::ParserPool()
{
    m_lock=XMLPlatformUtils::makeMutex();
}

XML::ParserPool::~ParserPool()
{
    while(!m_pool.empty())
    {
        m_pool.top()->release();
        m_pool.pop();
    }
    XMLPlatformUtils::closeMutex(m_lock);
}

void XML::ParserPool::registerSchema(const XMLCh* xmlns, const XMLCh* systemId, DOMEntityResolver* resolver)
{
    if (xmlns && systemId)
    {
#ifdef HAVE_GOOD_STL
        const XMLCh* _xmlns=xmlns;
        const XMLCh* _id=systemId;
#else
        auto_ptr_char t_xmlns(xmlns);
        auto_ptr_char t_id(systemId);
        const char* _xmlns=t_xmlns.get();
        const char* _id=t_id.get();
#endif
        XMLPlatformUtils::lockMutex(m_lock);
        try
        {
            // Insert the pair, but only if it's a new one or a change.
            SchemaLocationMap::const_iterator pos=m_SchemaLocMap.find(_xmlns);
            if (pos==m_SchemaLocMap.end() || XMLString::compareString(pos->second.c_str(),_id))
            {
                m_SchemaLocMap[_xmlns]=_id;
                if (resolver)
                    m_ResolverMap.insert(ResolverMap::value_type(_id,resolver));

                // Rebuild schemaLocation property string.
                m_schemaLocations.erase();
                for(SchemaLocationMap::const_iterator i=m_SchemaLocMap.begin(); i!=m_SchemaLocMap.end(); i++)
#ifdef HAVE_GOOD_STL
                    m_schemaLocations=m_schemaLocations + chSpace + i->first + chSpace + i->second + chSpace;
#else
                    m_schemaLocations=m_schemaLocations + ' ' + i->first + ' ' + i->second + ' ';
#endif
            }
        }
        catch (...)
        {
            XMLPlatformUtils::unlockMutex(m_lock);
            throw;
        }
        XMLPlatformUtils::unlockMutex(m_lock);
    }
}

void XML::ParserPool::put(DOMBuilder* p)
{
    if (!p) return;
    XMLPlatformUtils::lockMutex(m_lock);
    m_pool.push(p);
    XMLPlatformUtils::unlockMutex(m_lock);
}

DOMBuilder* XML::ParserPool::get(bool validating)
{
    XMLPlatformUtils::lockMutex(m_lock);
    try {
        if (m_pool.empty() || !validating) {
            static const XMLCh impltype[] = { chLatin_L, chLatin_S, chNull };
            DOMImplementation* impl=DOMImplementationRegistry::getDOMImplementation(impltype);
            DOMBuilder* parser=(static_cast<DOMImplementationLS*>(impl))->createDOMBuilder(DOMImplementationLS::MODE_SYNCHRONOUS,0);
            parser->setFeature(XMLUni::fgDOMNamespaces,true);

            if (validating) {
                parser->setFeature(XMLUni::fgXercesSchema,true);
                parser->setFeature(XMLUni::fgDOMValidation,true);
                parser->setFeature(XMLUni::fgXercesCacheGrammarFromParse,true);
                parser->setFeature(XMLUni::fgXercesValidationErrorAsFatal,true);
#ifdef HAVE_GOOD_STL
                parser->setProperty(XMLUni::fgXercesSchemaExternalSchemaLocation,const_cast<XMLCh*>(m_schemaLocations.c_str()));
#else
                auto_ptr_XMLCh temp(m_schemaLocations.c_str());
                parser->setProperty(XMLUni::fgXercesSchemaExternalSchemaLocation,const_cast<XMLCh*>(temp.get()));
#endif
            }
            XMLPlatformUtils::unlockMutex(m_lock);

            parser->setFeature(XMLUni::fgXercesUserAdoptsDOMDocument,true);
            parser->setEntityResolver(this);
            parser->setErrorHandler(this);
            return parser;
        }
        DOMBuilder* p=m_pool.top();
        m_pool.pop();
#ifdef HAVE_GOOD_STL
        p->setProperty(XMLUni::fgXercesSchemaExternalSchemaLocation,const_cast<XMLCh*>(m_schemaLocations.c_str()));
#else
        auto_ptr_XMLCh temp2(m_schemaLocations.c_str());
        p->setProperty(XMLUni::fgXercesSchemaExternalSchemaLocation,const_cast<XMLCh*>(temp2.get()));
#endif
        XMLPlatformUtils::unlockMutex(m_lock);
        return p;
    }
    catch(...) {
        XMLPlatformUtils::unlockMutex(m_lock);
        throw;
    }
}

DOMInputSource* XML::ParserPool::resolveEntity(const XMLCh* const publicId, const XMLCh* const systemId, const XMLCh* const baseURI)
{
    if (!systemId)
        return NULL;

    NDC ndc("resolveEntity");
    Category& log=Category::getInstance(SAML_LOGCAT".XML.ParserPool");
    if (log.isDebugEnabled())
    {
        auto_ptr_char sysId(systemId);
        auto_ptr_char base(baseURI);
        log.debug("asked to resolve %s with baseURI %s",sysId.get(),base.get() ? base.get() : "(null)");
    }

    const SAMLInternalConfig& config=static_cast<SAMLInternalConfig&>(SAMLConfig::getConfig());

    // Find well-known schemas in the specified location.
    if (XMLString::endsWith(systemId,SAML_SCHEMA_ID))
        return new Wrapper4InputSource(new LocalFileInputSource(config.wide_schema_dir,SAML_SCHEMA_ID));
    else if (XMLString::endsWith(systemId,SAML11_SCHEMA_ID))
        return new Wrapper4InputSource(new LocalFileInputSource(config.wide_schema_dir,SAML11_SCHEMA_ID));
    else if (XMLString::endsWith(systemId,SAMLP_SCHEMA_ID))
        return new Wrapper4InputSource(new LocalFileInputSource(config.wide_schema_dir,SAMLP_SCHEMA_ID));
    else if (XMLString::endsWith(systemId,SAMLP11_SCHEMA_ID))
        return new Wrapper4InputSource(new LocalFileInputSource(config.wide_schema_dir,SAMLP11_SCHEMA_ID));
    else if (XMLString::endsWith(systemId,SOAP11ENV_SCHEMA_ID))
        return new Wrapper4InputSource(new LocalFileInputSource(config.wide_schema_dir,SOAP11ENV_SCHEMA_ID));
    else if (XMLString::endsWith(systemId,XMLSIG_SCHEMA_ID))
        return new Wrapper4InputSource(new LocalFileInputSource(config.wide_schema_dir,XMLSIG_SCHEMA_ID));
    else if (XMLString::endsWith(systemId,XML_SCHEMA_ID))
        return new Wrapper4InputSource(new LocalFileInputSource(config.wide_schema_dir,XML_SCHEMA_ID));

    // Look for a custom resolver.
    DOMInputSource* src=NULL;
#ifdef HAVE_GOOD_STL
    ResolverMap::const_iterator i=m_ResolverMap.find(systemId);
#else
    auto_ptr_char temp(systemId);
    ResolverMap::const_iterator i=m_ResolverMap.find(temp.get());
#endif
    if (i!=m_ResolverMap.end() && (src=(i->second)->resolveEntity(publicId,systemId,baseURI)))
    {
        log.debug("using a custom resolver");
        return src;
    }

    // Look for it in the same location as the rest. In no case should we ever let it
    // search the network...
    log.debug("no custom resolver, looking in %s",config.schema_dir.c_str());
    return new Wrapper4InputSource(new LocalFileInputSource(config.wide_schema_dir,systemId));
}

bool XML::ParserPool::handleError(const DOMError& e)
{
    NDC ndc("handleError");
    Category& log=Category::getInstance(SAML_LOGCAT".XML.ParserPool");
    DOMLocator* locator=e.getLocation();
    auto_ptr_char temp(e.getMessage());

    switch (e.getSeverity())
    {
        case DOMError::DOM_SEVERITY_WARNING:
            log.warnStream() << "warning on line " << locator->getLineNumber()
                << ", column " << locator->getColumnNumber()
                << ", message: " << temp.get() << CategoryStream::ENDLINE;
            return true;

        case DOMError::DOM_SEVERITY_ERROR:
            log.errorStream() << "error on line " << locator->getLineNumber()
                << ", column " << locator->getColumnNumber()
                << ", message: " << temp.get() << CategoryStream::ENDLINE;
            throw MalformedException(SAMLException::RESPONDER,string("XML::Parser detected an error during parsing: ") + (temp.get() ? temp.get() : "no message"));

        case DOMError::DOM_SEVERITY_FATAL_ERROR:
            log.critStream() << "fatal error on line " << locator->getLineNumber()
                << ", column " << locator->getColumnNumber()
                << ", message: " << temp.get() << CategoryStream::ENDLINE;
            throw MalformedException(SAMLException::RESPONDER,string("XML::Parser detected a fatal error during parsing: ") + (temp.get() ? temp.get() : "no message"));
    }
    throw MalformedException(SAMLException::RESPONDER,string("XML::Parser detected an unexpected problem during parsing: ") + (temp.get() ? temp.get() : "no message"));
}

unsigned int XML::StreamInputSource::StreamBinInputStream::readBytes(XMLByte* const toFill, const unsigned int maxToRead)
{
    XMLByte* target=toFill;
    unsigned int bytes_read=0,request=maxToRead;

    // Fulfill the rest by reading from the stream.
    if (request && !m_is.eof())
    {
        try
        {
            m_is.read(reinterpret_cast<char* const>(target),request);
            m_pos+=m_is.gcount();
            bytes_read+=m_is.gcount();
        }
        catch(...)
        {
            cerr << "XML::StreamInputSource::StreamBinInputStream::readBytes caught an exception" << endl;
            *toFill=0;
            return 0;
        }
    }
    return bytes_read;
}


// Namespace and schema string literals

const XMLCh XML::OPENSAML_NS[] = // http://www.opensaml.org
{ chLatin_h, chLatin_t, chLatin_t, chLatin_p, chColon, chForwardSlash, chForwardSlash,
  chLatin_w, chLatin_w, chLatin_w, chPeriod,
  chLatin_o, chLatin_p, chLatin_e, chLatin_n, chLatin_s, chLatin_a, chLatin_m, chLatin_l, chPeriod,
  chLatin_o, chLatin_r, chLatin_g, chNull
};
const XMLCh XML::XSD_NS[] = // http://www.w3.org/2001/XMLSchema
{ chLatin_h, chLatin_t, chLatin_t, chLatin_p, chColon, chForwardSlash, chForwardSlash,
  chLatin_w, chLatin_w, chLatin_w, chPeriod, chLatin_w, chDigit_3, chPeriod, chLatin_o, chLatin_r, chLatin_g, chForwardSlash,
  chDigit_2, chDigit_0, chDigit_0, chDigit_1, chForwardSlash,
  chLatin_X, chLatin_M, chLatin_L, chLatin_S, chLatin_c, chLatin_h, chLatin_e, chLatin_m, chLatin_a, chNull
};
const XMLCh XML::XSI_NS[] = // http://www.w3.org/2001/XMLSchema-instance
{ chLatin_h, chLatin_t, chLatin_t, chLatin_p, chColon, chForwardSlash, chForwardSlash,
  chLatin_w, chLatin_w, chLatin_w, chPeriod, chLatin_w, chDigit_3, chPeriod, chLatin_o, chLatin_r, chLatin_g, chForwardSlash,
  chDigit_2, chDigit_0, chDigit_0, chDigit_1, chForwardSlash,
  chLatin_X, chLatin_M, chLatin_L, chLatin_S, chLatin_c, chLatin_h, chLatin_e, chLatin_m, chLatin_a, chDash,
  chLatin_i, chLatin_n, chLatin_s, chLatin_t, chLatin_a, chLatin_n, chLatin_c, chLatin_e, chNull
};
const XMLCh XML::XMLNS_NS[] = // http://www.w3.org/2000/xmlns/
{ chLatin_h, chLatin_t, chLatin_t, chLatin_p, chColon, chForwardSlash, chForwardSlash,
  chLatin_w, chLatin_w, chLatin_w, chPeriod, chLatin_w, chDigit_3, chPeriod, chLatin_o, chLatin_r, chLatin_g, chForwardSlash,
  chDigit_2, chDigit_0, chDigit_0, chDigit_0, chForwardSlash,
  chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chForwardSlash, chNull
};
const XMLCh XML::XML_NS[] = // http://www.w3.org/XML/1998/namespace
{ chLatin_h, chLatin_t, chLatin_t, chLatin_p, chColon, chForwardSlash, chForwardSlash,
  chLatin_w, chLatin_w, chLatin_w, chPeriod, chLatin_w, chDigit_3, chPeriod, chLatin_o, chLatin_r, chLatin_g, chForwardSlash,
  chLatin_X, chLatin_M, chLatin_L, chForwardSlash, chDigit_1, chDigit_9, chDigit_9, chDigit_8, chForwardSlash,
  chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chLatin_p, chLatin_a, chLatin_c, chLatin_e, chNull
};
const XMLCh XML::SAML_NS[] = // urn:oasis:names:tc:SAML:1.0:assertion
{ chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
  chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
  chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
  chLatin_a, chLatin_s, chLatin_s, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull
};
const XMLCh XML::SAMLP_NS[] = // urn:oasis:names:tc:SAML:1.0:protocol
{ chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
  chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
  chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_0, chColon,
  chLatin_p, chLatin_r, chLatin_o, chLatin_t, chLatin_o, chLatin_c, chLatin_o, chLatin_l, chNull
};
const XMLCh XML::SAML11_PROTOCOL_ENUM[] = // urn:oasis:names:tc:SAML:1.1:protocol
{ chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
  chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
  chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon, chDigit_1, chPeriod, chDigit_1, chColon,
  chLatin_p, chLatin_r, chLatin_o, chLatin_t, chLatin_o, chLatin_c, chLatin_o, chLatin_l, chNull
};
const XMLCh XML::SAML_ARTIFACT_SOURCEID[] = // urn:oasis:names:tc:SAML:profiles:v1metadata
{ chLatin_u, chLatin_r, chLatin_n, chColon, chLatin_o, chLatin_a, chLatin_s, chLatin_i, chLatin_s, chColon,
  chLatin_n, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chColon, chLatin_t, chLatin_c, chColon,
  chLatin_S, chLatin_A, chLatin_M, chLatin_L, chColon,
  chLatin_p, chLatin_r, chLatin_o, chLatin_f, chLatin_i, chLatin_l, chLatin_e, chLatin_s, chColon,
  chLatin_v, chDigit_1, chLatin_m, chLatin_e, chLatin_t, chLatin_a, chLatin_d, chLatin_a, chLatin_t, chLatin_a, chNull
};
const XMLCh XML::XMLSIG_NS[] = // http://www.w3.org/2000/09/xmldsig#
{ chLatin_h, chLatin_t, chLatin_t, chLatin_p, chColon, chForwardSlash, chForwardSlash,
  chLatin_w, chLatin_w, chLatin_w, chPeriod, chLatin_w, chDigit_3, chPeriod, chLatin_o, chLatin_r, chLatin_g, chForwardSlash,
  chDigit_2, chDigit_0, chDigit_0, chDigit_0, chForwardSlash, chDigit_0, chDigit_9, chForwardSlash,
  chLatin_x, chLatin_m, chLatin_l, chLatin_d, chLatin_s, chLatin_i, chLatin_g, chPound, chNull
};
const XMLCh XML::SOAP11ENV_NS[] = // http://schemas.xmlsoap.org/soap/envelope/
{ chLatin_h, chLatin_t, chLatin_t, chLatin_p, chColon, chForwardSlash, chForwardSlash,
  chLatin_s, chLatin_c, chLatin_h, chLatin_e, chLatin_m, chLatin_a, chLatin_s, chPeriod,
      chLatin_x, chLatin_m, chLatin_l, chLatin_s, chLatin_o, chLatin_a, chLatin_p, chPeriod,
      chLatin_o, chLatin_r, chLatin_g, chForwardSlash,
  chLatin_s, chLatin_o, chLatin_a, chLatin_p, chForwardSlash,
  chLatin_e, chLatin_n, chLatin_v, chLatin_e, chLatin_l, chLatin_o, chLatin_p, chLatin_e, chForwardSlash, chNull
};
const XMLCh XML::SAML_SCHEMA_ID[] = // cs-sstc-schema-assertion-01.xsd
{ chLatin_c, chLatin_s, chDash, chLatin_s, chLatin_s, chLatin_t, chLatin_c, chDash,
  chLatin_s, chLatin_c, chLatin_h, chLatin_e, chLatin_m, chLatin_a, chDash,
  chLatin_a, chLatin_s, chLatin_s, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chDash,
  chDigit_0, chDigit_1, chPeriod, chLatin_x, chLatin_s, chLatin_d, chNull
};
const XMLCh XML::SAMLP_SCHEMA_ID[] = // cs-sstc-schema-protocol-01.xsd
{ chLatin_c, chLatin_s, chDash, chLatin_s, chLatin_s, chLatin_t, chLatin_c, chDash,
  chLatin_s, chLatin_c, chLatin_h, chLatin_e, chLatin_m, chLatin_a, chDash,
  chLatin_p, chLatin_r, chLatin_o, chLatin_t, chLatin_o, chLatin_c, chLatin_o, chLatin_l, chDash,
  chDigit_0, chDigit_1, chPeriod, chLatin_x, chLatin_s, chLatin_d, chNull
};
const XMLCh XML::SAML11_SCHEMA_ID[] = // cs-sstc-schema-assertion-1.1.xsd
{ chLatin_c, chLatin_s, chDash, chLatin_s, chLatin_s, chLatin_t, chLatin_c, chDash,
  chLatin_s, chLatin_c, chLatin_h, chLatin_e, chLatin_m, chLatin_a, chDash,
  chLatin_a, chLatin_s, chLatin_s, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chDash,
  chDigit_1, chPeriod, chDigit_1, chPeriod, chLatin_x, chLatin_s, chLatin_d, chNull
};
const XMLCh XML::SAMLP11_SCHEMA_ID[] = // cs-sstc-schema-protocol-1.1.xsd
{ chLatin_c, chLatin_s, chDash, chLatin_s, chLatin_s, chLatin_t, chLatin_c, chDash,
  chLatin_s, chLatin_c, chLatin_h, chLatin_e, chLatin_m, chLatin_a, chDash,
  chLatin_p, chLatin_r, chLatin_o, chLatin_t, chLatin_o, chLatin_c, chLatin_o, chLatin_l, chDash,
  chDigit_1, chPeriod, chDigit_1, chPeriod, chLatin_x, chLatin_s, chLatin_d, chNull
};
const XMLCh XML::XMLSIG_SCHEMA_ID[] = // xmldsig-core-schema.xsd
{ chLatin_x, chLatin_m, chLatin_l, chLatin_d, chLatin_s, chLatin_i, chLatin_g, chDash,
  chLatin_c, chLatin_o, chLatin_r, chLatin_e, chDash,
  chLatin_s, chLatin_c, chLatin_h, chLatin_e, chLatin_m, chLatin_a,
  chPeriod, chLatin_x, chLatin_s, chLatin_d, chNull
};
const XMLCh XML::XML_SCHEMA_ID[] = // xml.xsd
{ chLatin_x, chLatin_m, chLatin_l, chPeriod, chLatin_x, chLatin_s, chLatin_d, chNull };
const XMLCh XML::SOAP11ENV_SCHEMA_ID[] = // soap-envelope.xsd
{ chLatin_s, chLatin_o, chLatin_a, chLatin_p, chDash,
  chLatin_e, chLatin_n, chLatin_v, chLatin_e, chLatin_l, chLatin_o, chLatin_p, chLatin_e,
  chPeriod, chLatin_x, chLatin_s, chLatin_d, chNull
};


// SAML vocabulary literals

const XMLCh XML::Literals::Action[]= 
{ chLatin_A, chLatin_c, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::Advice[]=
{ chLatin_A, chLatin_d, chLatin_v, chLatin_i, chLatin_c, chLatin_e, chNull };

const XMLCh XML::Literals::Assertion[]=
{ chLatin_A, chLatin_s, chLatin_s, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::AssertionArtifact[]=
{ chLatin_A, chLatin_s, chLatin_s, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_o, chLatin_n,
  chLatin_A, chLatin_r, chLatin_t, chLatin_i, chLatin_f, chLatin_a, chLatin_c, chLatin_t, chNull };

const XMLCh XML::Literals::AssertionID[]=
{ chLatin_A, chLatin_s, chLatin_s, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_I, chLatin_D, chNull };

const XMLCh XML::Literals::AssertionIDReference[]=
{ chLatin_A, chLatin_s, chLatin_s, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_I, chLatin_D,
  chLatin_R, chLatin_e, chLatin_f, chLatin_e, chLatin_r, chLatin_e, chLatin_n, chLatin_c, chLatin_e, chNull };

const XMLCh XML::Literals::saml_AssertionIDReference[]=
{ chLatin_s, chLatin_a, chLatin_m, chLatin_l, chColon,
  chLatin_A, chLatin_s, chLatin_s, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_I, chLatin_D,
  chLatin_R, chLatin_e, chLatin_f, chLatin_e, chLatin_r, chLatin_e, chLatin_n, chLatin_c, chLatin_e, chNull };

const XMLCh XML::Literals::Attribute[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e, chNull };

const XMLCh XML::Literals::AttributeDesignator[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e,
    chLatin_D, chLatin_e, chLatin_s, chLatin_i, chLatin_g, chLatin_n, chLatin_a, chLatin_t, chLatin_o, chLatin_r, chNull };

const XMLCh XML::Literals::AttributeName[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e, chLatin_N, chLatin_a, chLatin_m, chLatin_e, chNull };

const XMLCh XML::Literals::AttributeNamespace[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e, chLatin_N, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chLatin_p, chLatin_a, chLatin_c, chLatin_e, chNull };

const XMLCh XML::Literals::AttributeQuery[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e, chLatin_Q, chLatin_u, chLatin_e, chLatin_r, chLatin_y, chNull };

const XMLCh XML::Literals::AttributeQueryType[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e, chLatin_Q, chLatin_u, chLatin_e, chLatin_r, chLatin_y, chLatin_T, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::AttributeStatement[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e, chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_e, chLatin_m, chLatin_e, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::AttributeStatementType[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e, chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_e, chLatin_m, chLatin_e, chLatin_n, chLatin_t, chLatin_T, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::AttributeValue[]=
{ chLatin_A, chLatin_t, chLatin_t, chLatin_r, chLatin_i, chLatin_b, chLatin_u, chLatin_t, chLatin_e, chLatin_V, chLatin_a, chLatin_l, chLatin_u, chLatin_e, chNull };

const XMLCh XML::Literals::Audience[]=
{ chLatin_A, chLatin_u, chLatin_d, chLatin_i, chLatin_e, chLatin_n, chLatin_c, chLatin_e, chNull };

const XMLCh XML::Literals::AudienceRestrictionCondition[]=
{ chLatin_A, chLatin_u, chLatin_d, chLatin_i, chLatin_e, chLatin_n, chLatin_c, chLatin_e, chLatin_R, chLatin_e, chLatin_s, chLatin_t, chLatin_r, chLatin_i, chLatin_c, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_C, chLatin_o, chLatin_n, chLatin_d, chLatin_i, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::AudienceRestrictionConditionType[]=
{ chLatin_A, chLatin_u, chLatin_d, chLatin_i, chLatin_e, chLatin_n, chLatin_c, chLatin_e, chLatin_R, chLatin_e, chLatin_s, chLatin_t, chLatin_r, chLatin_i, chLatin_c, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_C, chLatin_o, chLatin_n, chLatin_d, chLatin_i, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_T, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::AuthenticationInstant[]=
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_e, chLatin_n, chLatin_t, chLatin_i, chLatin_c, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_I, chLatin_n, chLatin_s, chLatin_t, chLatin_a, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::AuthenticationMethod[]=
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_e, chLatin_n, chLatin_t, chLatin_i, chLatin_c, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_M, chLatin_e, chLatin_t, chLatin_h, chLatin_o, chLatin_d, chNull };

const XMLCh XML::Literals::AuthenticationQuery[]=
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_e, chLatin_n, chLatin_t, chLatin_i, chLatin_c, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_Q, chLatin_u, chLatin_e, chLatin_r, chLatin_y, chNull };

const XMLCh XML::Literals::AuthenticationQueryType[]=
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_e, chLatin_n, chLatin_t, chLatin_i, chLatin_c, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_Q, chLatin_u, chLatin_e, chLatin_r, chLatin_y, chLatin_T, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::AuthenticationStatement[]=
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_e, chLatin_n, chLatin_t, chLatin_i, chLatin_c, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_e, chLatin_m, chLatin_e, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::AuthenticationStatementType[]=
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_e, chLatin_n, chLatin_t, chLatin_i, chLatin_c, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_e, chLatin_m, chLatin_e, chLatin_n, chLatin_t, chLatin_T, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::AuthorityBinding[]=
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_o, chLatin_r, chLatin_i, chLatin_t, chLatin_y, chLatin_B, chLatin_i, chLatin_n, chLatin_d, chLatin_i, chLatin_n, chLatin_g, chNull };

const XMLCh XML::Literals::AuthorityKind[]=
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_o, chLatin_r, chLatin_i, chLatin_t, chLatin_y, chLatin_K, chLatin_i, chLatin_n, chLatin_d, chNull };

const XMLCh XML::Literals::AuthorizationDecisionQuery[]= 
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_o, chLatin_r, chLatin_i, chLatin_z, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_D, chLatin_e, chLatin_c, chLatin_i, chLatin_s, chLatin_i, chLatin_o, chLatin_n, chLatin_Q, chLatin_u, chLatin_e, chLatin_r, chLatin_y, chNull };

const XMLCh XML::Literals::AuthorizationDecisionQueryType[]= 
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_o, chLatin_r, chLatin_i, chLatin_z, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_D, chLatin_e, chLatin_c, chLatin_i, chLatin_s, chLatin_i, chLatin_o, chLatin_n, chLatin_Q, chLatin_u, chLatin_e, chLatin_r, chLatin_y, chLatin_T, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::AuthorizationDecisionStatement[]= 
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_o, chLatin_r, chLatin_i, chLatin_z, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_D, chLatin_e, chLatin_c, chLatin_i, chLatin_s, chLatin_i, chLatin_o, chLatin_n, chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_e, chLatin_m, chLatin_e, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::AuthorizationDecisionStatementType[]= 
{ chLatin_A, chLatin_u, chLatin_t, chLatin_h, chLatin_o, chLatin_r, chLatin_i, chLatin_z, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_D, chLatin_e, chLatin_c, chLatin_i, chLatin_s, chLatin_i, chLatin_o, chLatin_n, chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_e, chLatin_m, chLatin_e, chLatin_n, chLatin_t, chLatin_T, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::Binding[]=
{ chLatin_B, chLatin_i, chLatin_n, chLatin_d, chLatin_i, chLatin_n, chLatin_g, chNull };

const XMLCh XML::Literals::Client[]=
{ chLatin_C, chLatin_l, chLatin_i, chLatin_e, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::Condition[]=
{ chLatin_C, chLatin_o, chLatin_n, chLatin_d, chLatin_i, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::Conditions[]=
{ chLatin_C, chLatin_o, chLatin_n, chLatin_d, chLatin_i, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_s, chNull };

const XMLCh XML::Literals::ConfirmationMethod[]=
{ chLatin_C, chLatin_o, chLatin_n, chLatin_f, chLatin_i, chLatin_r, chLatin_m, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_M, chLatin_e, chLatin_t, chLatin_h, chLatin_o, chLatin_d, chNull };

const XMLCh XML::Literals::Decision[]= 
{ chLatin_D, chLatin_e, chLatin_c, chLatin_i, chLatin_s, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::DNSAddress[]=
{ chLatin_D, chLatin_N, chLatin_S, chLatin_A, chLatin_d, chLatin_d, chLatin_r, chLatin_e, chLatin_s, chLatin_s, chNull };

const XMLCh XML::Literals::DoNotCacheCondition[]=
{ chLatin_D, chLatin_o, chLatin_N, chLatin_o, chLatin_t, chLatin_C, chLatin_a, chLatin_c, chLatin_h, chLatin_e,
  chLatin_C, chLatin_o, chLatin_n, chLatin_d, chLatin_i, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull
};

const XMLCh XML::Literals::DoNotCacheConditionType[]=
{ chLatin_D, chLatin_o, chLatin_N, chLatin_o, chLatin_t, chLatin_C, chLatin_a, chLatin_c, chLatin_h, chLatin_e,
  chLatin_C, chLatin_o, chLatin_n, chLatin_d, chLatin_i, chLatin_t, chLatin_i, chLatin_o, chLatin_n,
  chLatin_T, chLatin_y, chLatin_p, chLatin_e, chNull
};

const XMLCh XML::Literals::Evidence[]= 
{ chLatin_E, chLatin_v, chLatin_i, chLatin_d, chLatin_e, chLatin_n, chLatin_c, chLatin_e, chNull };

const XMLCh XML::Literals::Format[]=
{ chLatin_F, chLatin_o, chLatin_r, chLatin_m, chLatin_a, chLatin_t, chNull };

const XMLCh XML::Literals::InResponseTo[]=
{ chLatin_I, chLatin_n, chLatin_R, chLatin_e, chLatin_s, chLatin_p, chLatin_o, chLatin_n, chLatin_s, chLatin_e, chLatin_T, chLatin_o, chNull };

const XMLCh XML::Literals::IPAddress[]=
{ chLatin_I, chLatin_P, chLatin_A, chLatin_d, chLatin_d, chLatin_r, chLatin_e, chLatin_s, chLatin_s, chNull };

const XMLCh XML::Literals::Issuer[]=
{ chLatin_I, chLatin_s, chLatin_s, chLatin_u, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::IssueInstant[]=
{ chLatin_I, chLatin_s, chLatin_s, chLatin_u, chLatin_e, chLatin_I, chLatin_n, chLatin_s, chLatin_t, chLatin_a, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::Location[]=
{ chLatin_L, chLatin_o, chLatin_c, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::MajorVersion[]=
{ chLatin_M, chLatin_a, chLatin_j, chLatin_o, chLatin_r, chLatin_V, chLatin_e, chLatin_r, chLatin_s, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::MinorVersion[]=
{ chLatin_M, chLatin_i, chLatin_n, chLatin_o, chLatin_r, chLatin_V, chLatin_e, chLatin_r, chLatin_s, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::MustUnderstand[]=
{ chLatin_M, chLatin_u, chLatin_s, chLatin_t, chLatin_U, chLatin_n, chLatin_d, chLatin_e, chLatin_r, chLatin_s, chLatin_t, chLatin_a, chLatin_n, chLatin_d, chNull };

const XMLCh XML::Literals::NameIdentifier[]=
{ chLatin_N, chLatin_a, chLatin_m, chLatin_e, chLatin_I, chLatin_d, chLatin_e, chLatin_n, chLatin_t, chLatin_i, chLatin_f, chLatin_i, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::NameQualifier[]=
{ chLatin_N, chLatin_a, chLatin_m, chLatin_e, chLatin_Q, chLatin_u, chLatin_a, chLatin_l, chLatin_i, chLatin_f, chLatin_i, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::Namespace[]= 
{ chLatin_N, chLatin_a, chLatin_m, chLatin_e, chLatin_s, chLatin_p, chLatin_a, chLatin_c, chLatin_e, chNull };

const XMLCh XML::Literals::NotBefore[]=
{ chLatin_N, chLatin_o, chLatin_t, chLatin_B, chLatin_e, chLatin_f, chLatin_o, chLatin_r, chLatin_e, chNull };

const XMLCh XML::Literals::NotOnOrAfter[]=
{ chLatin_N, chLatin_o, chLatin_t, chLatin_O, chLatin_n, chLatin_O, chLatin_r, chLatin_A, chLatin_f, chLatin_t, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::Query[]=
{ chLatin_Q, chLatin_u, chLatin_e, chLatin_r, chLatin_y, chNull };

const XMLCh XML::Literals::Recipient[]=
{ chLatin_R, chLatin_e, chLatin_c, chLatin_i, chLatin_p, chLatin_i, chLatin_e, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::Request[]=
{ chLatin_R, chLatin_e, chLatin_q, chLatin_u, chLatin_e, chLatin_s, chLatin_t, chNull };

const XMLCh XML::Literals::Requester[]=
{ chLatin_R, chLatin_e, chLatin_q, chLatin_u, chLatin_e, chLatin_s, chLatin_t, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::RequestID[]=
{ chLatin_R, chLatin_e, chLatin_q, chLatin_u, chLatin_e, chLatin_s, chLatin_t, chLatin_I, chLatin_D, chNull };

const XMLCh XML::Literals::Resource[]=
{ chLatin_R, chLatin_e, chLatin_s, chLatin_o, chLatin_u, chLatin_r, chLatin_c, chLatin_e, chNull };

const XMLCh XML::Literals::Responder[]=
{ chLatin_R, chLatin_e, chLatin_s, chLatin_p, chLatin_o, chLatin_n, chLatin_d, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::samlp_Responder[]=
{ chLatin_s, chLatin_a, chLatin_m, chLatin_l, chLatin_p, chColon,
  chLatin_R, chLatin_e, chLatin_s, chLatin_p, chLatin_o, chLatin_n, chLatin_d, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::RespondWith[]=
{ chLatin_R, chLatin_e, chLatin_s, chLatin_p, chLatin_o, chLatin_n, chLatin_d, chLatin_W, chLatin_i, chLatin_t, chLatin_h, chNull };

const XMLCh XML::Literals::Response[]=
{ chLatin_R, chLatin_e, chLatin_s, chLatin_p, chLatin_o, chLatin_n, chLatin_s, chLatin_e, chNull };

const XMLCh XML::Literals::ResponseID[]=
{ chLatin_R, chLatin_e, chLatin_s, chLatin_p, chLatin_o, chLatin_n, chLatin_s, chLatin_e, chLatin_I, chLatin_D, chNull };

const XMLCh XML::Literals::Server[]=
{ chLatin_S, chLatin_e, chLatin_r, chLatin_v, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::Statement[]=
{ chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_e, chLatin_m, chLatin_e, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::Status[]=
{ chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_u, chLatin_s, chNull };

const XMLCh XML::Literals::StatusMessage[]=
{ chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_u, chLatin_s, chLatin_M, chLatin_e, chLatin_s, chLatin_s, chLatin_a, chLatin_g, chLatin_e, chNull };

const XMLCh XML::Literals::StatusCode[]=
{ chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_u, chLatin_s, chLatin_C, chLatin_o, chLatin_d, chLatin_e, chNull };

const XMLCh XML::Literals::StatusDetail[]=
{ chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_u, chLatin_s, chLatin_D, chLatin_e, chLatin_t, chLatin_a, chLatin_i, chLatin_l, chNull };

const XMLCh XML::Literals::Subject[]=
{ chLatin_S, chLatin_u, chLatin_b, chLatin_j, chLatin_e, chLatin_c, chLatin_t, chNull };

const XMLCh XML::Literals::saml_Subject[]=
{ chLatin_s, chLatin_a, chLatin_m, chLatin_l, chColon,
  chLatin_S, chLatin_u, chLatin_b, chLatin_j, chLatin_e, chLatin_c, chLatin_t, chNull };

const XMLCh XML::Literals::SubjectConfirmation[]=
{ chLatin_S, chLatin_u, chLatin_b, chLatin_j, chLatin_e, chLatin_c, chLatin_t, chLatin_C, chLatin_o, chLatin_n, chLatin_f, chLatin_i, chLatin_r, chLatin_m, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chNull };

const XMLCh XML::Literals::SubjectConfirmationData[]=
{ chLatin_S, chLatin_u, chLatin_b, chLatin_j, chLatin_e, chLatin_c, chLatin_t, chLatin_C, chLatin_o, chLatin_n, chLatin_f, chLatin_i, chLatin_r, chLatin_m, chLatin_a, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_D, chLatin_a, chLatin_t, chLatin_a, chNull };

const XMLCh XML::Literals::SubjectLocality[]=
{ chLatin_S, chLatin_u, chLatin_b, chLatin_j, chLatin_e, chLatin_c, chLatin_t, chLatin_L, chLatin_o, chLatin_c, chLatin_a, chLatin_l, chLatin_i, chLatin_t, chLatin_y, chNull };

const XMLCh XML::Literals::SubjectQuery[]=
{ chLatin_S, chLatin_u, chLatin_b, chLatin_j, chLatin_e, chLatin_c, chLatin_t, chLatin_Q, chLatin_u, chLatin_e, chLatin_r, chLatin_y, chNull };

const XMLCh XML::Literals::SubjectStatement[]=
{ chLatin_S, chLatin_u, chLatin_b, chLatin_j, chLatin_e, chLatin_c, chLatin_t, chLatin_S, chLatin_t, chLatin_a, chLatin_t, chLatin_e, chLatin_m, chLatin_e, chLatin_n, chLatin_t, chNull };

const XMLCh XML::Literals::Success[]=
{ chLatin_S, chLatin_u, chLatin_c, chLatin_c, chLatin_e, chLatin_s, chLatin_s, chNull };

const XMLCh XML::Literals::Value[]=
{ chLatin_V, chLatin_a, chLatin_l, chLatin_u, chLatin_e, chNull };

const XMLCh XML::Literals::VersionMismatch[]=
{ chLatin_V, chLatin_e, chLatin_r, chLatin_s, chLatin_i, chLatin_o, chLatin_n, chLatin_M, chLatin_i, chLatin_s, chLatin_m, chLatin_a, chLatin_t, chLatin_c, chLatin_h, chNull };

const XMLCh XML::Literals::Body[]=
{ chLatin_B, chLatin_o, chLatin_d, chLatin_y, chNull };

const XMLCh XML::Literals::Envelope[]=
{ chLatin_E, chLatin_n, chLatin_v, chLatin_e, chLatin_l, chLatin_o, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::Fault[]=
{ chLatin_F, chLatin_a, chLatin_u, chLatin_l, chLatin_t, chNull };

const XMLCh XML::Literals::faultcode[]=
{ chLatin_f, chLatin_a, chLatin_u, chLatin_l, chLatin_t, chLatin_c, chLatin_o, chLatin_d, chLatin_e, chNull };

const XMLCh XML::Literals::faultstring[]=
{ chLatin_f, chLatin_a, chLatin_u, chLatin_l, chLatin_t, chLatin_s, chLatin_t, chLatin_r, chLatin_i, chLatin_n, chLatin_g, chNull };

const XMLCh XML::Literals::Header[]=
{ chLatin_H, chLatin_e, chLatin_a, chLatin_d, chLatin_e, chLatin_r, chNull };

const XMLCh XML::Literals::mustUnderstand[]=
{ chLatin_m, chLatin_u, chLatin_s, chLatin_t, chLatin_U, chLatin_n, chLatin_d, chLatin_e, chLatin_r, chLatin_s, chLatin_t, chLatin_a, chLatin_n, chLatin_d, chNull };

const XMLCh XML::Literals::Digest[]=
{ chLatin_D, chLatin_i, chLatin_g, chLatin_e, chLatin_s, chLatin_t, chNull };

const XMLCh XML::Literals::KeyInfo[]=
{ chLatin_K, chLatin_e, chLatin_y, chLatin_I, chLatin_n, chLatin_f, chLatin_o, chNull };

const XMLCh XML::Literals::Signature[]=
{ chLatin_S, chLatin_i, chLatin_g, chLatin_n, chLatin_a, chLatin_t, chLatin_u, chLatin_r, chLatin_e, chNull };

const XMLCh XML::Literals::X509Data[]=
{ chLatin_X, chDigit_5, chDigit_0, chDigit_9, chLatin_D, chLatin_a, chLatin_t, chLatin_a, chNull };

const XMLCh XML::Literals::X509Certificate[]=
{ chLatin_X, chDigit_5, chDigit_0, chDigit_9, chLatin_C, chLatin_e, chLatin_r, chLatin_t, chLatin_i, chLatin_f, chLatin_i, chLatin_c, chLatin_a, chLatin_t, chLatin_e, chNull };

const XMLCh XML::Literals::lang[]=
{ chLatin_l, chLatin_a, chLatin_n, chLatin_g, chNull };

const XMLCh XML::Literals::saml[]=
{ chLatin_s, chLatin_a, chLatin_m, chLatin_l, chNull };

const XMLCh XML::Literals::samlp[]=
{ chLatin_s, chLatin_a, chLatin_m, chLatin_l, chLatin_p, chNull };

const XMLCh XML::Literals::soap[]=
{ chLatin_s, chLatin_o, chLatin_a, chLatin_p, chNull };

const XMLCh XML::Literals::type[]=
{ chLatin_t, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::xml[]=
{ chLatin_x, chLatin_m, chLatin_l, chNull };

const XMLCh XML::Literals::xsd[]=
{ chLatin_x, chLatin_s, chLatin_d, chNull };

const XMLCh XML::Literals::xsi[]=
{ chLatin_x, chLatin_s, chLatin_i, chNull };

const XMLCh XML::Literals::xmlns[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chNull };

const XMLCh XML::Literals::xmlns_xsi[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_x, chLatin_s, chLatin_i, chNull };

const XMLCh XML::Literals::xmlns_xsd[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_x, chLatin_s, chLatin_d, chNull };

const XMLCh XML::Literals::xmlns_saml[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_s, chLatin_a, chLatin_m, chLatin_l, chNull };

const XMLCh XML::Literals::xmlns_samlp[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_s, chLatin_a, chLatin_m, chLatin_l, chLatin_p, chNull };

const XMLCh XML::Literals::xmlns_soap[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_s, chLatin_o, chLatin_a, chLatin_p, chNull };

const XMLCh XML::Literals::xmlns_code[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_c, chLatin_o, chLatin_d, chLatin_e, chNull };

const XMLCh XML::Literals::xmlns_kind[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_k, chLatin_i, chLatin_n, chLatin_d, chNull };

const XMLCh XML::Literals::xmlns_rw[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_r, chLatin_w, chNull };

const XMLCh XML::Literals::xmlns_typens[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_t, chLatin_y, chLatin_p, chLatin_e, chLatin_n, chLatin_s, chNull };

const XMLCh XML::Literals::xmlns_opensaml[]=
{ chLatin_x, chLatin_m, chLatin_l, chLatin_n, chLatin_s, chColon, chLatin_o, chLatin_p, chLatin_e, chLatin_n, chLatin_s, chLatin_a, chLatin_m, chLatin_l, chNull };

const XMLCh XML::Literals::xsi_type[]=
{ chLatin_x, chLatin_s, chLatin_i, chColon, chLatin_t, chLatin_y, chLatin_p, chLatin_e, chNull };

const XMLCh XML::Literals::typens[]=
{ chLatin_t, chLatin_y, chLatin_p, chLatin_e, chLatin_n, chLatin_s, chNull };

const XMLCh XML::Literals::Code[]= 
{ chLatin_C, chLatin_o, chLatin_d, chLatin_e, chNull };

const XMLCh XML::Literals::ExceptionClass[]=
{ chLatin_E, chLatin_x, chLatin_c, chLatin_e, chLatin_p, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_C, chLatin_l, chLatin_a, chLatin_s, chLatin_s, chNull };

const XMLCh XML::Literals::Name[]= 
{ chLatin_N, chLatin_a, chLatin_m, chLatin_e, chNull };

const XMLCh XML::Literals::Param[]= 
{ chLatin_P, chLatin_a, chLatin_r, chLatin_a, chLatin_m, chNull };

const XMLCh XML::Literals::opensaml_Code[]= 
{ chLatin_o, chLatin_p, chLatin_e, chLatin_n, chLatin_s, chLatin_a, chLatin_m, chLatin_l, chColon,
  chLatin_C, chLatin_o, chLatin_d, chLatin_e, chNull
};

const XMLCh XML::Literals::opensaml_ExceptionClass[]=
{ chLatin_o, chLatin_p, chLatin_e, chLatin_n, chLatin_s, chLatin_a, chLatin_m, chLatin_l, chColon,
  chLatin_E, chLatin_x, chLatin_c, chLatin_e, chLatin_p, chLatin_t, chLatin_i, chLatin_o, chLatin_n, chLatin_C, chLatin_l, chLatin_a, chLatin_s, chLatin_s, chNull
};

const XMLCh XML::Literals::opensaml_Param[]=
{ chLatin_o, chLatin_p, chLatin_e, chLatin_n, chLatin_s, chLatin_a, chLatin_m, chLatin_l, chColon,
  chLatin_P, chLatin_a, chLatin_r, chLatin_a, chLatin_m, chNull
};
